<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class PerpindahanAnggota extends Model
{
    use HasFactory;

    protected $table = 'perpindahan_anggota';

    protected $fillable = [
        'anggota_id',
        'tanggal_perpindahan',
        'dpd_id_before',
        'dpc_id_before',
        'upa_id_before',
        'ref_jenjang_id_before',
        'ref_grup_upa_id_before',
        'dpd_id_after',
        'dpc_id_after',
        'upa_id_after',
        'ref_jenjang_id_after',
        'ref_grup_upa_id_after',
        'keterangan',
        'status_perpindahan',
        'catatan_admin',
        'created_by',
        'approved_by'
    ];

    protected $casts = [
        'tanggal_perpindahan' => 'date',
    ];

    // ===== RELATIONSHIPS =====
    
    // Relasi ke Anggota
    public function anggota()
    {
        return $this->belongsTo(Anggota::class, 'anggota_id');
    }

    // Relasi ke User yang membuat perpindahan
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Relasi ke User yang menyetujui perpindahan
    public function approvedBy()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    // Data sebelum perpindahan
    public function dpdBefore()
    {
        return $this->belongsTo(Dpd::class, 'dpd_id_before', 'dpd_id');
    }

    public function dpcBefore()
    {
        return $this->belongsTo(Dpc::class, 'dpc_id_before', 'dpc_id');
    }

    public function upaBefore()
    {
        return $this->belongsTo(Upa::class, 'upa_id_before', 'upa_id');
    }

    public function jenjangBefore()
    {
        return $this->belongsTo(RefJenjang::class, 'ref_jenjang_id_before', 'ref_jenjang_id');
    }

    public function grupUpaBefore()
    {
        return $this->belongsTo(RefGrupUpa::class, 'ref_grup_upa_id_before', 'ref_grup_upa_id');
    }

    // Data setelah perpindahan
    public function dpdAfter()
    {
        return $this->belongsTo(Dpd::class, 'dpd_id_after', 'dpd_id');
    }

    public function dpcAfter()
    {
        return $this->belongsTo(Dpc::class, 'dpc_id_after', 'dpc_id');
    }

    public function upaAfter()
    {
        return $this->belongsTo(Upa::class, 'upa_id_after', 'upa_id');
    }

    public function jenjangAfter()
    {
        return $this->belongsTo(RefJenjang::class, 'ref_jenjang_id_after', 'ref_jenjang_id');
    }

    public function grupUpaAfter()
    {
        return $this->belongsTo(RefGrupUpa::class, 'ref_grup_upa_id_after', 'ref_grup_upa_id');
    }

    // ===== ACCESSORS =====
    
    public function getStatusBadgeClassAttribute()
    {
        return match($this->status_perpindahan) {
            'Diajukan' => 'warning',
            'Disetujui' => 'success',
            'Ditolak' => 'danger',
            default => 'secondary'
        };
    }

    public function getStatusLabelAttribute()
    {
        return match($this->status_perpindahan) {
            'Diajukan' => 'Diajukan',
            'Disetujui' => 'Disetujui',
            'Ditolak' => 'Ditolak',
            default => 'Tidak Diketahui'
        };
    }

    public function getTanggalPerpindahanFormattedAttribute()
    {
        return $this->tanggal_perpindahan ? $this->tanggal_perpindahan->format('d F Y') : '-';
    }

    // ===== SCOPES =====
    
    public function scopeByStatus($query, $status)
    {
        return $query->where('status_perpindahan', $status);
    }

    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('tanggal_perpindahan', [$startDate, $endDate]);
    }

    public function scopeByAnggota($query, $anggotaId)
    {
        return $query->where('anggota_id', $anggotaId);
    }

    public function scopeApproved($query)
    {
        return $query->where('status_perpindahan', 'Disetujui');
    }

    public function scopePending($query)
    {
        return $query->where('status_perpindahan', 'Diajukan');
    }

    // ===== HELPER METHODS =====
    
    /**
     * Cek apakah ada perubahan data
     */
    public function hasDataChanges()
    {
        return $this->dpd_id_before != $this->dpd_id_after ||
               $this->dpc_id_before != $this->dpc_id_after ||
               $this->upa_id_before != $this->upa_id_after ||
               $this->ref_jenjang_id_before != $this->ref_jenjang_id_after ||
               $this->ref_grup_upa_id_before != $this->ref_grup_upa_id_after;
    }

    /**
     * Dapatkan ringkasan perubahan
     */
    public function getChangesSummary()
    {
        $changes = [];
        
        if ($this->dpd_id_before != $this->dpd_id_after) {
            $changes[] = 'DPD';
        }
        if ($this->dpc_id_before != $this->dpc_id_after) {
            $changes[] = 'DPC';
        }
        if ($this->upa_id_before != $this->upa_id_after) {
            $changes[] = 'UPA';
        }
        if ($this->ref_jenjang_id_before != $this->ref_jenjang_id_after) {
            $changes[] = 'Jenjang';
        }
        if ($this->ref_grup_upa_id_before != $this->ref_grup_upa_id_after) {
            $changes[] = 'Grup UPA';
        }
        
        return implode(', ', $changes);
    }

    /**
     * Apply perpindahan ke data anggota
     */
    public function applyToAnggota()
    {
        if ($this->status_perpindahan !== 'Disetujui') {
            return false;
        }

        $anggota = $this->anggota;
        if (!$anggota) {
            return false;
        }

        // Update data anggota
        $anggota->update([
            'dpd_id' => $this->dpd_id_after,
            'dpc_id' => $this->dpc_id_after,
            'upa_id' => $this->upa_id_after,
            'ref_jenjang_id' => $this->ref_jenjang_id_after,
            'ref_grup_upa_id' => $this->ref_grup_upa_id_after,
        ]);

        return true;
    }

    /**
     * Revert perpindahan ke data anggota
     */
    public function revertFromAnggota()
    {
        $anggota = $this->anggota;
        if (!$anggota) {
            return false;
        }

        // Revert data anggota ke kondisi sebelum perpindahan
        $anggota->update([
            'dpd_id' => $this->dpd_id_before,
            'dpc_id' => $this->dpc_id_before,
            'upa_id' => $this->upa_id_before,
            'ref_jenjang_id' => $this->ref_jenjang_id_before,
            'ref_grup_upa_id' => $this->ref_grup_upa_id_before,
        ]);

        return true;
    }

    /**
     * Approve perpindahan
     */
    public function approve($approvedBy, $catatanAdmin = null)
    {
        $this->update([
            'status_perpindahan' => 'Disetujui',
            'approved_by' => $approvedBy,
            'catatan_admin' => $catatanAdmin,
        ]);

        // Apply perubahan ke data anggota
        $this->applyToAnggota();

        return true;
    }

    /**
     * Reject perpindahan
     */
    public function reject($approvedBy, $catatanAdmin = null)
    {
        $this->update([
            'status_perpindahan' => 'Ditolak',
            'approved_by' => $approvedBy,
            'catatan_admin' => $catatanAdmin,
        ]);

        return true;
    }
}
