<?php


namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Mutasi extends Model
{
    protected $table = 'tbl_mutasi';

    protected $fillable = [
        'nomor_sk',
        'anggota_id',
        'jenis_mutasi',
        'tanggal_mutasi',
        'status_mutasi',
        'struktur_dari',
        'struktur_ke',
        'dpd_dari',
        'dpd_ke',
        'alamat_tujuan',
        'alasan_mutasi',
        'catatan',
        'disetujui_oleh',
        'tanggal_sk'
    ];

    protected $casts = [
        'tanggal_mutasi' => 'date',
        'tanggal_sk' => 'datetime',
    ];

    // Relationships
    public function anggota()
    {
        return $this->belongsTo(Anggota::class, 'anggota_id');
    }

    public function strukturDari()
    {
        return $this->belongsTo(StrukturOrganisasi::class, 'struktur_dari');
    }

    public function strukturKe()
    {
        return $this->belongsTo(StrukturOrganisasi::class, 'struktur_ke');
    }

    public function dpcTujuan()
    {
        return $this->belongsTo(Dpc::class, 'struktur_ke', 'dpc_id');
    }

    public function disetujuiOleh()
    {
        return $this->belongsTo(User::class, 'disetujui_oleh');
    }

    // Accessors
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'Pending' => 'warning',
            'Disetujui' => 'success',
            'Ditolak' => 'danger'
        ];
        
        return $badges[$this->status_mutasi] ?? 'secondary';
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status_mutasi', 'Pending');
    }

    public function scopeDisetujui($query)
    {
        return $query->where('status_mutasi', 'Disetujui');
    }

    public function scopeKeluar($query)
    {
        return $query->where('jenis_mutasi', 'Keluar');
    }

    public function scopeDibatalkan($query)
    {
        return $query->where('status_mutasi', 'Dibatalkan');
    }

    // Methods
    public function generateNomorSK()
    {
        if ($this->nomor_sk) {
            return $this->nomor_sk;
        }

        $tahun = $this->tanggal_mutasi->format('Y');
        $bulan = $this->getBulanRomawi($this->tanggal_mutasi->format('n'));
        
        $counter = static::whereYear('tanggal_mutasi', $tahun)
                        ->whereNotNull('nomor_sk')
                        ->count() + 1;

        $this->nomor_sk = sprintf(
            '%03d/D/Mts/A1-04-PKS/%s/%s',
            $counter,
            $bulan,
            $tahun
        );

        $this->save();
        
        return $this->nomor_sk;
    }

    private function getBulanRomawi($bulan)
    {
        $romawi = [
            1 => 'I', 2 => 'II', 3 => 'III', 4 => 'IV', 5 => 'V', 6 => 'VI',
            7 => 'VII', 8 => 'VIII', 9 => 'IX', 10 => 'X', 11 => 'XI', 12 => 'XII'
        ];
        return $romawi[(int)$bulan];
    }

    public function canBeEdited()
    {
        return $this->status_mutasi === 'Pending';
    }

    public function canBeApproved()
    {
        return $this->status_mutasi === 'Pending';
    }
}