<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;


namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Kegiatan extends Model
{
    protected $table = 'tbl_kegiatan';

    protected $fillable = [
        'nama_kegiatan',
        'deskripsi',
        'tanggal_kegiatan',
        'tanggal_mulai',
        'tanggal_selesai',
        'waktu_mulai',
        'waktu_selesai',
        'tempat',
        'ref_jenis_kegiatan_id',
        'jenis_kegiatan_lainnya',
        'status_kegiatan',
        'target_peserta',
        'catatan',
        'pic_id',
        'pic_anggota_id'
    ];

    protected $casts = [
        'tanggal_kegiatan' => 'date',
        'tanggal_mulai' => 'date',
        'tanggal_selesai' => 'date',
        'waktu_mulai' => 'datetime:H:i',
        'waktu_selesai' => 'datetime:H:i',
    ];

    // Relationships
    public function pic()
    {
        return $this->belongsTo(User::class, 'pic_id');
    }

    public function jenisKegiatan()
    {
        return $this->belongsTo(JenisKegiatan::class, 'ref_jenis_kegiatan_id');
    }

    public function picAnggota()
    {
        return $this->belongsTo(Anggota::class, 'pic_anggota_id');
    }
     public function absensiKegiatan()
    {
        return $this->hasMany(AbsensiKegiatan::class, 'kegiatan_id');
    }

    public function partisipan()
    {
        return $this->hasMany(KegiatanPartisipan::class, 'kegiatan_id');
    }

    public function anggotaPartisipan()
    {
        return $this->belongsToMany(Anggota::class, 'tbl_kegiatan_partisipan', 'kegiatan_id', 'anggota_id')
                    ->withPivot('status_partisipasi', 'status_kehadiran', 'keterangan', 'waktu_konfirmasi')
                    ->withTimestamps();
    }

    // Accessors
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'Direncanakan' => 'primary',
            'Berlangsung' => 'warning',
            'Selesai' => 'success',
            'Dibatalkan' => 'danger'
        ];
        
        return $badges[$this->status_kegiatan] ?? 'secondary';
    }

    public function getJenisBadgeAttribute()
    {
        $badges = [
            'Mabit' => 'success',
            'Rapat' => 'primary',
            'Pelatihan' => 'info',
            'Sosialisasi' => 'warning',
            'Lainnya' => 'secondary'
        ];
        
        $jenisNama = $this->getNamaJenisKegiatanAttribute();
        return $badges[$jenisNama] ?? 'secondary';
    }

    public function getNamaJenisKegiatanAttribute()
    {
        if ($this->jenis_kegiatan_lainnya) {
            return $this->jenis_kegiatan_lainnya;
        }
        
        return $this->jenisKegiatan->nama_jenis_kegiatan ?? 'Lainnya';
    }

    public function getWaktuLengkapAttribute()
    {
        $waktu = $this->waktu_mulai ? Carbon::parse($this->waktu_mulai)->format('H:i') : '';
        if ($this->waktu_selesai) {
            $waktu .= ' - ' . Carbon::parse($this->waktu_selesai)->format('H:i');
        }
        return $waktu;
    }

    // Scopes
    public function scopeUpcoming($query)
    {
        return $query->where('tanggal_kegiatan', '>=', Carbon::today())
                     ->where('status_kegiatan', '!=', 'Selesai');
    }

    public function scopeByJenis($query, $jenis)
    {
        return $query->where('jenis_kegiatan', $jenis);
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('status_kegiatan', $status);
    }

    // Methods
    public function getTotalPartisipan()
    {
        return $this->partisipan()->count();
    }

    public function getTotalHadir()
    {
        return $this->partisipan()->where('status_kehadiran', 'Hadir')->count();
    }

    public function getPersentaseKehadiran()
    {
        $total = $this->getTotalPartisipan();
        if ($total == 0) return 0;
        
        $hadir = $this->getTotalHadir();
        return round(($hadir / $total) * 100, 2);
    }

    public function canBeEdited()
    {
        return in_array($this->status_kegiatan, ['Direncanakan', 'Berlangsung']);
    }

    public function isUpcoming()
    {
        return $this->tanggal_kegiatan >= Carbon::today() && $this->status_kegiatan != 'Selesai';
    }
}