<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class AbsensiKegiatan extends Model
{
    use HasFactory;

    protected $table = 'tbl_absensi_kegiatan';
    
    protected $fillable = [
        'kegiatan_id',
        'anggota_id',
        'waktu_checkin',
        'status',
        'keterangan'
    ];

    protected $casts = [
        'waktu_checkin' => 'datetime',
    ];

    /**
     * Get the kegiatan that owns the absensi
     */
    public function kegiatan()
    {
        return $this->belongsTo(Kegiatan::class, 'kegiatan_id');
    }

    /**
     * Get the anggota that owns the absensi
     */
    public function anggota()
    {
        return $this->belongsTo(Anggota::class, 'anggota_id');
    }

    /**
     * Scope for hadir status
     */
    public function scopeHadir($query)
    {
        return $query->where('status', 'hadir');
    }

    /**
     * Scope for tidak hadir status
     */
    public function scopeTidakHadir($query)
    {
        return $query->where('status', 'tidak_hadir');
    }

    /**
     * Scope for izin status
     */
    public function scopeIzin($query)
    {
        return $query->where('status', 'izin');
    }

    /**
     * Get formatted waktu checkin
     */
    public function getWaktuCheckinFormattedAttribute()
    {
        return $this->waktu_checkin ? $this->waktu_checkin->format('d/m/Y H:i:s') : '-';
    }

    /**
     * Get status badge class
     */
    public function getStatusBadgeClassAttribute()
    {
        switch ($this->status) {
            case 'hadir':
                return 'success';
            case 'tidak_hadir':
                return 'danger';
            case 'izin':
                return 'warning';
            default:
                return 'secondary';
        }
    }

    /**
     * Get status label
     */
    public function getStatusLabelAttribute()
    {
        switch ($this->status) {
            case 'hadir':
                return 'Hadir';
            case 'tidak_hadir':
                return 'Tidak Hadir';
            case 'izin':
                return 'Izin';
            default:
                return 'Belum Konfirmasi';
        }
    }

    /**
     * Check if checkin is late
     */
    public function isLate()
    {
        if (!$this->kegiatan || !$this->waktu_checkin) {
            return false;
        }

        $kegiatanStart = Carbon::parse($this->kegiatan->tanggal_mulai . ' ' . $this->kegiatan->waktu_mulai);
        return $this->waktu_checkin->gt($kegiatanStart->addMinutes(15)); // 15 minutes tolerance
    }

    /**
     * Get checkin status with late indicator
     */
    public function getCheckinStatusAttribute()
    {
        $status = $this->status_label;
        
        if ($this->status === 'hadir' && $this->isLate()) {
            $status .= ' (Terlambat)';
        }
        
        return $status;
    }
}