<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class UserManagementController extends Controller
{
    public function __construct()
    {
        // Middleware akan diterapkan di routes
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $users = User::orderBy('created_at', 'desc')->paginate(10);
        return view('user-management.index', compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $roles = [
            'admin' => 'Admin',
            'user' => 'User'
        ];

        // Superadmin bisa membuat superadmin juga
        if (auth()->user()->isSuperAdmin()) {
            $roles['superadmin'] = 'Super Admin';
        }

        return view('user-management.create', compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|regex:/^[a-zA-Z\s]+$/',
            'email' => 'required|string|email|max:255|unique:users|regex:/^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/',
            'password' => 'required|string|min:8|max:255|confirmed|regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]/',
            'role' => ['required', Rule::in(['superadmin', 'admin', 'user'])],
            'is_active' => 'boolean'
        ], [
            'name.required' => 'Nama wajib diisi.',
            'name.regex' => 'Nama hanya boleh mengandung huruf dan spasi.',
            'email.required' => 'Email wajib diisi.',
            'email.email' => 'Format email tidak valid.',
            'email.unique' => 'Email sudah digunakan.',
            'email.regex' => 'Format email tidak valid.',
            'password.required' => 'Password wajib diisi.',
            'password.min' => 'Password minimal 8 karakter.',
            'password.regex' => 'Password harus mengandung huruf besar, huruf kecil, angka, dan simbol khusus.',
            'role.required' => 'Role wajib dipilih.',
            'role.in' => 'Role tidak valid.',
        ]);

        // Cek permission untuk membuat superadmin
        if ($validated['role'] === 'superadmin' && !auth()->user()->isSuperAdmin()) {
            return back()->withErrors(['role' => 'Anda tidak memiliki izin untuk membuat Super Admin.']);
        }

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => $validated['role'],
            'is_active' => $request->has('is_active') ? true : false,
        ]);

        return redirect()->route('user-management.index')
            ->with('success', 'User berhasil dibuat!');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        return view('user-management.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $roles = [
            'admin' => 'Admin',
            'user' => 'User'
        ];

        // Superadmin bisa mengubah role ke superadmin juga
        if (auth()->user()->isSuperAdmin()) {
            $roles['superadmin'] = 'Super Admin';
        }

        return view('user-management.edit', compact('user', 'roles'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'password' => 'nullable|string|min:6|confirmed',
            'role' => ['required', Rule::in(['superadmin', 'admin', 'user'])],
            'is_active' => 'boolean'
        ]);

        // Cek permission untuk mengubah ke superadmin
        if ($validated['role'] === 'superadmin' && !auth()->user()->isSuperAdmin()) {
            return back()->withErrors(['role' => 'Anda tidak memiliki izin untuk mengubah role ke Super Admin.']);
        }

        // Cek permission untuk mengubah superadmin lain
        if ($user->isSuperAdmin() && !auth()->user()->isSuperAdmin()) {
            return back()->withErrors(['role' => 'Anda tidak memiliki izin untuk mengubah Super Admin.']);
        }

        $user->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'role' => $validated['role'],
            'is_active' => $request->has('is_active') ? true : false,
        ]);

        // Update password jika diisi
        if (!empty($validated['password'])) {
            $user->update([
                'password' => Hash::make($validated['password'])
            ]);
        }

        return redirect()->route('user-management.index')
            ->with('success', 'User berhasil diperbarui!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        // Cek permission untuk menghapus superadmin
        if ($user->isSuperAdmin() && !auth()->user()->isSuperAdmin()) {
            return back()->withErrors(['error' => 'Anda tidak memiliki izin untuk menghapus Super Admin.']);
        }

        // Tidak bisa menghapus diri sendiri
        if ($user->id === auth()->id()) {
            return back()->withErrors(['error' => 'Anda tidak dapat menghapus akun sendiri.']);
        }

        $user->delete();

        return redirect()->route('user-management.index')
            ->with('success', 'User berhasil dihapus!');
    }

    /**
     * Toggle user active status
     */
    public function toggleStatus(User $user_management)
    {
        // Cek permission untuk mengubah status superadmin
        if ($user_management->isSuperAdmin() && !auth()->user()->isSuperAdmin()) {
            return back()->withErrors(['error' => 'Anda tidak memiliki izin untuk mengubah status Super Admin.']);
        }

        // Tidak bisa menonaktifkan diri sendiri
        if ($user_management->id === auth()->id()) {
            return back()->withErrors(['error' => 'Anda tidak dapat menonaktifkan akun sendiri.']);
        }

        $user_management->update([
            'is_active' => !$user_management->is_active
        ]);

        $status = $user_management->is_active ? 'diaktifkan' : 'dinonaktifkan';
        
        return back()->with('success', "User berhasil {$status}!");
    }
}
