<?php

namespace App\Http\Controllers;

use App\Models\UpaAgendaMaster;
use Illuminate\Http\Request;

class UpaAgendaMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = UpaAgendaMaster::query();
        
        // Filter berdasarkan pencarian
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_agenda', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%");
            });
        }
        
        // Filter berdasarkan status aktif
        if ($request->has('status') && $request->status) {
            if ($request->status == 'aktif') {
                $query->where('is_active', true);
            } elseif ($request->status == 'tidak_aktif') {
                $query->where('is_active', false);
            }
        }
        
        $agendas = $query->orderBy('nama_agenda')->paginate(20);
        
        return view('upa-agenda-master.index', compact('agendas'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('upa-agenda-master.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'nama_agenda' => 'required|string|max:255|unique:upa_agenda_masters',
            'deskripsi' => 'nullable|string',
            'is_active' => 'boolean'
        ]);

        UpaAgendaMaster::create($validated);

        return redirect()->route('upa-agenda-master.index')
                        ->with('success', 'Agenda master berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(UpaAgendaMaster $upaAgendaMaster)
    {
        return view('upa-agenda-master.show', compact('upaAgendaMaster'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(UpaAgendaMaster $upaAgendaMaster)
    {
        return view('upa-agenda-master.edit', compact('upaAgendaMaster'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, UpaAgendaMaster $upaAgendaMaster)
    {
        $validated = $request->validate([
            'nama_agenda' => 'required|string|max:255|unique:upa_agenda_masters,nama_agenda,' . $upaAgendaMaster->agenda_id . ',agenda_id',
            'deskripsi' => 'nullable|string',
            'is_active' => 'boolean'
        ]);

        $upaAgendaMaster->update($validated);

        return redirect()->route('upa-agenda-master.index')
                        ->with('success', 'Agenda master berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(UpaAgendaMaster $upaAgendaMaster)
    {
        $upaAgendaMaster->delete();

        return redirect()->route('upa-agenda-master.index')
                        ->with('success', 'Agenda master berhasil dihapus.');
    }

    /**
     * Toggle status aktif/tidak aktif
     */
    public function toggleStatus(UpaAgendaMaster $upaAgendaMaster)
    {
        $upaAgendaMaster->update(['is_active' => !$upaAgendaMaster->is_active]);
        
        $status = $upaAgendaMaster->is_active ? 'diaktifkan' : 'dinonaktifkan';
        
        return redirect()->route('upa-agenda-master.index')
                        ->with('success', "Agenda master berhasil {$status}.");
    }
}
