<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Anggota;
use App\Models\Kegiatan;
use App\Models\Mutasi;
use App\Models\Takwim;
use App\Models\StrukturOrganisasi;
use App\Models\AbsensiKegiatan;
use App\Models\PesertaTakwim;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\StatistikAnggotaExport;
use App\Exports\StatistikAnggotaCsvExport;

class StatistikController extends Controller
{
    public function index()
    {
        // Overview statistics dengan caching
        $stats = cache()->remember('dashboard_stats', 300, function() {
            return [
                'total_anggota' => Anggota::count(),
                'anggota_aktif' => Anggota::where('status_keanggotaan', 'Aktif')->count(),
                'total_kegiatan' => Kegiatan::count(),
                'kegiatan_bulan_ini' => Kegiatan::whereMonth('tanggal_kegiatan', now()->month)
                    ->whereYear('tanggal_kegiatan', now()->year)->count(),
                'total_mutasi' => Mutasi::count(),
                'mutasi_pending' => Mutasi::where('status_mutasi', 'Pending')->count(),
                'total_takwim' => Takwim::count(),
                'takwim_aktif' => Takwim::where('status_takwim', 'Aktif')->count(),
            ];
        });

        // Chart data dengan caching
        $chartData = cache()->remember('dashboard_chart_data', 600, function() {
            return [
                'anggota_per_bulan' => $this->getAnggotaPerBulan(),
                'kegiatan_per_bulan' => $this->getKegiatanPerBulan(),
                'mutasi_per_bulan' => $this->getMutasiPerBulan(),
                'anggota_per_struktur' => $this->getAnggotaPerStruktur(),
                'kegiatan_per_jenis' => $this->getKegiatanPerJenis(),
            ];
        });

        return view('statistik.index', compact('stats', 'chartData'));
    }

    public function anggota(Request $request)
    {
        $query = Anggota::query();

        // Filter berdasarkan tanggal
        if ($request->filled('tanggal_mulai')) {
            $query->where('created_at', '>=', $request->tanggal_mulai);
        }
        if ($request->filled('tanggal_selesai')) {
            $query->where('created_at', '<=', $request->tanggal_selesai . ' 23:59:59');
        }

        // Filter berdasarkan jenjang anggota
        if ($request->filled('ref_jenjang_id')) {
            $query->where('ref_jenjang_id', $request->ref_jenjang_id);
        }

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status_keanggotaan', $request->status);
        }

        // Filter berdasarkan jenis kelamin
        if ($request->filled('jenis_kelamin')) {
            $query->where('jenis_kelamin', $request->jenis_kelamin);
        }

        // Filter berdasarkan rentang umur
        if ($request->filled('umur_min')) {
            $query->whereRaw('TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) >= ?', [$request->umur_min]);
        }
        if ($request->filled('umur_max')) {
            $query->whereRaw('TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) <= ?', [$request->umur_max]);
        }

        $anggota = $query->with('refJenjang')->paginate(20);
        $strukturOrganisasi = cache()->remember('struktur_organisasi_list', 3600, function() {
            return StrukturOrganisasi::all();
        });

        // Statistik dengan single query untuk menghindari N+1
        $statistik = cache()->remember("anggota_statistik_" . md5(serialize($request->all())), 300, function() use ($query) {
            $baseQuery = clone $query;
            $baseQuery->getQuery()->orders = null; // Remove ordering for count queries
            
            return [
                'total' => $baseQuery->count(),
                'laki_laki' => $baseQuery->where('jenis_kelamin', 'L')->count(),
                'perempuan' => $baseQuery->where('jenis_kelamin', 'P')->count(),
                'aktif' => $baseQuery->where('status_keanggotaan', 'Aktif')->count(),
                'tidak_aktif' => $baseQuery->where('status_keanggotaan', 'Tidak Aktif')->count(),
                'rata_rata_umur' => $baseQuery->selectRaw('AVG(TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE())) as rata_umur')->first()->rata_umur ?? 0,
            ];
        });

        // Chart data
        $chartData = [
            'per_struktur' => $this->getAnggotaPerStruktur($request),
            'per_bulan' => $this->getAnggotaPerBulan($request),
            'per_jenis_kelamin' => $this->getAnggotaPerJenisKelamin($request),
            'per_umur' => $this->getAnggotaPerUmur($request),
        ];

        return view('statistik.anggota', compact('anggota', 'strukturOrganisasi', 'statistik', 'chartData'));
    }

    public function exportAnggota(Request $request)
    {
        try {
            // Set memory limit untuk export
            ini_set('memory_limit', '512M');
            set_time_limit(300); // 5 menit timeout

            // Gunakan query yang sama dengan method anggota() untuk konsistensi
            $query = Anggota::query();

            // Filter berdasarkan tanggal
            if ($request->filled('tanggal_mulai')) {
                $query->where('created_at', '>=', $request->tanggal_mulai);
            }
            if ($request->filled('tanggal_selesai')) {
                $query->where('created_at', '<=', $request->tanggal_selesai . ' 23:59:59');
            }

            // Filter berdasarkan jenjang anggota
            if ($request->filled('ref_jenjang_id')) {
                $query->where('ref_jenjang_id', $request->ref_jenjang_id);
            }

            // Filter berdasarkan status
            if ($request->filled('status')) {
                $query->where('status_keanggotaan', $request->status);
            }

            // Filter berdasarkan jenis kelamin
            if ($request->filled('jenis_kelamin')) {
                $query->where('jenis_kelamin', $request->jenis_kelamin);
            }

            // Filter berdasarkan rentang umur
            if ($request->filled('umur_min')) {
                $query->whereRaw('TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) >= ?', [$request->umur_min]);
            }
            if ($request->filled('umur_max')) {
                $query->whereRaw('TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) <= ?', [$request->umur_max]);
            }

            // Cek jumlah data terlebih dahulu
            $totalCount = $query->count();
            
            // Jika data terlalu banyak, beri peringatan
            if ($totalCount > 5000) {
                return redirect()->back()->with('warning', 'Data terlalu banyak (' . number_format($totalCount) . ' records). Silakan gunakan filter yang lebih spesifik untuk mengurangi jumlah data.');
            }
            
            // Jika data masih banyak, batasi untuk export
            if ($totalCount > 1000) {
                $query->limit(1000);
            }

            // Generate nama file dengan timestamp dan filter info
            $filename = 'statistik_anggota_' . date('Y-m-d_H-i-s');
            if ($request->filled('tanggal_mulai') || $request->filled('tanggal_selesai')) {
                $filename .= '_filtered';
            }

            // Export CSV untuk menghemat memory (lebih ringan dari Excel)
            return Excel::download(new StatistikAnggotaCsvExport($query), $filename . '.csv');
            
        } catch (\Exception $e) {
            \Log::error('Export statistik anggota error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Terjadi kesalahan saat export: ' . $e->getMessage());
        }
    }

    public function kegiatan(Request $request)
    {
        $query = Kegiatan::query();

        // Filter berdasarkan tanggal
        if ($request->filled('tanggal_mulai')) {
            $query->where('tanggal_kegiatan', '>=', $request->tanggal_mulai);
        }
        if ($request->filled('tanggal_selesai')) {
            $query->where('tanggal_kegiatan', '<=', $request->tanggal_selesai);
        }

        // Filter berdasarkan jenis kegiatan
        if ($request->filled('jenis_kegiatan')) {
            $query->where('jenis_kegiatan', $request->jenis_kegiatan);
        }

        // Filter berdasarkan status
        if ($request->filled('status_kegiatan')) {
            $query->where('status_kegiatan', $request->status_kegiatan);
        }

        // Filter berdasarkan struktur organisasi (melalui partisipan)
        if ($request->filled('struktur_organisasi_id')) {
            $query->whereHas('anggotaPartisipan.mutasi', function($q) use ($request) {
                $q->where('struktur_ke', $request->struktur_organisasi_id)
                  ->where('status_mutasi', 'Disetujui')
                  ->whereRaw('tanggal_mutasi = (
                      SELECT MAX(tanggal_mutasi) 
                      FROM tbl_mutasi m2 
                      WHERE m2.anggota_id = tbl_mutasi.anggota_id 
                      AND m2.status_mutasi = "Disetujui"
                  )');
            });
        }

        $kegiatan = $query->with(['absensiKegiatan'])->paginate(20);
        $strukturOrganisasi = StrukturOrganisasi::all();

        // Statistik
        $statistik = [
            'total' => $query->count(),
            'selesai' => $query->where('status_kegiatan', 'Selesai')->count(),
            'dibatalkan' => $query->where('status_kegiatan', 'Dibatalkan')->count(),
            'total_peserta' => $query->withCount('absensiKegiatan')->get()->sum('absensi_kegiatan_count'),
            'rata_rata_peserta' => $query->withCount('absensiKegiatan')->get()->avg('absensi_kegiatan_count') ?? 0,
        ];

        // Chart data
        $chartData = [
            'per_jenis' => $this->getKegiatanPerJenis($request),
            'per_bulan' => $this->getKegiatanPerBulan($request),
            'per_struktur' => $this->getKegiatanPerStruktur($request),
            'per_status' => $this->getKegiatanPerStatus($request),
        ];

        return view('statistik.kegiatan', compact('kegiatan', 'strukturOrganisasi', 'statistik', 'chartData'));
    }

    public function mutasi(Request $request)
    {
        $query = Mutasi::query();

        // Filter berdasarkan tanggal
        if ($request->filled('tanggal_mulai')) {
            $query->where('tanggal_mutasi', '>=', $request->tanggal_mulai);
        }
        if ($request->filled('tanggal_selesai')) {
            $query->where('tanggal_mutasi', '<=', $request->tanggal_selesai);
        }

        // Filter berdasarkan jenis mutasi
        if ($request->filled('jenis_mutasi')) {
            $query->where('jenis_mutasi', $request->jenis_mutasi);
        }

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status_mutasi', $request->status);
        }

        // Filter berdasarkan struktur organisasi (struktur tujuan)
        if ($request->filled('struktur_organisasi_id')) {
            $query->where('struktur_ke', $request->struktur_organisasi_id);
        }

        $mutasi = $query->with(['anggota.dpc', 'strukturDari', 'dpcTujuan'])->paginate(20);
        $strukturOrganisasi = StrukturOrganisasi::all();

        // Statistik
        $statistik = [
            'total' => $query->count(),
            'pending' => $query->where('status_mutasi', 'Pending')->count(),
            'disetujui' => $query->where('status_mutasi', 'Disetujui')->count(),
            'ditolak' => $query->where('status_mutasi', 'Ditolak')->count(),
            'keluar' => $query->where('jenis_mutasi', 'Keluar')->count(),
            'masuk' => $query->where('jenis_mutasi', 'Masuk')->count(),
        ];

        // Chart data
        $chartData = [
            'per_jenis' => $this->getMutasiPerJenis($request),
            'per_bulan' => $this->getMutasiPerBulan($request),
            'per_status' => $this->getMutasiPerStatus($request),
            'per_struktur' => $this->getMutasiPerStruktur($request),
        ];

        return view('statistik.mutasi', compact('mutasi', 'strukturOrganisasi', 'statistik', 'chartData'));
    }

    public function takwim(Request $request)
    {
        $query = Takwim::query();

        // Filter berdasarkan tanggal
        if ($request->filled('tanggal_mulai')) {
            $query->where('tanggal_mulai', '>=', $request->tanggal_mulai);
        }
        if ($request->filled('tanggal_selesai')) {
            $query->where('tanggal_selesai', '<=', $request->tanggal_selesai);
        }

        // Filter berdasarkan status
        if ($request->filled('status_takwim')) {
            $query->where('status_takwim', $request->status_takwim);
        }

        // Filter berdasarkan struktur organisasi
        if ($request->filled('struktur_penyelenggara_id')) {
            $query->where('struktur_penyelenggara_id', $request->struktur_penyelenggara_id);
        }

        $takwim = $query->with(['strukturPenyelenggara', 'pesertaTakwim.anggota'])->paginate(20);
        $strukturOrganisasi = StrukturOrganisasi::all();

        // Statistik
        $statistik = [
            'total' => $query->count(),
            'aktif' => $query->where('status_takwim', 'Aktif')->count(),
            'selesai' => $query->where('status_takwim', 'Selesai')->count(),
            'dibatalkan' => $query->where('status_takwim', 'Dibatalkan')->count(),
            'total_peserta' => $query->withCount('pesertaTakwim')->get()->sum('peserta_takwim_count'),
            'rata_rata_peserta' => $query->withCount('pesertaTakwim')->get()->avg('peserta_takwim_count') ?? 0,
        ];

        // Chart data
        $chartData = [
            'per_status' => $this->getTakwimPerStatus($request),
            'per_bulan' => $this->getTakwimPerBulan($request),
            'per_struktur' => $this->getTakwimPerStruktur($request),
            'per_durasi' => $this->getTakwimPerDurasi($request),
        ];

        return view('statistik.takwim', compact('takwim', 'strukturOrganisasi', 'statistik', 'chartData'));
    }

    // Helper methods for chart data
    private function getAnggotaPerBulan($request = null)
    {
        $query = Anggota::selectRaw('YEAR(created_at) as tahun, MONTH(created_at) as bulan, COUNT(*) as total')
            ->groupBy('tahun', 'bulan')
            ->orderBy('tahun', 'asc')
            ->orderBy('bulan', 'asc');

        if ($request && $request->filled('tanggal_mulai')) {
            $query->where('created_at', '>=', $request->tanggal_mulai);
        }
        if ($request && $request->filled('tanggal_selesai')) {
            $query->where('created_at', '<=', $request->tanggal_selesai . ' 23:59:59');
        }

        return $query->get();
    }

    private function getKegiatanPerBulan($request = null)
    {
        $query = Kegiatan::selectRaw('YEAR(tanggal_kegiatan) as tahun, MONTH(tanggal_kegiatan) as bulan, COUNT(*) as total')
            ->groupBy('tahun', 'bulan')
            ->orderBy('tahun', 'asc')
            ->orderBy('bulan', 'asc');

        if ($request && $request->filled('tanggal_mulai')) {
            $query->where('tanggal_kegiatan', '>=', $request->tanggal_mulai);
        }
        if ($request && $request->filled('tanggal_selesai')) {
            $query->where('tanggal_kegiatan', '<=', $request->tanggal_selesai);
        }

        return $query->get();
    }

    private function getMutasiPerBulan($request = null)
    {
        $query = Mutasi::selectRaw('YEAR(tanggal_mutasi) as tahun, MONTH(tanggal_mutasi) as bulan, COUNT(*) as total')
            ->groupBy('tahun', 'bulan')
            ->orderBy('tahun', 'asc')
            ->orderBy('bulan', 'asc');

        if ($request && $request->filled('tanggal_mulai')) {
            $query->where('tanggal_mutasi', '>=', $request->tanggal_mulai);
        }
        if ($request && $request->filled('tanggal_selesai')) {
            $query->where('tanggal_mutasi', '<=', $request->tanggal_selesai);
        }

        return $query->get();
    }

    private function getAnggotaPerStruktur($request = null)
    {
        // Anggota tidak memiliki relasi dengan struktur organisasi
        // Return empty collection atau data berdasarkan jenjang
        return Anggota::join('tbl_ref_jenjang', 'tbl_anggota.ref_jenjang_id', '=', 'tbl_ref_jenjang.ref_jenjang_id')
            ->selectRaw('tbl_ref_jenjang.jenjang as nama_struktur, COUNT(*) as total')
            ->groupBy('tbl_ref_jenjang.jenjang')
            ->orderByRaw("FIELD(tbl_ref_jenjang.jenjang, 'Pendukung', 'Penggerak', 'Madya', 'Dewasa', 'Utama')")
            ->get();
    }

    private function getKegiatanPerJenis($request = null)
    {
        $query = Kegiatan::selectRaw('jenis_kegiatan, COUNT(*) as total')
            ->groupBy('jenis_kegiatan');

        if ($request && $request->filled('jenis_kegiatan')) {
            $query->where('jenis_kegiatan', $request->jenis_kegiatan);
        }

        return $query->get();
    }

    private function getAnggotaPerJenisKelamin($request = null)
    {
        $query = Anggota::selectRaw('jenis_kelamin, COUNT(*) as total')
            ->groupBy('jenis_kelamin');

        if ($request && $request->filled('jenis_kelamin')) {
            $query->where('jenis_kelamin', $request->jenis_kelamin);
        }

        return $query->get();
    }

    private function getAnggotaPerUmur($request = null)
    {
        $query = Anggota::selectRaw('
            CASE 
                WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) < 20 THEN "0-19"
                WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) BETWEEN 20 AND 29 THEN "20-29"
                WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) BETWEEN 30 AND 39 THEN "30-39"
                WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) BETWEEN 40 AND 49 THEN "40-49"
                WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) BETWEEN 50 AND 59 THEN "50-59"
                ELSE "60+"
            END as kelompok_umur,
            COUNT(*) as total
        ')->groupBy('kelompok_umur');

        return $query->get();
    }

    private function getKegiatanPerStruktur($request = null)
    {
        // Get activities through participants and their current organizational structure
        $query = Kegiatan::join('tbl_kegiatan_partisipan', 'tbl_kegiatan.id', '=', 'tbl_kegiatan_partisipan.kegiatan_id')
            ->join('tbl_anggota', 'tbl_kegiatan_partisipan.anggota_id', '=', 'tbl_anggota.id')
            ->join('tbl_mutasi', function($join) {
                $join->on('tbl_anggota.id', '=', 'tbl_mutasi.anggota_id')
                     ->where('tbl_mutasi.status_mutasi', '=', 'Disetujui')
                     ->whereRaw('tbl_mutasi.tanggal_mutasi = (
                         SELECT MAX(tanggal_mutasi) 
                         FROM tbl_mutasi m2 
                         WHERE m2.anggota_id = tbl_anggota.id 
                         AND m2.status_mutasi = "Disetujui"
                     )');
            })
            ->join('tbl_struktur_organisasi', 'tbl_mutasi.struktur_ke', '=', 'tbl_struktur_organisasi.id')
            ->selectRaw('tbl_struktur_organisasi.nama_struktur, COUNT(DISTINCT tbl_kegiatan.id) as total')
            ->groupBy('tbl_struktur_organisasi.id', 'tbl_struktur_organisasi.nama_struktur');

        if ($request && $request->filled('struktur_organisasi_id')) {
            $query->where('tbl_mutasi.struktur_ke', $request->struktur_organisasi_id);
        }

        return $query->get();
    }

    private function getKegiatanPerStatus($request = null)
    {
        $query = Kegiatan::selectRaw('status_kegiatan, COUNT(*) as total')
            ->groupBy('status_kegiatan');

        if ($request && $request->filled('status_kegiatan')) {
            $query->where('status_kegiatan', $request->status_kegiatan);
        }

        return $query->get();
    }

    private function getMutasiPerJenis($request = null)
    {
        $query = Mutasi::selectRaw('jenis_mutasi, COUNT(*) as total')
            ->groupBy('jenis_mutasi');

        if ($request && $request->filled('jenis_mutasi')) {
            $query->where('jenis_mutasi', $request->jenis_mutasi);
        }

        return $query->get();
    }

    private function getMutasiPerStatus($request = null)
    {
        $query = Mutasi::selectRaw('status_mutasi as status, COUNT(*) as total')
            ->groupBy('status_mutasi');

        if ($request && $request->filled('status')) {
            $query->where('status_mutasi', $request->status);
        }

        return $query->get();
    }

    private function getMutasiPerStruktur($request = null)
    {
        // Mutasi tidak memiliki struktur_organisasi_id, gunakan struktur_ke
        $query = Mutasi::join('tbl_struktur_organisasi', 'tbl_mutasi.struktur_ke', '=', 'tbl_struktur_organisasi.id')
            ->selectRaw('tbl_struktur_organisasi.nama_struktur, COUNT(*) as total')
            ->groupBy('tbl_struktur_organisasi.id', 'tbl_struktur_organisasi.nama_struktur');

        if ($request && $request->filled('struktur_organisasi_id')) {
            $query->where('tbl_mutasi.struktur_ke', $request->struktur_organisasi_id);
        }

        return $query->get();
    }

    private function getTakwimPerStatus($request = null)
    {
        $query = Takwim::selectRaw('status_takwim, COUNT(*) as total')
            ->groupBy('status_takwim');

        if ($request && $request->filled('status_takwim')) {
            $query->where('status_takwim', $request->status_takwim);
        }

        return $query->get();
    }

    private function getTakwimPerBulan($request = null)
    {
        $query = Takwim::selectRaw('YEAR(tanggal_mulai) as tahun, MONTH(tanggal_mulai) as bulan, COUNT(*) as total')
            ->groupBy('tahun', 'bulan')
            ->orderBy('tahun', 'asc')
            ->orderBy('bulan', 'asc');

        if ($request && $request->filled('tanggal_mulai')) {
            $query->where('tanggal_mulai', '>=', $request->tanggal_mulai);
        }
        if ($request && $request->filled('tanggal_selesai')) {
            $query->where('tanggal_selesai', '<=', $request->tanggal_selesai);
        }

        return $query->get();
    }

    private function getTakwimPerStruktur($request = null)
    {
        $query = Takwim::join('tbl_struktur_organisasi', 'tbl_takwim.struktur_penyelenggara_id', '=', 'tbl_struktur_organisasi.id')
            ->selectRaw('tbl_struktur_organisasi.nama_struktur, COUNT(*) as total')
            ->groupBy('tbl_struktur_organisasi.id', 'tbl_struktur_organisasi.nama_struktur');

        if ($request && $request->filled('struktur_penyelenggara_id')) {
            $query->where('tbl_takwim.struktur_penyelenggara_id', $request->struktur_penyelenggara_id);
        }

        return $query->get();
    }

    private function getTakwimPerDurasi($request = null)
    {
        $query = Takwim::selectRaw('
            CASE 
                WHEN DATEDIFF(tanggal_selesai, tanggal_mulai) <= 7 THEN "1-7 hari"
                WHEN DATEDIFF(tanggal_selesai, tanggal_mulai) BETWEEN 8 AND 30 THEN "8-30 hari"
                WHEN DATEDIFF(tanggal_selesai, tanggal_mulai) BETWEEN 31 AND 90 THEN "31-90 hari"
                ELSE "90+ hari"
            END as durasi,
            COUNT(*) as total
        ')->groupBy('durasi');

        return $query->get();
    }
}
