<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\RekomposisiUpa;
use App\Models\RekomposisiUpaDetail;
use App\Models\Dpc;
use App\Models\Anggota;
use App\Models\Upa;
use Illuminate\Support\Facades\DB;
use PDF;

class RekomposisiUpaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = RekomposisiUpa::with(['dpc', 'creator', 'approver', 'details']);

        // Filter by DPC
        if ($request->filled('dpc_id')) {
            $query->where('dpc_id', $request->dpc_id);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->filled('tanggal_dari')) {
            $query->whereDate('tanggal_sk', '>=', $request->tanggal_dari);
        }

        if ($request->filled('tanggal_sampai')) {
            $query->whereDate('tanggal_sk', '<=', $request->tanggal_sampai);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nomor_sk', 'like', "%{$search}%")
                  ->orWhere('judul_sk', 'like', "%{$search}%")
                  ->orWhereHas('dpc', function($q) use ($search) {
                      $q->where('nama_dpc', 'like', "%{$search}%");
                  });
            });
        }

        $rekomposisi = $query->orderBy('created_at', 'desc')->paginate(15);

        $dpcList = Dpc::where('is_active', true)->get();

        return view('rekomposisi-upa.index', compact('rekomposisi', 'dpcList'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $dpcList = Dpc::where('is_active', true)->get();
        $pembimbingList = Anggota::where('status_keanggotaan', 'Aktif')
            ->whereHas('refJenjang', function($q) {
                $q->whereIn('jenjang', ['Madya', 'Utama', 'Dewasa']);
            })
            ->get();

        return view('rekomposisi-upa.create', compact('dpcList', 'pembimbingList'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'dpc_id' => 'required|exists:tbl_dpc,dpc_id',
            'judul_sk' => 'required|string|max:255',
            'tanggal_sk' => 'required|date',
            'tanggal_rapat' => 'required|date',
            'ditetapkan_tanggal' => 'required|date',
            'ketua_nama' => 'required|string|max:255',
            'ketua_jabatan' => 'required|string|max:255',
            'ditetapkan_di' => 'required|string|max:255',
            'nomor_sk_lampiran' => 'nullable|string|max:100',
            'details' => 'required|array|min:1',
            'details.*.kode_upa' => 'required|string|max:50',
            'details.*.pembimbing_baru_id' => 'required|exists:tbl_anggota,id',
            'details.*.anggota_id' => 'required|exists:tbl_anggota,id',
            'details.*.dpr_id' => 'nullable|string|max:50',
            'details.*.keterangan' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            // Get DPC data
            $dpc = Dpc::find($request->dpc_id);
            
            // Create rekomposisi
            $rekomposisi = RekomposisiUpa::create([
                'nomor_sk' => RekomposisiUpa::generateNomorSK($request->dpc_id),
                'tanggal_sk' => $request->tanggal_sk,
                'tanggal_rapat' => $request->tanggal_rapat,
                'dpc_id' => $request->dpc_id,
                'judul_sk' => $request->judul_sk,
                'menimbang_a' => 'bahwa dalam upaya mendukung kegitan rekrutmen, pembinaan dan melakukan kegiatan untuk pemenangan partai dilingkungan daerah Jakarta Timur;',
                'menimbang_b' => 'bahwa dalam rangka membantu efektifitas dan memperkuat struktur partai di tingkat rukun warga, kelurahan atau kecamatan di daerah Jakarta Timur;',
                'menimbang_c' => 'bahwa terkait dengan poin 2 tersebut diatas maka dikeluarkan Surat Keputusan.',
                'mengingat_1' => 'Anggaran Dasar Bab IV Pasal 10 dan 11 tentang Anggota Partai;',
                'mengingat_2' => 'Anggaran Rumah Tangga Bab III Pasal 3 tentang Anggota Partai.',
                'mengingat_3' => 'Panduan UPA',
                'memperhatikan' => 'Hasil Rapat Bidang Kaderisasi tanggal ' . \Carbon\Carbon::parse($request->tanggal_rapat)->format('d F Y'),
                'memutuskan' => 'Menetapkan',
                'menetapkan' => 'REKOMPOSISI UNIT PEMBINAAN ANGGOTA MADYA DEWAN PENGURUS CABANG KECAMATAN ' . strtoupper($dpc->nama_dpc ?? ''),
                'pertama' => 'Menetapkan adanya Rekomposisi UPA Madya untuk memperkuat struktur Partai Dewan Pengurus Cabang Kecamatan ' . $dpc->nama_dpc,
                'kedua' => 'Menetapkan Rekomposisi Unit Pembina Anggota Madya, Dewan Pengurus Cabang Kecamatan ' . $dpc->nama_dpc . ' terlampir',
                'ketiga' => 'Keputusan ini berlaku sejak tanggal ditetapkan;',
                'keempat' => 'Keputusan ini akan ditinjau kembali apabila terdapat kekeliruan di kemudian hari.',
                'ditetapkan_di' => $request->ditetapkan_di,
                'nomor_sk_lampiran' => $request->nomor_sk_lampiran,
                'ditetapkan_tanggal' => $request->ditetapkan_tanggal,
                'ketua_nama' => $request->ketua_nama,
                'ketua_jabatan' => $request->ketua_jabatan,
                'created_by' => auth()->id(),
                'status' => 'Draft'
            ]);

            // Create details
            foreach ($request->details as $index => $detail) {
                RekomposisiUpaDetail::create([
                    'rekomposisi_id' => $rekomposisi->id,
                    'kode_upa' => $detail['kode_upa'],
                    'pembimbing_baru_id' => $detail['pembimbing_baru_id'],
                    'pembimbing_lama_id' => $detail['pembimbing_lama_id'] ?? null,
                    'anggota_id' => $detail['anggota_id'],
                    'dpr_id' => $detail['dpr_id'] ?? null,
                    'keterangan' => $detail['keterangan'] ?? null,
                    'urutan' => $index + 1
                ]);
            }

            DB::commit();

            return redirect()->route('rekomposisi-upa.show', $rekomposisi->id)
                ->with('success', 'Rekomposisi UPA berhasil dibuat');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()
                ->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(RekomposisiUpa $rekomposisiUpa)
    {
        $rekomposisiUpa->load(['dpc', 'creator', 'approver', 'details.anggota', 'details.pembimbingBaru', 'details.pembimbingLama']);

        return view('rekomposisi-upa.show', compact('rekomposisiUpa'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(RekomposisiUpa $rekomposisiUpa)
    {
        if (!$rekomposisiUpa->canEdit()) {
            return redirect()->route('rekomposisi-upa.show', $rekomposisiUpa->id)
                ->with('error', 'Rekomposisi tidak dapat diedit karena sudah diproses');
        }

        $rekomposisiUpa->load(['details']);
        $dpcList = Dpc::where('is_active', true)->get();
        $pembimbingList = Anggota::where('status_keanggotaan', 'Aktif')
            ->whereHas('refJenjang', function($q) {
                $q->whereIn('jenjang', ['Madya', 'Utama', 'Dewasa']);
            })
            ->get();

        return view('rekomposisi-upa.edit', compact('rekomposisiUpa', 'dpcList', 'pembimbingList'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, RekomposisiUpa $rekomposisiUpa)
    {
        if (!$rekomposisiUpa->canEdit()) {
            return redirect()->route('rekomposisi-upa.show', $rekomposisiUpa->id)
                ->with('error', 'Rekomposisi tidak dapat diedit karena sudah diproses');
        }

        $request->validate([
            'judul_sk' => 'required|string|max:255',
            'tanggal_sk' => 'required|date',
            'tanggal_rapat' => 'required|date',
            'ditetapkan_tanggal' => 'required|date',
            'ketua_nama' => 'required|string|max:255',
            'ketua_jabatan' => 'required|string|max:255',
            'ditetapkan_di' => 'required|string|max:255',
            'nomor_sk_lampiran' => 'nullable|string|max:100',
        ]);

        // Get DPC data
        $dpc = $rekomposisiUpa->dpc;
        
        $rekomposisiUpa->update([
            'judul_sk' => $request->judul_sk,
            'tanggal_sk' => $request->tanggal_sk,
            'tanggal_rapat' => $request->tanggal_rapat,
            'ditetapkan_tanggal' => $request->ditetapkan_tanggal,
            'ketua_nama' => $request->ketua_nama,
            'ketua_jabatan' => $request->ketua_jabatan,
            'ditetapkan_di' => $request->ditetapkan_di,
            'nomor_sk_lampiran' => $request->nomor_sk_lampiran,
            'memperhatikan' => 'Hasil Rapat Bidang Kaderisasi tanggal ' . \Carbon\Carbon::parse($request->tanggal_rapat)->format('d F Y'),
            'menetapkan' => 'REKOMPOSISI UNIT PEMBINAAN ANGGOTA MADYA DEWAN PENGURUS CABANG KECAMATAN ' . strtoupper($dpc->nama_dpc ?? ''),
            'pertama' => 'Menetapkan adanya Rekomposisi UPA Madya untuk memperkuat struktur Partai Dewan Pengurus Cabang Kecamatan ' . $dpc->nama_dpc,
            'kedua' => 'Menetapkan Rekomposisi Unit Pembina Anggota Madya, Dewan Pengurus Cabang Kecamatan ' . $dpc->nama_dpc . ' terlampir',
        ]);

        return redirect()->route('rekomposisi-upa.show', $rekomposisiUpa->id)
            ->with('success', 'Rekomposisi UPA berhasil diperbarui');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(RekomposisiUpa $rekomposisiUpa)
    {
        if (!$rekomposisiUpa->canEdit()) {
            return redirect()->route('rekomposisi-upa.index')
                ->with('error', 'Rekomposisi tidak dapat dihapus karena sudah diproses');
        }

        $rekomposisiUpa->delete();

        return redirect()->route('rekomposisi-upa.index')
            ->with('success', 'Rekomposisi UPA berhasil dihapus');
    }

    /**
     * Approve rekomposisi
     */
    public function approve(RekomposisiUpa $rekomposisiUpa)
    {
        if (!$rekomposisiUpa->canApprove()) {
            return redirect()->route('rekomposisi-upa.show', $rekomposisiUpa->id)
                ->with('error', 'Rekomposisi tidak dapat disetujui');
        }

        $rekomposisiUpa->approve(auth()->id());

        return redirect()->route('rekomposisi-upa.show', $rekomposisiUpa->id)
            ->with('success', 'Rekomposisi UPA berhasil disetujui');
    }

    /**
     * Publish rekomposisi
     */
    public function publish(RekomposisiUpa $rekomposisiUpa)
    {
        if (!$rekomposisiUpa->canPublish()) {
            return redirect()->route('rekomposisi-upa.show', $rekomposisiUpa->id)
                ->with('error', 'Rekomposisi tidak dapat dipublikasikan');
        }

        $rekomposisiUpa->publish();

        return redirect()->route('rekomposisi-upa.show', $rekomposisiUpa->id)
            ->with('success', 'Rekomposisi UPA berhasil dipublikasikan');
    }

    /**
     * Print SK rekomposisi
     */
    public function printSK(RekomposisiUpa $rekomposisiUpa)
    {
        $rekomposisiUpa->load(['dpc', 'details.anggota', 'details.pembimbingBaru', 'details.pembimbingLama']);

        // Group details by kode_upa
        $groupedDetails = $rekomposisiUpa->details->groupBy('kode_upa');

        $data = [
            'rekomposisi' => $rekomposisiUpa,
            'groupedDetails' => $groupedDetails
        ];

        return view('surat.rekomposisi-upa', $data);
    }

    /**
     * Generate PDF SK rekomposisi
     */
    public function generatePDF(RekomposisiUpa $rekomposisiUpa)
    {
        $rekomposisiUpa->load(['dpc', 'details.anggota', 'details.pembimbingBaru', 'details.pembimbingLama']);

        // Group details by kode_upa
        $groupedDetails = $rekomposisiUpa->details->groupBy('kode_upa');

        $data = [
            'rekomposisi' => $rekomposisiUpa,
            'groupedDetails' => $groupedDetails
        ];

        $pdf = PDF::loadView('surat.rekomposisi-upa', $data);
        $pdf->setPaper('A4', 'portrait');
        $pdf->setOptions([
            'isHtml5ParserEnabled' => true,
            'isRemoteEnabled' => false,
            'isPhpEnabled' => false,
            'isJavascriptEnabled' => false,
            'defaultFont' => 'Arial',
            'dpi' => 150,
        ]);

        // Clean filename to remove path separators
        $cleanNomorSk = str_replace(['/', '\\', ':', '*', '?', '"', '<', '>', '|'], '_', $rekomposisiUpa->nomor_sk);
        $filename = "SK_Rekomposisi_UPA_{$cleanNomorSk}.pdf";
        
        return $pdf->download($filename);
    }

    /**
     * Get anggota by DPC for AJAX
     */
    public function getAnggotaByDpc(Request $request)
    {
        $dpcId = $request->dpc_id;
        
        $anggota = Anggota::where('dpc_id', $dpcId)
            ->where('status_keanggotaan', 'Aktif')
            ->select('id', 'nama_lengkap', 'nomor_identitas_anggota')
            ->get();

        return response()->json($anggota);
    }

    /**
     * Get pembimbing by DPC for AJAX
     */
    public function getPembimbingByDpc(Request $request)
    {
        $dpcId = $request->dpc_id;
        
        $pembimbing = Anggota::where('dpc_id', $dpcId)
            ->where('status_keanggotaan', 'Aktif')
            ->whereHas('refJenjang', function($q) {
                $q->whereIn('jenjang', ['Madya', 'Utama', 'Dewasa']);
            })
            ->select('id', 'nama_lengkap', 'nomor_identitas_anggota')
            ->get();

        return response()->json($pembimbing);
    }
}
