<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PerpindahanAnggota;
use App\Models\Anggota;
use App\Models\Dpd;
use App\Models\Dpc;
use App\Models\Upa;
use App\Models\RefJenjang;
use App\Models\RefGrupUpa;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PerpindahanController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = PerpindahanAnggota::with([
            'anggota',
            'dpdBefore',
            'dpcBefore',
            'upaBefore',
            'jenjangBefore',
            'grupUpaBefore',
            'dpdAfter',
            'dpcAfter',
            'upaAfter',
            'jenjangAfter',
            'grupUpaAfter',
            'createdBy',
            'approvedBy'
        ]);

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status_perpindahan', $request->status);
        }

        // Filter berdasarkan tanggal
        if ($request->filled('tanggal_mulai')) {
            $query->where('tanggal_perpindahan', '>=', $request->tanggal_mulai);
        }

        if ($request->filled('tanggal_selesai')) {
            $query->where('tanggal_perpindahan', '<=', $request->tanggal_selesai);
        }

        // Filter berdasarkan anggota
        if ($request->filled('anggota_id')) {
            $query->where('anggota_id', $request->anggota_id);
        }

        // Search berdasarkan nama anggota
        if ($request->filled('search')) {
            $query->whereHas('anggota', function($q) use ($request) {
                $q->where('nama_lengkap', 'like', '%' . $request->search . '%')
                  ->orWhere('nomor_identitas_anggota', 'like', '%' . $request->search . '%');
            });
        }

        $perpindahan = $query->orderBy('tanggal_perpindahan', 'desc')
                            ->orderBy('created_at', 'desc')
                            ->paginate(20);

        // Data untuk filter
        $anggotaList = Anggota::select('id', 'nama_lengkap', 'nomor_identitas_anggota')
                             ->orderBy('nama_lengkap')
                             ->get();

        return view('perpindahan.index', compact('perpindahan', 'anggotaList'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $anggota = null;
        
        // Jika ada anggota_id di request, load data anggota
        if ($request->filled('anggota_id')) {
            $anggota = Anggota::with(['dpd', 'dpc', 'upa', 'refJenjang', 'refGrupUpa'])
                             ->findOrFail($request->anggota_id);
        }

        // Data untuk dropdown
        $dpdList = Dpd::active()->orderBy('nama_dpd')->get();
        $dpcList = Dpc::active()->orderBy('nama_dpc')->get();
        $upaList = Upa::active()->orderBy('nama_upa')->get();
        $jenjangList = RefJenjang::active()->orderBy('jenjang')->get();
        $grupUpaList = RefGrupUpa::active()->orderBy('nama_grup_upa')->get();

        return view('perpindahan.create', compact(
            'anggota',
            'dpdList',
            'dpcList',
            'upaList',
            'jenjangList',
            'grupUpaList'
        ));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'anggota_id' => 'required|exists:tbl_anggota,id',
            'tanggal_perpindahan' => 'required|date',
            'dpd_id_after' => 'nullable|exists:tbl_dpd,dpd_id',
            'dpc_id_after' => 'nullable|exists:tbl_dpc,dpc_id',
            'upa_id_after' => 'nullable|exists:tbl_upa,upa_id',
            'ref_jenjang_id_after' => 'nullable|exists:tbl_ref_jenjang,ref_jenjang_id',
            'ref_grup_upa_id_after' => 'nullable|exists:tbl_ref_grup_upa,ref_grup_upa_id',
            'keterangan' => 'nullable|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            // Ambil data anggota saat ini
            $anggota = Anggota::findOrFail($request->anggota_id);

            // Cek apakah ada perubahan data
            $hasChanges = $anggota->dpd_id != $request->dpd_id_after ||
                         $anggota->dpc_id != $request->dpc_id_after ||
                         $anggota->upa_id != $request->upa_id_after ||
                         $anggota->ref_jenjang_id != $request->ref_jenjang_id_after ||
                         $anggota->ref_grup_upa_id != $request->ref_grup_upa_id_after;

            if (!$hasChanges) {
                return redirect()->back()
                    ->withErrors(['error' => 'Tidak ada perubahan data yang diajukan.'])
                    ->withInput();
            }

            // Buat perpindahan
            $perpindahan = PerpindahanAnggota::create([
                'anggota_id' => $request->anggota_id,
                'tanggal_perpindahan' => $request->tanggal_perpindahan,
                'dpd_id_before' => $anggota->dpd_id,
                'dpc_id_before' => $anggota->dpc_id,
                'upa_id_before' => $anggota->upa_id,
                'ref_jenjang_id_before' => $anggota->ref_jenjang_id,
                'ref_grup_upa_id_before' => $anggota->ref_grup_upa_id,
                'dpd_id_after' => $request->dpd_id_after,
                'dpc_id_after' => $request->dpc_id_after,
                'upa_id_after' => $request->upa_id_after,
                'ref_jenjang_id_after' => $request->ref_jenjang_id_after,
                'ref_grup_upa_id_after' => $request->ref_grup_upa_id_after,
                'keterangan' => $request->keterangan,
                'status_perpindahan' => 'Diajukan',
                'created_by' => auth()->id(),
            ]);

            DB::commit();

            return redirect()->route('perpindahan.index')
                ->with('success', 'Perpindahan anggota berhasil diajukan.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors(['error' => 'Terjadi kesalahan: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $perpindahan = PerpindahanAnggota::with([
            'anggota',
            'dpdBefore',
            'dpcBefore',
            'upaBefore',
            'jenjangBefore',
            'grupUpaBefore',
            'dpdAfter',
            'dpcAfter',
            'upaAfter',
            'jenjangAfter',
            'grupUpaAfter',
            'createdBy',
            'approvedBy'
        ])->findOrFail($id);

        return view('perpindahan.show', compact('perpindahan'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $perpindahan = PerpindahanAnggota::with(['anggota'])->findOrFail($id);

        // Hanya bisa edit jika status masih Diajukan
        if ($perpindahan->status_perpindahan !== 'Diajukan') {
            return redirect()->route('perpindahan.show', $id)
                ->withErrors(['error' => 'Perpindahan yang sudah disetujui atau ditolak tidak dapat diedit.']);
        }

        // Data untuk dropdown
        $dpdList = Dpd::active()->orderBy('nama_dpd')->get();
        $dpcList = Dpc::active()->orderBy('nama_dpc')->get();
        $upaList = Upa::active()->orderBy('nama_upa')->get();
        $jenjangList = RefJenjang::active()->orderBy('jenjang')->get();
        $grupUpaList = RefGrupUpa::active()->orderBy('nama_grup_upa')->get();

        return view('perpindahan.edit', compact(
            'perpindahan',
            'dpdList',
            'dpcList',
            'upaList',
            'jenjangList',
            'grupUpaList'
        ));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $perpindahan = PerpindahanAnggota::findOrFail($id);

        // Hanya bisa edit jika status masih Diajukan
        if ($perpindahan->status_perpindahan !== 'Diajukan') {
            return redirect()->route('perpindahan.show', $id)
                ->withErrors(['error' => 'Perpindahan yang sudah disetujui atau ditolak tidak dapat diedit.']);
        }

        $request->validate([
            'tanggal_perpindahan' => 'required|date',
            'dpd_id_after' => 'nullable|exists:tbl_dpd,dpd_id',
            'dpc_id_after' => 'nullable|exists:tbl_dpc,dpc_id',
            'upa_id_after' => 'nullable|exists:tbl_upa,upa_id',
            'ref_jenjang_id_after' => 'nullable|exists:tbl_ref_jenjang,ref_jenjang_id',
            'ref_grup_upa_id_after' => 'nullable|exists:tbl_ref_grup_upa,ref_grup_upa_id',
            'keterangan' => 'nullable|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            // Cek apakah ada perubahan data
            $hasChanges = $perpindahan->dpd_id_before != $request->dpd_id_after ||
                         $perpindahan->dpc_id_before != $request->dpc_id_after ||
                         $perpindahan->upa_id_before != $request->upa_id_after ||
                         $perpindahan->ref_jenjang_id_before != $request->ref_jenjang_id_after ||
                         $perpindahan->ref_grup_upa_id_before != $request->ref_grup_upa_id_after;

            if (!$hasChanges) {
                return redirect()->back()
                    ->withErrors(['error' => 'Tidak ada perubahan data yang diajukan.'])
                    ->withInput();
            }

            // Update perpindahan
            $perpindahan->update([
                'tanggal_perpindahan' => $request->tanggal_perpindahan,
                'dpd_id_after' => $request->dpd_id_after,
                'dpc_id_after' => $request->dpc_id_after,
                'upa_id_after' => $request->upa_id_after,
                'ref_jenjang_id_after' => $request->ref_jenjang_id_after,
                'ref_grup_upa_id_after' => $request->ref_grup_upa_id_after,
                'keterangan' => $request->keterangan,
            ]);

            DB::commit();

            return redirect()->route('perpindahan.show', $id)
                ->with('success', 'Perpindahan anggota berhasil diperbarui.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors(['error' => 'Terjadi kesalahan: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $perpindahan = PerpindahanAnggota::findOrFail($id);

        // Hanya bisa hapus jika status masih Diajukan
        if ($perpindahan->status_perpindahan !== 'Diajukan') {
            return redirect()->route('perpindahan.show', $id)
                ->withErrors(['error' => 'Perpindahan yang sudah disetujui atau ditolak tidak dapat dihapus.']);
        }

        try {
            $perpindahan->delete();
            return redirect()->route('perpindahan.index')
                ->with('success', 'Perpindahan anggota berhasil dihapus.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->withErrors(['error' => 'Terjadi kesalahan: ' . $e->getMessage()]);
        }
    }

    /**
     * Approve perpindahan
     */
    public function approve(Request $request, string $id)
    {
        $perpindahan = PerpindahanAnggota::findOrFail($id);

        if ($perpindahan->status_perpindahan !== 'Diajukan') {
            return redirect()->back()
                ->withErrors(['error' => 'Hanya perpindahan dengan status Diajukan yang dapat disetujui.']);
        }

        $request->validate([
            'catatan_admin' => 'nullable|string|max:1000',
        ]);

        try {
            $perpindahan->approve(auth()->id(), $request->catatan_admin);
            
            return redirect()->route('perpindahan.show', $id)
                ->with('success', 'Perpindahan anggota berhasil disetujui.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->withErrors(['error' => 'Terjadi kesalahan: ' . $e->getMessage()]);
        }
    }

    /**
     * Reject perpindahan
     */
    public function reject(Request $request, string $id)
    {
        $perpindahan = PerpindahanAnggota::findOrFail($id);

        if ($perpindahan->status_perpindahan !== 'Diajukan') {
            return redirect()->back()
                ->withErrors(['error' => 'Hanya perpindahan dengan status Diajukan yang dapat ditolak.']);
        }

        $request->validate([
            'catatan_admin' => 'required|string|max:1000',
        ]);

        try {
            $perpindahan->reject(auth()->id(), $request->catatan_admin);
            
            return redirect()->route('perpindahan.show', $id)
                ->with('success', 'Perpindahan anggota berhasil ditolak.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->withErrors(['error' => 'Terjadi kesalahan: ' . $e->getMessage()]);
        }
    }

    /**
     * Print surat resmi (laporan perpindahan dalam periode)
     */
    public function printSuratResmi(Request $request)
    {
        $request->validate([
            'tanggal_mulai' => 'required|date',
            'tanggal_selesai' => 'required|date|after_or_equal:tanggal_mulai',
        ]);

        $perpindahan = PerpindahanAnggota::with([
            'anggota',
            'dpdBefore',
            'dpcBefore',
            'upaBefore',
            'jenjangBefore',
            'grupUpaBefore',
            'dpdAfter',
            'dpcAfter',
            'upaAfter',
            'jenjangAfter',
            'grupUpaAfter',
            'createdBy',
            'approvedBy'
        ])
        ->whereBetween('tanggal_perpindahan', [$request->tanggal_mulai, $request->tanggal_selesai])
        ->where('status_perpindahan', 'Disetujui')
        ->orderBy('tanggal_perpindahan')
        ->get();

        return view('perpindahan.print.surat-resmi', compact('perpindahan'));
    }

    /**
     * Print surat penempatan (per anggota)
     */
    public function printSuratPenempatan(string $id)
    {
        $perpindahan = PerpindahanAnggota::with([
            'anggota',
            'dpdBefore',
            'dpcBefore',
            'upaBefore',
            'jenjangBefore',
            'grupUpaBefore',
            'dpdAfter',
            'dpcAfter',
            'upaAfter',
            'jenjangAfter',
            'grupUpaAfter',
            'createdBy',
            'approvedBy'
        ])->findOrFail($id);

        if ($perpindahan->status_perpindahan !== 'Disetujui') {
            return redirect()->route('perpindahan.show', $id)
                ->withErrors(['error' => 'Hanya perpindahan yang sudah disetujui yang dapat dicetak.']);
        }

        return view('perpindahan.print.surat-penempatan', compact('perpindahan'));
    }
}
