<?php

namespace App\Http\Controllers;

use App\Models\Anggota;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class OptimizedAnggotaController extends Controller
{
    /**
     * Display a listing of the resource with optimizations.
     */
    public function index(Request $request)
    {
        // Cache key based on request parameters
        $cacheKey = 'anggota_index_' . md5(serialize($request->all()));
        
        return Cache::remember($cacheKey, 300, function () use ($request) {
            // Optimized query with specific column selection
            $query = Anggota::select([
                'id', 'nomor_identitas_anggota', 'nama_lengkap', 'jenis_kelamin', 
                'kota', 'ref_jenjang_id', 'upa_id', 'dpc_id', 'dpd_id', 
                'status_keanggotaan', 'email', 'foto', 'created_at'
            ])->with([
                'refJenjang:id,ref_jenjang_id,jenjang',
                'upa:id,upa_id,nama_upa',
                'dpc:id,dpc_id,nama_dpc',
                'dpd:id,dpd_id,nama_dpd'
            ]);
            
            // Apply filters
            $this->applyFilters($query, $request);
            
            // Order by for better performance
            $query->orderBy('created_at', 'desc');
            
            $anggota = $query->paginate(10);
            
            // Get cached filter data
            $filterData = $this->getCachedFilterData();
            
            return view('anggota.index', array_merge([
                'anggota' => $anggota
            ], $filterData));
        });
    }

    /**
     * Apply filters to the query
     */
    private function applyFilters($query, Request $request)
    {
        // Search filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_lengkap', 'like', "%{$search}%")
                  ->orWhere('nomor_identitas_anggota', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }
        
        // Status filter
        if ($request->filled('status')) {
            $query->where('status_keanggotaan', $request->status);
        }
        
        // Jenjang filter
        if ($request->filled('jenjang')) {
            $query->where('ref_jenjang_id', $request->jenjang);
        }
        
        // Provinsi filter
        if ($request->filled('provinsi')) {
            $query->where('provinsi', $request->provinsi);
        }
        
        // DPC filter
        if ($request->filled('dpc')) {
            $query->where('dpc_id', $request->dpc);
        }
        
        // DPD filter
        if ($request->filled('dpd')) {
            $query->where('dpd_id', $request->dpd);
        }
    }

    /**
     * Get cached filter data
     */
    private function getCachedFilterData()
    {
        return [
            'statusList' => Cache::remember('anggota_status_list', 3600, function() {
                return Anggota::select('status_keanggotaan')
                             ->distinct()
                             ->pluck('status_keanggotaan');
            }),
            
            'jenjangList' => Cache::remember('ref_jenjang_active', 3600, function() {
                return \App\Models\RefJenjang::select('ref_jenjang_id', 'jenjang')
                                            ->where('status', 'Aktif')
                                            ->get();
            }),
            
            'provinsiList' => Cache::remember('anggota_provinsi_list', 3600, function() {
                return Anggota::select('provinsi')
                             ->distinct()
                             ->whereNotNull('provinsi')
                             ->pluck('provinsi');
            })
        ];
    }

    /**
     * Clear cache for anggota data
     */
    public function clearCache()
    {
        Cache::forget('anggota_status_list');
        Cache::forget('ref_jenjang_active');
        Cache::forget('anggota_provinsi_list');
        
        // Clear paginated cache
        if (config('cache.default') === 'redis' && method_exists(Cache::getStore(), 'getRedis')) {
            $keys = Cache::getRedis()->keys('*anggota_index_*');
            foreach ($keys as $key) {
                Cache::forget(str_replace(config('cache.prefix') . ':', '', $key));
            }
        } else {
            // Untuk driver lain, clear semua cache
            Cache::flush();
        }
        
        return response()->json(['message' => 'Cache cleared successfully']);
    }
}