<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Mutasi;
use App\Models\Anggota;
use App\Models\StrukturOrganisasi;
use App\Models\Dpd;
use App\Models\Dpc;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use PDF;

class MutasiController extends Controller
{
    public function index(Request $request)
    {
        $query = Mutasi::with(['anggota.dpc', 'anggota.upa', 'strukturDari', 'strukturKe']);
        
        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status_mutasi', $request->status);
        }
        
        // Search berdasarkan nama anggota
        if ($request->filled('search')) {
            $query->whereHas('anggota', function($q) use ($request) {
                $q->where('nama_lengkap', 'like', '%' . $request->search . '%')
                  ->orWhere('nomor_identitas_anggota', 'like', '%' . $request->search . '%');
            });
        }
        
        $mutasis = $query->orderBy('created_at', 'desc')->paginate(10);
        
        return view('mutasi.index', compact('mutasis'));
    }

    public function create()
    {
        // Don't cache large datasets to avoid max_allowed_packet errors
        $anggotas = Anggota::where('status_keanggotaan', 'Aktif')
                          ->select('id', 'nama_lengkap', 'nomor_identitas_anggota')
                          ->orderBy('nama_lengkap')
                          ->get();
        
        $strukturs = StrukturOrganisasi::select('id', 'nama_struktur')
                          ->orderBy('nama_struktur')
                          ->get();
        
        $dpds = Dpd::active()
                   ->select('dpd_id', 'nama_dpd')
                   ->orderBy('nama_dpd')
                   ->get();
        
        $dpcs = Dpc::active()
                   ->select('dpc_id', 'nama_dpc')
                   ->orderBy('nama_dpc')
                   ->get();
        
        return view('mutasi.create', compact('anggotas', 'strukturs', 'dpds', 'dpcs'));
    }

    public function getAnggotaDetails(Request $request)
    {
        try {
            $anggotaId = $request->get('anggota_id');
            
            if (!$anggotaId) {
                return response()->json(['error' => 'Anggota ID is required'], 400);
            }
            
            $anggota = Anggota::with(['dpc', 'dpd'])->find($anggotaId);
            
            if (!$anggota) {
                return response()->json(['error' => 'Anggota not found'], 404);
            }
            
            return response()->json([
                'dpc_id' => $anggota->dpc_id,
                'dpc_nama' => $anggota->dpc ? $anggota->dpc->nama_dpc : null,
                'dpd_id' => $anggota->dpd_id,
                'dpd_nama' => $anggota->dpd ? $anggota->dpd->nama_dpd : null,
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getAnggotaDetails: ' . $e->getMessage());
            return response()->json(['error' => 'Internal server error'], 500);
        }
    }

    public function getDpcByDpd(Request $request)
    {
        try {
            $dpdId = $request->get('dpd_id');
            
            if (!$dpdId) {
                return response()->json(['error' => 'DPD ID is required'], 400);
            }
            
            $dpcs = Dpc::where('dpd_id', $dpdId)
                       ->active()
                       ->orderBy('nama_dpc')
                       ->get();
            
            return response()->json($dpcs);
        } catch (\Exception $e) {
            \Log::error('Error in getDpcByDpd: ' . $e->getMessage());
            return response()->json(['error' => 'Internal server error'], 500);
        }
    }

    public function store(Request $request)
    {
        $rules = [
            'anggota_id' => 'required|exists:tbl_anggota,id',
            'jenis_mutasi' => 'required|in:Keluar,Penempatan,Rekomposisi',
            'tanggal_mutasi' => 'required|date',
            'dpd_tujuan' => 'required|exists:tbl_dpd,dpd_id',
            'dpc_tujuan' => 'nullable|exists:tbl_dpc,dpc_id',
            'alamat_tujuan' => 'required|string|max:500',
            'alasan_mutasi' => 'required|string|max:200',
            'catatan' => 'nullable|string|max:1000',
            'dpc_asal' => 'nullable|exists:tbl_dpc,dpc_id',
            'dpd_asal' => 'nullable|exists:tbl_dpd,dpd_id',
            'struktur_dari' => 'nullable|exists:tbl_struktur_organisasi,id',
            'struktur_ke' => 'nullable|exists:tbl_struktur_organisasi,id',
        ];
        
        $messages = [
            'anggota_id.required' => 'Anggota wajib dipilih.',
            'anggota_id.exists' => 'Anggota yang dipilih tidak valid.',
            'jenis_mutasi.required' => 'Jenis mutasi wajib dipilih.',
            'jenis_mutasi.in' => 'Jenis mutasi tidak valid.',
            'tanggal_mutasi.required' => 'Tanggal mutasi wajib diisi.',
            'dpd_tujuan.required' => 'DPD Tujuan wajib dipilih.',
            'dpd_tujuan.exists' => 'DPD Tujuan yang dipilih tidak valid.',
            'dpc_tujuan.exists' => 'DPC Tujuan yang dipilih tidak valid.',
            'alamat_tujuan.required' => 'Alamat tujuan wajib diisi.',
            'alamat_tujuan.max' => 'Alamat tujuan maksimal 500 karakter.',
            'alasan_mutasi.required' => 'Alasan mutasi wajib dipilih.',
            'alasan_mutasi.max' => 'Alasan mutasi maksimal 200 karakter.',
            'catatan.max' => 'Catatan maksimal 1000 karakter.',
            'struktur_dari.exists' => 'Struktur asal tidak valid.',
            'struktur_ke.exists' => 'Struktur tujuan tidak valid.',
        ];

        $request->validate($rules, $messages);

        try {
            DB::beginTransaction();

            $data = $request->only([
                'anggota_id', 'jenis_mutasi', 'tanggal_mutasi', 
                'alamat_tujuan', 'alasan_mutasi', 'catatan'
            ]);
            
            // Ambil data anggota terlebih dahulu
            $anggota = Anggota::with(['dpc', 'dpd'])->find($request->anggota_id);
            
            // Set struktur_ke - untuk mutasi keluar, gunakan DPD tujuan
            if ($request->dpc_tujuan) {
                // Jika ada DPC tujuan, cari struktur organisasi yang sesuai
                $dpc = Dpc::find($request->dpc_tujuan);
                if ($dpc) {
                    $data['struktur_ke'] = $this->getValidStrukturOrganisasi($dpc->nama_dpc);
                }
            } else {
                // Jika tidak ada DPC, gunakan DPD tujuan
                $dpd = Dpd::find($request->dpd_tujuan);
                if ($dpd) {
                    $data['struktur_ke'] = $this->getValidStrukturOrganisasi($dpd->nama_dpd);
                }
            }
            
            // Set struktur_dari - ambil dari data anggota
            if ($anggota && $anggota->dpc) {
                $data['struktur_dari'] = $this->getValidStrukturOrganisasi($anggota->dpc->nama_dpc);
            }
            
            // Set DPD dan DPC - pastikan nilai valid
            $data['dpd_dari'] = $anggota && $anggota->dpd ? $anggota->dpd_id : null;
            $data['dpd_ke'] = $request->dpd_tujuan;
            
            // Validasi tambahan untuk memastikan foreign key valid
            if ($data['struktur_dari'] && !StrukturOrganisasi::find($data['struktur_dari'])) {
                $data['struktur_dari'] = null;
            }
            if ($data['struktur_ke'] && !StrukturOrganisasi::find($data['struktur_ke'])) {
                $data['struktur_ke'] = null;
            }
            if ($data['dpd_dari'] && !Dpd::find($data['dpd_dari'])) {
                $data['dpd_dari'] = null;
            }
            if ($data['dpd_ke'] && !Dpd::find($data['dpd_ke'])) {
                $data['dpd_ke'] = null;
            }
            
            // Fallback: jika semua foreign key null, set ke null untuk menghindari constraint error
            if (!$data['struktur_dari'] && !$data['struktur_ke'] && !$data['dpd_dari'] && !$data['dpd_ke']) {
                // Set minimal data yang diperlukan
                $data['struktur_dari'] = null;
                $data['struktur_ke'] = null;
                $data['dpd_dari'] = null;
                $data['dpd_ke'] = null;
            }

            $mutasi = Mutasi::create($data);
            
            // Auto approve jika diminta
            if ($request->has('langsung_setujui')) {
                $mutasi->update([
                    'status_mutasi' => 'Disetujui',
                    'disetujui_oleh' => Auth::id(),
                    'tanggal_sk' => now()
                ]);
                $mutasi->generateNomorSK();
                
                // Jika mutasi keluar, ubah status anggota menjadi Tidak Aktif
                if ($mutasi->jenis_mutasi === 'Keluar') {
                    $mutasi->anggota->update([
                        'status_keanggotaan' => 'Tidak Aktif'
                    ]);
                }
            }

            DB::commit();
            
            return redirect()->route('mutasi.show', $mutasi->id)
                           ->with('success', 'Data mutasi berhasil dibuat!');
                           
        } catch (\Exception $e) {
            DB::rollback();
            
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Gagal membuat data mutasi: ' . $e->getMessage());
        }
    }

    public function show(Mutasi $mutasi)
    {
        $mutasi->load(['anggota', 'strukturDari', 'strukturKe', 'disetujuiOleh']);
        
        return view('mutasi.show', compact('mutasi'));
    }

    public function edit(Mutasi $mutasi)
    {
        if (!$mutasi->canBeEdited()) {
            return redirect()->route('mutasi.show', $mutasi)
                           ->with('error', 'Mutasi yang sudah disetujui tidak dapat diedit!');
        }
        
        $anggotas = Anggota::where('status_keanggotaan', 'Aktif')
                          ->orderBy('nama_lengkap')
                          ->get();
        $strukturs = StrukturOrganisasi::orderBy('nama_struktur')->get();
        
        return view('mutasi.edit', compact('mutasi', 'anggotas', 'strukturs'));
    }

    public function update(Request $request, Mutasi $mutasi)
    {
        if (!$mutasi->canBeEdited()) {
            return redirect()->route('mutasi.show', $mutasi)
                           ->with('error', 'Mutasi yang sudah disetujui tidak dapat diedit!');
        }

        $rules = [
            'anggota_id' => 'required|exists:tbl_anggota,id',
            'tanggal_mutasi' => 'required|date',
            'struktur_ke' => 'required|exists:tbl_struktur_organisasi,id',
            'alamat_tujuan' => 'required|string|max:500',
            'alasan_mutasi' => 'required|string|max:200',
            'catatan' => 'nullable|string|max:1000',
        ];

        $request->validate($rules);

        try {
            DB::beginTransaction();
            
            // Simpan jenis mutasi lama untuk perbandingan
            $jenisMutasiLama = $mutasi->jenis_mutasi;
            
            $mutasi->update($request->only([
                'anggota_id', 'tanggal_mutasi', 'struktur_dari', 
                'struktur_ke', 'alamat_tujuan', 'alasan_mutasi', 'catatan'
            ]));
            
            // Jika jenis mutasi diubah dari "Keluar" ke jenis lain, kembalikan status anggota
            if ($jenisMutasiLama === 'Keluar' && $mutasi->jenis_mutasi !== 'Keluar') {
                $mutasi->anggota->update([
                    'status_keanggotaan' => 'Aktif'
                ]);
            }
            
            DB::commit();
            
            return redirect()->route('mutasi.show', $mutasi)
                           ->with('success', 'Data mutasi berhasil diupdate!');
                           
        } catch (\Exception $e) {
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Gagal update data mutasi: ' . $e->getMessage());
        }
    }

    public function destroy(Mutasi $mutasi)
    {
        if (!$mutasi->canBeEdited()) {
            return redirect()->route('mutasi.index')
                           ->with('error', 'Mutasi yang sudah disetujui tidak dapat dihapus!');
        }

        try {
            DB::beginTransaction();
            
            // Jika mutasi keluar yang sudah disetujui dihapus, kembalikan status anggota
            if ($mutasi->jenis_mutasi === 'Keluar' && $mutasi->status_mutasi === 'Disetujui') {
                $mutasi->anggota->update([
                    'status_keanggotaan' => 'Aktif'
                ]);
            }
            
            $mutasi->delete();
            
            DB::commit();
            
            return redirect()->route('mutasi.index')
                           ->with('success', 'Data mutasi berhasil dihapus!');
                           
        } catch (\Exception $e) {
            return redirect()->route('mutasi.index')
                           ->with('error', 'Gagal menghapus data mutasi: ' . $e->getMessage());
        }
    }

    public function approve(Request $request, $id)
    {
        $mutasi = Mutasi::findOrFail($id);
        
        if (!$mutasi->canBeApproved()) {
            return redirect()->route('mutasi.show', $mutasi)
                           ->with('error', 'Mutasi ini tidak dapat disetujui!');
        }

        try {
            DB::beginTransaction();
            
            $mutasi->update([
                'status_mutasi' => 'Disetujui',
                'disetujui_oleh' => Auth::id(),
                'tanggal_sk' => now()
            ]);
            
            $mutasi->generateNomorSK();
            
            // Jika mutasi keluar, ubah status anggota menjadi Tidak Aktif
            if ($mutasi->jenis_mutasi === 'Keluar') {
                $mutasi->anggota->update([
                    'status_keanggotaan' => 'Tidak Aktif'
                ]);
            }
            
            DB::commit();
            
            return redirect()->route('mutasi.show', $mutasi)
                           ->with('success', 'Mutasi berhasil disetujui! Nomor SK: ' . $mutasi->nomor_sk);
                           
        } catch (\Exception $e) {
            DB::rollback();
            
            return redirect()->route('mutasi.show', $mutasi)
                           ->with('error', 'Gagal menyetujui mutasi: ' . $e->getMessage());
        }
    }

    /**
     * Menampilkan keranjang anggota yang bermutasi (status Tidak Aktif)
     */
    public function keranjangMutasi(Request $request)
    {
        $query = Anggota::where('status_keanggotaan', 'Tidak Aktif')
                        ->with(['dpc', 'dpd', 'refJenjang', 'mutasi' => function($q) {
                            $q->where('status_mutasi', 'Disetujui')
                              ->where('jenis_mutasi', 'Keluar')
                              ->orderBy('tanggal_sk', 'desc');
                        }]);
        
        // Filter berdasarkan DPC
        if ($request->filled('dpc_id')) {
            $query->where('dpc_id', $request->dpc_id);
        }
        
        // Filter berdasarkan DPD
        if ($request->filled('dpd_id')) {
            $query->where('dpd_id', $request->dpd_id);
        }
        
        // Search berdasarkan nama anggota
        if ($request->filled('search')) {
            $query->where(function($q) use ($request) {
                $q->where('nama_lengkap', 'like', '%' . $request->search . '%')
                  ->orWhere('nomor_identitas_anggota', 'like', '%' . $request->search . '%');
            });
        }
        
        // Filter berdasarkan tanggal mutasi
        if ($request->filled('tanggal_dari')) {
            $query->whereHas('mutasi', function($q) use ($request) {
                $q->where('tanggal_sk', '>=', $request->tanggal_dari);
            });
        }
        
        if ($request->filled('tanggal_sampai')) {
            $query->whereHas('mutasi', function($q) use ($request) {
                $q->where('tanggal_sk', '<=', $request->tanggal_sampai);
            });
        }
        
        $anggotas = $query->orderBy('updated_at', 'desc')->paginate(15);
        
        // Data untuk filter
        $dpcs = Dpc::active()->orderBy('nama_dpc')->get();
        $dpds = Dpd::active()->orderBy('nama_dpd')->get();
        
        return view('mutasi.keranjang', compact('anggotas', 'dpcs', 'dpds'));
    }

    /**
     * Mengembalikan status anggota dari Tidak Aktif ke Aktif
     */
    public function kembalikanStatus(Request $request, $id)
    {
        $anggota = Anggota::findOrFail($id);
        
        if ($anggota->status_keanggotaan !== 'Tidak Aktif') {
            return redirect()->back()
                           ->with('error', 'Hanya anggota dengan status Tidak Aktif yang dapat dikembalikan!');
        }

        try {
            DB::beginTransaction();
            
            $anggota->update([
                'status_keanggotaan' => 'Aktif'
            ]);
            
            // Update mutasi terakhir menjadi dibatalkan
            $mutasiTerakhir = $anggota->mutasi()
                                    ->where('status_mutasi', 'Disetujui')
                                    ->where('jenis_mutasi', 'Keluar')
                                    ->orderBy('tanggal_sk', 'desc')
                                    ->first();
            
            if ($mutasiTerakhir) {
                $mutasiTerakhir->update([
                    'status_mutasi' => 'Dibatalkan',
                    'catatan' => $mutasiTerakhir->catatan . "\n\nDibatalkan pada: " . now()->format('d-m-Y H:i:s') . " oleh: " . Auth::user()->name
                ]);
            }
            
            DB::commit();
            
            return redirect()->back()
                           ->with('success', 'Status anggota berhasil dikembalikan ke Aktif!');
                           
        } catch (\Exception $e) {
            DB::rollback();
            
            return redirect()->back()
                           ->with('error', 'Gagal mengembalikan status anggota: ' . $e->getMessage());
        }
    }

    public function reject(Request $request, $id)
    {
        $mutasi = Mutasi::findOrFail($id);
        
        if (!$mutasi->canBeApproved()) {
            return redirect()->route('mutasi.show', $mutasi)
                           ->with('error', 'Mutasi ini tidak dapat ditolak!');
        }

        $request->validate([
            'alasan_penolakan' => 'required|string|max:500'
        ]);

        try {
            DB::beginTransaction();
            
            $mutasi->update([
                'status_mutasi' => 'Ditolak',
                'catatan' => $mutasi->catatan . "\n\nDitolak: " . $request->alasan_penolakan,
                'disetujui_oleh' => Auth::id(),
            ]);
            
            // Jika mutasi keluar ditolak, kembalikan status anggota menjadi Aktif
            if ($mutasi->jenis_mutasi === 'Keluar' && $mutasi->anggota->status_keanggotaan === 'Tidak Aktif') {
                $mutasi->anggota->update([
                    'status_keanggotaan' => 'Aktif'
                ]);
            }
            
            DB::commit();
            
            return redirect()->route('mutasi.show', $mutasi)
                           ->with('success', 'Mutasi berhasil ditolak!');
                           
        } catch (\Exception $e) {
            return redirect()->route('mutasi.show', $mutasi)
                           ->with('error', 'Gagal menolak mutasi: ' . $e->getMessage());
        }
    }

    public function printSurat($id)
    {
        $mutasi = Mutasi::with(['anggota', 'strukturKe'])->findOrFail($id);
        
        if ($mutasi->status_mutasi !== 'Disetujui') {
            return redirect()->route('mutasi.show', $mutasi)
                           ->with('error', 'Hanya mutasi yang sudah disetujui yang dapat dicetak!');
        }

        $data = [
            'nomor_surat' => $mutasi->nomor_sk,
            'tanggal' => now()->format('d-m-Y'),
            'mutasi' => $mutasi,
            'anggota' => $mutasi->anggota,
            'dpc_tujuan' => $mutasi->strukturKe,
            'ketua_kaderisasi' => 'H. Indarmawan, Ak'
        ];
        
        $pdf = PDF::loadView('surat.mutasi-keluar', $data);
        $pdf->setPaper('A4', 'portrait');
        $pdf->setOptions([
            'isHtml5ParserEnabled' => true,
            'isRemoteEnabled' => false,
            'isPhpEnabled' => false,
            'isJavascriptEnabled' => false,
            'defaultFont' => 'Arial',
            'dpi' => 150,
            'enable_font_subsetting' => false,
            'isFontSubsettingEnabled' => false,
            'chroot' => public_path(),
        ]);
        
        return $pdf->download("Surat_Mutasi_Keluar_{$mutasi->anggota->nama_lengkap}.pdf");
    }

    /**
     * Helper method untuk mendapatkan struktur organisasi yang valid
     */
    private function getValidStrukturOrganisasi($nama)
    {
        if (!$nama) {
            return null;
        }
        
        // Cari struktur organisasi yang cocok
        $struktur = StrukturOrganisasi::where('nama_struktur', 'like', '%' . $nama . '%')->first();
        
        if (!$struktur) {
            // Jika tidak ditemukan, coba buat struktur baru
            try {
                $struktur = StrukturOrganisasi::create([
                    'kode_struktur' => 'AUTO-' . time() . '-' . rand(1000, 9999),
                    'nama_struktur' => $nama,
                    'deskripsi' => 'Struktur organisasi untuk ' . $nama,
                    'status' => 'Aktif'
                ]);
            } catch (\Exception $e) {
                \Log::error('Error creating struktur organisasi: ' . $e->getMessage());
                return null;
            }
        }
        
        return $struktur ? $struktur->id : null;
    }
}