<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Dpc;
use App\Models\Upa;
use App\Models\Anggota;
use App\Models\RefJenjang;
use Illuminate\Support\Facades\DB;

class KomposisiController extends Controller
{
    /**
     * Menampilkan halaman utama komposisi DPC dan UPA
     */
    public function index(Request $request)
    {
        $query = Dpc::with(['dpd', 'upa' => function($query) {
            $query->active();
        }])
        ->active();

        // Filter DPD
        if ($request->filled('dpd_id')) {
            $query->where('dpd_id', $request->dpd_id);
        }

        // Filter Status UPA
        if ($request->filled('status_upa')) {
            $query->whereHas('upa', function($q) use ($request) {
                $q->where('status', $request->status_upa);
            });
        }

        // Filter Jenjang Anggota
        if ($request->filled('jenjang_id')) {
            $query->whereHas('anggota', function($q) use ($request) {
                $q->where('ref_jenjang_id', $request->jenjang_id)
                  ->where('status_keanggotaan', 'Aktif');
            });
        }

        // Filter berdasarkan jumlah anggota
        if ($request->filled('min_anggota')) {
            $query->whereHas('anggota', function($q) use ($request) {
                $q->where('status_keanggotaan', 'Aktif');
            }, '>=', $request->min_anggota);
        }

        $dpcList = $query->orderBy('nama_dpc')->get();

        // Data untuk filter
        $dpdList = \App\Models\Dpd::orderBy('nama_dpd')->get();
        $jenjangList = \App\Models\RefJenjang::active()->orderBy('jenjang')->get();
        
        $statusUpaOptions = [
            'Aktif' => 'Aktif',
            'Non-Aktif' => 'Non-Aktif',
            'Dibekukan' => 'Dibekukan'
        ];

        $statistik = $this->getStatistikKomposisi($request);

        return view('komposisi.index', compact('dpcList', 'statistik', 'dpdList', 'jenjangList', 'statusUpaOptions'));
    }

    /**
     * Menampilkan detail komposisi DPC tertentu
     */
    public function showDpc($dpcId)
    {
        $dpc = Dpc::with(['dpd', 'upa' => function($query) {
            $query->active()->with(['pembimbingAktif', 'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            }]);
        }])
        ->findOrFail($dpcId);

        $statistikDpc = $this->getStatistikDpc($dpcId);

        return view('komposisi.show-dpc', compact('dpc', 'statistikDpc'));
    }

    /**
     * Menampilkan detail komposisi UPA tertentu
     */
    public function showUpa($upaId)
    {
        $upa = Upa::with([
            'dpc.dpd',
            'pembimbingAktif' => function($query) {
                $query->with('refJenjang');
            },
            'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            },
            'refProvinsi',
            'refKota',
            'refKecamatan',
            'refKelurahan'
        ])
        ->findOrFail($upaId);

        $statistikUpa = $this->getStatistikUpa($upaId);

        return view('komposisi.show-upa', compact('upa', 'statistikUpa'));
    }

    /**
     * Print komposisi DPC
     */
    public function printDpc($dpcId)
    {
        $dpc = Dpc::with(['dpd', 'upa' => function($query) {
            $query->active()->with(['pembimbingAktif', 'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            }]);
        }])
        ->findOrFail($dpcId);

        $statistikDpc = $this->getStatistikDpc($dpcId);

        return view('komposisi.print-dpc', compact('dpc', 'statistikDpc'));
    }

    /**
     * Print komposisi UPA
     */
    public function printUpa($upaId)
    {
        $upa = Upa::with([
            'dpc.dpd',
            'pembimbingAktif' => function($query) {
                $query->with('refJenjang');
            },
            'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            },
            'refProvinsi',
            'refKota',
            'refKecamatan',
            'refKelurahan'
        ])
        ->findOrFail($upaId);

        $statistikUpa = $this->getStatistikUpa($upaId);

        return view('komposisi.print-upa', compact('upa', 'statistikUpa'));
    }

    /**
     * Print komposisi semua DPC
     */
    public function printAll(Request $request)
    {
        $query = Dpc::with(['dpd', 'upa' => function($query) {
            $query->active()->with(['pembimbingAktif', 'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            }]);
        }])
        ->active();

        // Apply filters if provided
        if ($request->filled('dpd_id')) {
            $query->where('dpd_id', $request->dpd_id);
        }

        if ($request->filled('status_upa')) {
            $query->whereHas('upa', function($q) use ($request) {
                $q->where('status', $request->status_upa);
            });
        }

        if ($request->filled('jenjang_id')) {
            $query->whereHas('anggota', function($q) use ($request) {
                $q->where('ref_jenjang_id', $request->jenjang_id)
                  ->where('status_keanggotaan', 'Aktif');
            });
        }

        if ($request->filled('min_anggota')) {
            $query->whereHas('anggota', function($q) use ($request) {
                $q->where('status_keanggotaan', 'Aktif');
            }, '>=', $request->min_anggota);
        }

        $dpcList = $query->orderBy('nama_dpc')->get();

        // Data untuk filter info
        $dpdList = \App\Models\Dpd::orderBy('nama_dpd')->get();
        $jenjangList = \App\Models\RefJenjang::active()->orderBy('jenjang')->get();
        
        $statusUpaOptions = [
            'Aktif' => 'Aktif',
            'Non-Aktif' => 'Non-Aktif',
            'Dibekukan' => 'Dibekukan'
        ];

        $statistik = $this->getStatistikKomposisi($request);

        return view('komposisi.print-all', compact('dpcList', 'statistik', 'dpdList', 'jenjangList', 'statusUpaOptions'));
    }

    /**
     * Print komposisi dalam format Surat Keputusan
     */
    public function printSk(Request $request)
    {
        $query = Dpc::with(['dpd', 'upa' => function($query) {
            $query->active()->with(['pembimbingAktif', 'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            }]);
        }])
        ->active();

        // Apply filters if provided
        if ($request->filled('dpd_id')) {
            $query->where('dpd_id', $request->dpd_id);
        }

        if ($request->filled('status_upa')) {
            $query->whereHas('upa', function($q) use ($request) {
                $q->where('status', $request->status_upa);
            });
        }

        if ($request->filled('jenjang_id')) {
            $query->whereHas('anggota', function($q) use ($request) {
                $q->where('ref_jenjang_id', $request->jenjang_id)
                  ->where('status_keanggotaan', 'Aktif');
            });
        }

        if ($request->filled('min_anggota')) {
            $query->whereHas('anggota', function($q) use ($request) {
                $q->where('status_keanggotaan', 'Aktif');
            }, '>=', $request->min_anggota);
        }

        $dpcList = $query->orderBy('nama_dpc')->get();

        return view('komposisi.print-sk', compact('dpcList'));
    }

    /**
     * Get statistik komposisi keseluruhan
     */
    private function getStatistikKomposisi($request = null)
    {
        $dpcQuery = Dpc::active();
        $upaQuery = Upa::active();
        $anggotaQuery = Anggota::where('status_keanggotaan', 'Aktif');

        // Apply filters if provided
        if ($request) {
            if ($request->filled('dpd_id')) {
                $dpcQuery->where('dpd_id', $request->dpd_id);
                $upaQuery->whereHas('dpc', function($q) use ($request) {
                    $q->where('dpd_id', $request->dpd_id);
                });
                $anggotaQuery->whereHas('dpc', function($q) use ($request) {
                    $q->where('dpd_id', $request->dpd_id);
                });
            }

            if ($request->filled('status_upa')) {
                $upaQuery->where('status', $request->status_upa);
                $anggotaQuery->whereHas('upa', function($q) use ($request) {
                    $q->where('status', $request->status_upa);
                });
            }

            if ($request->filled('jenjang_id')) {
                $anggotaQuery->where('ref_jenjang_id', $request->jenjang_id);
            }
        }

        $totalDpc = $dpcQuery->count();
        $totalUpa = $upaQuery->count();
        $totalAnggota = $anggotaQuery->count();

        // Statistik per jenjang
        $statistikJenjang = $anggotaQuery->clone()
            ->join('tbl_ref_jenjang', 'tbl_anggota.ref_jenjang_id', '=', 'tbl_ref_jenjang.ref_jenjang_id')
            ->select('tbl_ref_jenjang.jenjang', DB::raw('COUNT(*) as total'))
            ->groupBy('tbl_ref_jenjang.jenjang')
            ->orderBy('tbl_ref_jenjang.jenjang')
            ->get();

        // Statistik per DPC
        $statistikDpc = $dpcQuery->clone()
            ->withCount(['anggota' => function($query) {
                $query->where('status_keanggotaan', 'Aktif');
            }])
            ->withCount(['upa' => function($query) {
                $query->active();
            }])
            ->get();

        return [
            'total_dpc' => $totalDpc,
            'total_upa' => $totalUpa,
            'total_anggota' => $totalAnggota,
            'statistik_jenjang' => $statistikJenjang,
            'statistik_dpc' => $statistikDpc
        ];
    }

    /**
     * Get statistik DPC tertentu
     */
    private function getStatistikDpc($dpcId)
    {
        $totalUpa = Upa::where('dpc_id', $dpcId)->active()->count();
        $totalAnggota = Anggota::where('dpc_id', $dpcId)
            ->where('status_keanggotaan', 'Aktif')
            ->count();

        // Statistik per jenjang di DPC ini
        $statistikJenjang = Anggota::where('dpc_id', $dpcId)
            ->where('status_keanggotaan', 'Aktif')
            ->join('tbl_ref_jenjang', 'tbl_anggota.ref_jenjang_id', '=', 'tbl_ref_jenjang.ref_jenjang_id')
            ->select('tbl_ref_jenjang.jenjang', DB::raw('COUNT(*) as total'))
            ->groupBy('tbl_ref_jenjang.jenjang')
            ->orderBy('tbl_ref_jenjang.jenjang')
            ->get();

        // Statistik per UPA di DPC ini
        $statistikUpa = Upa::where('dpc_id', $dpcId)
            ->active()
            ->withCount(['anggota' => function($query) {
                $query->where('status_keanggotaan', 'Aktif');
            }])
            ->get();

        return [
            'total_upa' => $totalUpa,
            'total_anggota' => $totalAnggota,
            'statistik_jenjang' => $statistikJenjang,
            'statistik_upa' => $statistikUpa
        ];
    }

    /**
     * Get statistik UPA tertentu
     */
    private function getStatistikUpa($upaId)
    {
        $totalAnggota = Anggota::where('upa_id', $upaId)
            ->where('status_keanggotaan', 'Aktif')
            ->count();

        // Statistik per jenjang di UPA ini
        $statistikJenjang = Anggota::where('upa_id', $upaId)
            ->where('status_keanggotaan', 'Aktif')
            ->join('tbl_ref_jenjang', 'tbl_anggota.ref_jenjang_id', '=', 'tbl_ref_jenjang.ref_jenjang_id')
            ->select('tbl_ref_jenjang.jenjang', DB::raw('COUNT(*) as total'))
            ->groupBy('tbl_ref_jenjang.jenjang')
            ->orderBy('tbl_ref_jenjang.jenjang')
            ->get();

        // Data pembimbing aktif
        $pembimbingAktif = Anggota::whereHas('upaPembimbing', function($query) use ($upaId) {
            $query->where('upa_id', $upaId)
                  ->where('status', 'Aktif');
        })
        ->with('refJenjang')
        ->get();

        return [
            'total_anggota' => $totalAnggota,
            'statistik_jenjang' => $statistikJenjang,
            'pembimbing_aktif' => $pembimbingAktif
        ];
    }
}
