<?php

namespace App\Http\Controllers;

use App\Models\JenisKegiatan;
use Illuminate\Http\Request;

class JenisKegiatanController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = JenisKegiatan::query();
        
        // Filter berdasarkan pencarian
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_jenis_kegiatan', 'like', "%{$search}%")
                  ->orWhere('kode_jenis_kegiatan', 'like', "%{$search}%")
                  ->orWhere('kategori', 'like', "%{$search}%");
            });
        }
        
        // Filter berdasarkan kategori
        if ($request->has('kategori') && $request->kategori) {
            $query->where('kategori', $request->kategori);
        }
        
        // Filter berdasarkan status
        if ($request->has('status') && $request->status) {
            $query->where('is_active', $request->status == 'aktif');
        }
        
        $jenisKegiatans = $query->orderBy('kategori')->orderBy('nama_jenis_kegiatan')->paginate(10);
        
        // Data untuk filter dropdown
        $kategoriList = JenisKegiatan::select('kategori')
                                   ->whereNotNull('kategori')
                                   ->distinct()
                                   ->pluck('kategori');
        
        return view('jenis-kegiatan.index', compact('jenisKegiatans', 'kategoriList'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('jenis-kegiatan.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'kode_jenis_kegiatan' => 'required|string|max:20|unique:tbl_ref_jenis_kegiatan',
            'nama_jenis_kegiatan' => 'required|string|max:255',
            'deskripsi' => 'nullable|string',
            'kategori' => 'nullable|string|max:100',
            'durasi_standar' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
            'catatan' => 'nullable|string'
        ]);

        JenisKegiatan::create($validated);

        return redirect()->route('jenis-kegiatan.index')
                        ->with('success', 'Data jenis kegiatan berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(JenisKegiatan $jenisKegiatan)
    {
        return view('jenis-kegiatan.show', compact('jenisKegiatan'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(JenisKegiatan $jenisKegiatan)
    {
        return view('jenis-kegiatan.edit', compact('jenisKegiatan'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, JenisKegiatan $jenisKegiatan)
    {
        $validated = $request->validate([
            'kode_jenis_kegiatan' => 'required|string|max:20|unique:tbl_ref_jenis_kegiatan,kode_jenis_kegiatan,' . $jenisKegiatan->ref_jenis_kegiatan_id . ',ref_jenis_kegiatan_id',
            'nama_jenis_kegiatan' => 'required|string|max:255',
            'deskripsi' => 'nullable|string',
            'kategori' => 'nullable|string|max:100',
            'durasi_standar' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
            'catatan' => 'nullable|string'
        ]);

        $jenisKegiatan->update($validated);

        return redirect()->route('jenis-kegiatan.index')
                        ->with('success', 'Data jenis kegiatan berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(JenisKegiatan $jenisKegiatan)
    {
        $jenisKegiatan->delete();

        return redirect()->route('jenis-kegiatan.index')
                        ->with('success', 'Data jenis kegiatan berhasil dihapus.');
    }
}