<?php

namespace App\Http\Controllers;

use App\Models\IndikatorTakwim;
use Illuminate\Http\Request;

class IndikatorTakwimController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $indikators = IndikatorTakwim::orderBy('nomor')->get();
        return view('indikator-takwim.index', compact('indikators'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('indikator-takwim.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            \Log::info('Store request data:', $request->all());
            
            // Manual validation check
            if (empty($request->nomor)) {
                \Log::error('Nomor is empty');
                return back()->withInput()->with('error', 'Nomor urut harus diisi');
            }
            
            if (empty($request->indikator)) {
                \Log::error('Indikator is empty');
                return back()->withInput()->with('error', 'Indikator harus diisi');
            }
            
            // Check if nomor already exists
            if (IndikatorTakwim::where('nomor', $request->nomor)->exists()) {
                \Log::error('Nomor already exists: ' . $request->nomor);
                return back()->withInput()->with('error', 'Nomor urut sudah ada');
            }
            
            \Log::info('Validation passed, creating indikator...');
            
            $indikator = IndikatorTakwim::create([
                'nomor' => $request->nomor,
                'indikator' => $request->indikator,
                'is_active' => $request->has('is_active') ? true : false
            ]);

            \Log::info('Indikator created successfully with ID: ' . $indikator->id);

            return redirect()->route('indikator-takwim.index')
                ->with('success', 'Indikator berhasil ditambahkan.');
        } catch (Exception $e) {
            \Log::error('Store error: ' . $e->getMessage());
            return back()->withInput()->with('error', 'Error: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(IndikatorTakwim $indikatorTakwim)
    {
        return view('indikator-takwim.show', compact('indikatorTakwim'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(IndikatorTakwim $indikatorTakwim)
    {
        return view('indikator-takwim.edit', compact('indikatorTakwim'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, IndikatorTakwim $indikatorTakwim)
    {
        try {
            \Log::info('Update request data for ID ' . $indikatorTakwim->id . ':', $request->all());
            
            // Manual validation check
            if (empty($request->nomor)) {
                \Log::error('Nomor is empty');
                return back()->withInput()->with('error', 'Nomor urut harus diisi');
            }
            
            if (empty($request->indikator)) {
                \Log::error('Indikator is empty');
                return back()->withInput()->with('error', 'Indikator harus diisi');
            }
            
            // Check if nomor already exists (excluding current record)
            if (IndikatorTakwim::where('nomor', $request->nomor)->where('id', '!=', $indikatorTakwim->id)->exists()) {
                \Log::error('Nomor already exists: ' . $request->nomor);
                return back()->withInput()->with('error', 'Nomor urut sudah ada');
            }
            
            \Log::info('Validation passed, updating indikator...');
            
            $indikatorTakwim->update([
                'nomor' => $request->nomor,
                'indikator' => $request->indikator,
                'is_active' => $request->has('is_active') ? true : false
            ]);

            \Log::info('Indikator updated successfully with ID: ' . $indikatorTakwim->id);

            return redirect()->route('indikator-takwim.index')
                ->with('success', 'Indikator berhasil diperbarui.');
        } catch (Exception $e) {
            \Log::error('Update error: ' . $e->getMessage());
            return back()->withInput()->with('error', 'Error: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(IndikatorTakwim $indikatorTakwim)
    {
        $indikatorTakwim->delete();
        
        return redirect()->route('indikator-takwim.index')
            ->with('success', 'Indikator berhasil dihapus.');
    }
}
