<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Anggota;
use App\Models\Kegiatan;
use App\Models\Mutasi;
use App\Models\AbsensiKegiatan;
use App\Models\Takwim;
use Carbon\Carbon;
use DB;

class DashboardController extends Controller
{
    public function index2()
    {
        // Statistik Jenjang Anggota
        $jenjangStats = DB::table('tbl_anggota')
            ->select('jenjang_anggota', DB::raw('count(*) as total'))
            ->where('status_keanggotaan', 'Aktif')
            ->groupBy('jenjang_anggota')
            ->orderBy('total', 'desc')
            ->get();
        
        // Total Anggota
        $totalAnggota = $jenjangStats->sum('total');
        
        // Anggota Aktif (sama dengan total karena sudah filter aktif)
        $anggotaAktif = $totalAnggota;
        
        // Mutasi Pending
        $mutasiPending = Mutasi::where('status_mutasi', 'Pending')->count();
        
        // Kegiatan Bulan Ini (nanti bisa disesuaikan ketika ada tabel kegiatan)
        $kegiatanBulanIni = 0;
        
        // Recent Activities (ambil mutasi terbaru)
        $recentActivities = Mutasi::with(['anggota'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
        
        return view('dashboard', compact(
            'jenjangStats', 
            'totalAnggota', 
            'anggotaAktif', 
            'mutasiPending', 
            'kegiatanBulanIni',
            'recentActivities'
        ));
    }
    public function index()
    {
        // Cache statistik utama untuk 5 menit
        $stats = cache()->remember('dashboard_stats', 300, function() {
            return [
                'total_anggota' => Anggota::count(),
                'anggota_aktif' => Anggota::where('status_keanggotaan', 'Aktif')->count(),
                'anggota_tidak_aktif' => Anggota::where('status_keanggotaan', 'Tidak Aktif')->count(),
                'anggota_diberhentikan' => Anggota::where('status_keanggotaan', 'Diberhentikan')->count(),
                'anggota_pindah' => Anggota::where('status_keanggotaan', 'Pindah')->count(),
                'kegiatan_bulan_ini' => Kegiatan::whereMonth('tanggal_mulai', Carbon::now()->month)
                                               ->whereYear('tanggal_mulai', Carbon::now()->year)
                                               ->count(),
                'takwim_total' => Takwim::count()
            ];
        });

         // Cache statistik jenjang untuk 10 menit
        $jenjangStats = cache()->remember('dashboard_jenjang_stats', 600, function() {
            return DB::table('tbl_anggota')
                ->join('tbl_ref_jenjang', 'tbl_anggota.ref_jenjang_id', '=', 'tbl_ref_jenjang.ref_jenjang_id')
                ->select('tbl_ref_jenjang.jenjang', DB::raw('count(*) as total'))
                ->where('tbl_anggota.status_keanggotaan', 'Aktif')
                ->groupBy('tbl_ref_jenjang.jenjang')
                ->orderByRaw("CASE 
                    WHEN tbl_ref_jenjang.jenjang = 'Pendukung' THEN 1
                    WHEN tbl_ref_jenjang.jenjang = 'Penggerak' THEN 2
                    WHEN tbl_ref_jenjang.jenjang = 'Madya' THEN 3
                    WHEN tbl_ref_jenjang.jenjang = 'Dewasa' THEN 4
                    WHEN tbl_ref_jenjang.jenjang = 'Utama' THEN 5
                    ELSE 6
                END")
                ->get();
        });

                   // Total Anggota
        $totalAnggota = $jenjangStats->sum('total');
        

        // Cache anggota per DPC untuk 15 menit - Query yang sangat berat
        $anggota_per_dpc = cache()->remember('dashboard_anggota_per_dpc', 900, function() {
            $allDpcs = DB::table('tbl_dpc')
                ->where('is_active', true)
                ->orderBy('nama_dpc')
                ->get();

            $anggota_per_dpc = collect();
            
            foreach ($allDpcs as $dpc) {
                $anggotaData = DB::table('tbl_anggota')
                    ->join('tbl_ref_jenjang', 'tbl_anggota.ref_jenjang_id', '=', 'tbl_ref_jenjang.ref_jenjang_id')
                    ->selectRaw('
                        SUM(CASE WHEN tbl_ref_jenjang.jenjang = "Pendukung" THEN 1 ELSE 0 END) as pendukung,
                        SUM(CASE WHEN tbl_ref_jenjang.jenjang = "Penggerak" THEN 1 ELSE 0 END) as penggerak,
                        SUM(CASE WHEN tbl_ref_jenjang.jenjang = "Madya" THEN 1 ELSE 0 END) as madya,
                        SUM(CASE WHEN tbl_ref_jenjang.jenjang = "Dewasa" THEN 1 ELSE 0 END) as dewasa,
                        SUM(CASE WHEN tbl_ref_jenjang.jenjang = "Utama" THEN 1 ELSE 0 END) as utama,
                        COUNT(*) as total
                    ')
                    ->where('tbl_anggota.dpc_id', $dpc->dpc_id)
                    ->where('tbl_anggota.status_keanggotaan', 'Aktif')
                    ->first();

                $anggota_per_dpc->push([
                    'kecamatan' => $dpc->nama_dpc,
                    'pendukung' => $anggotaData->pendukung ?? 0,
                    'penggerak' => $anggotaData->penggerak ?? 0,
                    'madya' => $anggotaData->madya ?? 0,
                    'dewasa' => $anggotaData->dewasa ?? 0,
                    'utama' => $anggotaData->utama ?? 0,
                    'total' => $anggotaData->total ?? 0
                ]);
            }
            
            return $anggota_per_dpc;
        });

        // Cache kegiatan terbaru untuk 2 menit
        $kegiatan_terbaru = cache()->remember('dashboard_kegiatan_terbaru', 120, function() {
            return Kegiatan::with(['absensiKegiatan', 'jenisKegiatan'])
                           ->withCount(['absensiKegiatan as absensi_count'])
                           ->orderBy('created_at', 'desc')
                           ->take(5)
                           ->get();
        });

        // Cache mutasi terbaru untuk 2 menit
        $mutasi_terbaru = cache()->remember('dashboard_mutasi_terbaru', 120, function() {
            return Mutasi::with(['anggota'])
                        ->orderBy('created_at', 'desc')
                        ->take(5)
                        ->get();
        });

        // Cache pertumbuhan anggota untuk 30 menit
        $pertumbuhan_anggota = cache()->remember('dashboard_pertumbuhan_anggota', 1800, function() {
            $pertumbuhan_anggota = collect();
            for ($i = 11; $i >= 0; $i--) {
                $tanggal = Carbon::now()->subMonths($i);
                $jumlah = Anggota::whereYear('tanggal_bergabung', $tanggal->year)
                                ->whereMonth('tanggal_bergabung', $tanggal->month)
                                ->count();
                
                $pertumbuhan_anggota->push([
                    'bulan' => $tanggal->format('M Y'),
                    'jumlah' => $jumlah
                ]);
            }
            return $pertumbuhan_anggota;
        });

        // Cache statistik kegiatan per jenis untuk 10 menit
        $kegiatan_per_jenis = cache()->remember('dashboard_kegiatan_per_jenis', 600, function() {
            return Kegiatan::join('tbl_ref_jenis_kegiatan', 'tbl_kegiatan.ref_jenis_kegiatan_id', '=', 'tbl_ref_jenis_kegiatan.ref_jenis_kegiatan_id')
                         ->selectRaw('tbl_ref_jenis_kegiatan.nama_jenis_kegiatan as jenis_kegiatan, count(*) as total')
                         ->groupBy('tbl_ref_jenis_kegiatan.nama_jenis_kegiatan')
                         ->get();
        });

        // Cache tingkat kehadiran untuk 5 menit
        $tingkat_kehadiran = cache()->remember('dashboard_tingkat_kehadiran', 300, function() {
            $total_kegiatan = Kegiatan::where('status_kegiatan', 'Completed')->count();
            $total_absensi = AbsensiKegiatan::whereHas('kegiatan', function($query) {
                $query->where('status_kegiatan', 'Completed');
            })->count();
            
            $total_hadir = AbsensiKegiatan::where('status_kehadiran', 'Hadir')
                                         ->whereHas('kegiatan', function($query) {
                                             $query->where('status_kegiatan', 'Completed');
                                         })->count();

            return $total_absensi > 0 ? round(($total_hadir / $total_absensi) * 100, 2) : 0;
        });

        return view('dashboard', compact(
            'stats',
            'anggota_per_dpc', 
            'kegiatan_terbaru',
            'mutasi_terbaru',
            'pertumbuhan_anggota',
            'kegiatan_per_jenis',
            'tingkat_kehadiran',
            'jenjangStats',
            'totalAnggota'
        ));
    }

    public function getStatistikChart(Request $request)
    {
        $type = $request->get('type', 'provinsi');
        
        switch ($type) {
            case 'provinsi':
                $data = Anggota::selectRaw('provinsi as label, count(*) as value')
                              ->groupBy('provinsi')
                              ->orderBy('value', 'desc')
                              ->take(10)
                              ->get();
                break;
                
            case 'jenjang':
                $data = Anggota::join('tbl_ref_jenjang', 'tbl_anggota.ref_jenjang_id', '=', 'tbl_ref_jenjang.ref_jenjang_id')
                              ->selectRaw('tbl_ref_jenjang.jenjang as label, count(*) as value')
                              ->groupBy('tbl_ref_jenjang.jenjang')
                              ->orderBy('value', 'desc')
                              ->get();
                break;
                
            case 'umur':
                $data = Anggota::selectRaw("
                    CASE 
                        WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) < 25 THEN 'Dibawah 25'
                        WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) BETWEEN 25 AND 35 THEN '25-35'
                        WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) BETWEEN 36 AND 45 THEN '36-45'
                        WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) BETWEEN 46 AND 55 THEN '46-55'
                        ELSE 'Diatas 55'
                    END as label,
                    count(*) as value
                ")
                ->groupBy('label')
                ->orderBy('value', 'desc')
                ->get();
                break;
                
            case 'muzaki':
                $data = Anggota::selectRaw('muzaki_mustahiq as label, count(*) as value')
                              ->whereNotNull('muzaki_mustahiq')
                              ->groupBy('muzaki_mustahiq')
                              ->get();
                break;
                
            default:
                $data = collect();
        }
        
        return response()->json($data);
    }

    public function getStatistikTrend(Request $request)
    {
        $period = $request->get('period', '12'); // months
        $type = $request->get('type', 'anggota_baru');
        
        $data = collect();
        
        switch ($type) {
            case 'anggota_baru':
                for ($i = (int)$period - 1; $i >= 0; $i--) {
                    $tanggal = Carbon::now()->subMonths($i);
                    $jumlah = Anggota::whereYear('tanggal_bergabung', $tanggal->year)
                                    ->whereMonth('tanggal_bergabung', $tanggal->month)
                                    ->count();
                    
                    $data->push([
                        'period' => $tanggal->format('M Y'),
                        'value' => $jumlah
                    ]);
                }
                break;
                
            case 'kegiatan':
                for ($i = (int)$period - 1; $i >= 0; $i--) {
                    $tanggal = Carbon::now()->subMonths($i);
                    $jumlah = Kegiatan::whereYear('tanggal_mulai', $tanggal->year)
                                     ->whereMonth('tanggal_mulai', $tanggal->month)
                                     ->count();
                    
                    $data->push([
                        'period' => $tanggal->format('M Y'),
                        'value' => $jumlah
                    ]);
                }
                break;
                
            case 'mutasi':
                for ($i = (int)$period - 1; $i >= 0; $i--) {
                    $tanggal = Carbon::now()->subMonths($i);
                    $jumlah = Mutasi::whereYear('tanggal_mutasi', $tanggal->year)
                                   ->whereMonth('tanggal_mutasi', $tanggal->month)
                                   ->count();
                    
                    $data->push([
                        'period' => $tanggal->format('M Y'),
                        'value' => $jumlah
                    ]);
                }
                break;
        }
        
        return response()->json($data);
    }
}