<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Kegiatan;
use App\Models\Anggota;
use App\Models\AbsensiKegiatan;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class CheckinController extends Controller
{
    /**
     * Display check-in page for a specific kegiatan
     */
    public function index(Request $request, $kegiatanId)
    {
        try {
            // Decrypt kegiatan ID if needed
            $kegiatan = Kegiatan::findOrFail($kegiatanId);
            
            // Check if kegiatan is active (not expired)
            $now = Carbon::now();
            $kegiatanEnd = Carbon::parse($kegiatan->tanggal_mulai . ' ' . $kegiatan->waktu_selesai);
            
            if ($now->gt($kegiatanEnd)) {
                return view('checkin.expired', compact('kegiatan'));
            }
            
            return view('checkin.index', compact('kegiatan'));
            
        } catch (\Exception $e) {
            return view('checkin.error', ['message' => 'Kegiatan tidak ditemukan atau tidak valid.']);
        }
    }
    
    /**
     * Search anggota for check-in
     */
    public function searchAnggota(Request $request, $kegiatanId)
    {
        try {
            $searchTerm = $request->input('search');
            
            if (empty($searchTerm)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Masukkan NIA atau nama anggota!'
                ]);
            }
            
            // Search anggota by nama (NIA is encrypted, so we search by nama first)
            $anggota = Anggota::where('nama_lengkap', 'LIKE', "%{$searchTerm}%")
            ->where('status_keanggotaan', 'Aktif')
            ->limit(10)
            ->get();
            
            // If no results found by nama, try to search by NIA if it looks like a NIA format
            if ($anggota->isEmpty() && preg_match('/^PKS-\d+-\d+$/', $searchTerm)) {
                // For NIA search, we need to get all active members and filter by decrypted NIA
                $allAnggota = Anggota::where('status_keanggotaan', 'Aktif')->get();
                $anggota = $allAnggota->filter(function($item) use ($searchTerm) {
                    return $item->nomor_identitas_anggota === $searchTerm;
                })->take(10);
            }
            
            return response()->json([
                'success' => true,
                'data' => $anggota->map(function($item) {
                    return [
                        'id' => $item->id,
                        'nia' => $item->nomor_identitas_anggota,
                        'nama' => $item->nama_lengkap,
                        'jenjang' => $item->refJenjang->nama_jenjang ?? '-',
                        'status' => $item->status_keanggotaan
                    ];
                })
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat mencari anggota: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * Process check-in for anggota
     */
    public function checkIn(Request $request, $kegiatanId)
    {
        try {
            $anggotaId = $request->input('anggota_id');
            
            if (empty($anggotaId)) {
                return response()->json([
                    'success' => false,
                    'message' => 'ID anggota tidak valid!'
                ]);
            }
            
            // Check if anggota exists and is active
            $anggota = Anggota::where('id', $anggotaId)
                             ->where('status_keanggotaan', 'Aktif')
                             ->first();
            
            if (!$anggota) {
                return response()->json([
                    'success' => false,
                    'message' => 'Anggota tidak ditemukan atau tidak aktif!'
                ]);
            }
            
            // Check if already checked in
            $existingCheckin = AbsensiKegiatan::where('kegiatan_id', $kegiatanId)
                                            ->where('anggota_id', $anggotaId)
                                            ->first();
            
            if ($existingCheckin) {
                return response()->json([
                    'success' => false,
                    'message' => 'Anggota sudah check-in sebelumnya!'
                ]);
            }
            
            // Create check-in record
            DB::beginTransaction();
            
            AbsensiKegiatan::create([
                'kegiatan_id' => $kegiatanId,
                'anggota_id' => $anggotaId,
                'waktu_checkin' => Carbon::now(),
                'status' => 'hadir',
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now()
            ]);
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => $anggota->nama_lengkap . ' berhasil check-in!',
                'data' => [
                    'nama' => $anggota->nama_lengkap,
                    'nia' => $anggota->nia,
                    'waktu_checkin' => Carbon::now()->format('H:i:s')
                ]
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat check-in: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * Get check-in statistics for a kegiatan
     */
    public function getStats($kegiatanId)
    {
        try {
            $kegiatan = Kegiatan::findOrFail($kegiatanId);
            
            $totalPeserta = $kegiatan->partisipanKegiatan()->count();
            $sudahCheckin = AbsensiKegiatan::where('kegiatan_id', $kegiatanId)
                                         ->where('status', 'hadir')
                                         ->count();
            $belumCheckin = $totalPeserta - $sudahCheckin;
            $persentase = $totalPeserta > 0 ? round(($sudahCheckin / $totalPeserta) * 100, 1) : 0;
            
            return response()->json([
                'success' => true,
                'data' => [
                    'total_peserta' => $totalPeserta,
                    'sudah_checkin' => $sudahCheckin,
                    'belum_checkin' => $belumCheckin,
                    'persentase' => $persentase
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat mengambil statistik: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * Generate QR Code for kegiatan
     */
    public function generateQR($kegiatanId)
    {
        try {
            $kegiatan = Kegiatan::findOrFail($kegiatanId);
            
            // Generate QR code data
            $qrData = [
                'kegiatan_id' => $kegiatanId,
                'nama' => $kegiatan->nama_kegiatan,
                'tanggal' => $kegiatan->tanggal_mulai,
                'expires_at' => Carbon::parse($kegiatan->tanggal_mulai . ' ' . $kegiatan->waktu_selesai)->addHour()
            ];
            
            $qrString = base64_encode(json_encode($qrData));
            $checkinUrl = route('checkin.index', $kegiatanId);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'qr_string' => $qrString,
                    'checkin_url' => $checkinUrl,
                    'expires_at' => $qrData['expires_at']->format('Y-m-d H:i:s')
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat generate QR code: ' . $e->getMessage()
            ]);
        }
    }
}
