<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Anggota;
use App\Models\AnggotaJenjangHistory;
use App\Models\RefJenjang;
use App\Models\RefGrupUpa;
use App\Models\Dpc;
use App\Models\Dpd;
use App\Models\Upa;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AnggotaKeanggotaanController extends Controller
{
    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $anggota = Anggota::with([
            'refJenjang',
            'refGrupUpa', 
            'dpc',
            'dpd',
            'upa',
            'jenjangHistory.refJenjang',
            'jenjangHistory.refGrupUpa',
            'jenjangHistory.dpc',
            'jenjangHistory.dpd',
            'jenjangHistory.upa',
            'jenjangHistory.user'
        ])->findOrFail($id);

        $history = $anggota->jenjangHistory()
            ->with(['refJenjang', 'refGrupUpa', 'dpc', 'dpd', 'upa', 'user'])
            ->latest()
            ->paginate(10);

        return view('anggota.keanggotaan.show', compact('anggota', 'history'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $anggota = Anggota::with([
            'refJenjang',
            'refGrupUpa', 
            'dpc',
            'dpd',
            'upa'
        ])->findOrFail($id);

        // Get dropdown data - hanya jenjang standar yang aktif
        $jenjangs = RefJenjang::whereIn('jenjang', ['Pendukung', 'Penggerak', 'Madya', 'Dewasa', 'Utama'])
            ->where('is_active', true)
            ->orderByRaw("FIELD(jenjang, 'Pendukung', 'Penggerak', 'Madya', 'Dewasa', 'Utama')")
            ->get();
        $grupUpas = RefGrupUpa::orderBy('nama_grup_upa')->get();
        $dpcs = Dpc::orderBy('nama_dpc')->get();
        $dpds = Dpd::orderBy('nama_dpd')->get();
        $upas = Upa::orderBy('nama_upa')->get();

        return view('anggota.keanggotaan.edit', compact(
            'anggota', 
            'jenjangs', 
            'grupUpas', 
            'dpcs', 
            'dpds', 
            'upas'
        ));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'ref_jenjang_id' => 'nullable|exists:tbl_ref_jenjang,ref_jenjang_id',
            'ref_grup_upa_id' => 'nullable|exists:tbl_ref_grup_upa,ref_grup_upa_id',
            'dpc_id' => 'nullable|exists:tbl_dpc,dpc_id',
            'dpd_id' => 'nullable|exists:tbl_dpd,dpd_id',
            'upa_id' => 'nullable|exists:tbl_upa,upa_id',
            'status_keanggotaan' => 'required|in:Aktif,Tidak Aktif,Diberhentikan,Pindah',
            'tanggal_perubahan' => 'required|date',
            'keterangan' => 'nullable|string|max:1000'
        ]);

        $anggota = Anggota::findOrFail($id);

        DB::beginTransaction();
        try {
            // Simpan data lama ke history
            $anggota->saveToHistory($request->keterangan);

            // Update data baru
            $anggota->update([
                'ref_jenjang_id' => $request->ref_jenjang_id,
                'ref_grup_upa_id' => $request->ref_grup_upa_id,
                'dpc_id' => $request->dpc_id,
                'dpd_id' => $request->dpd_id,
                'upa_id' => $request->upa_id,
                'status_keanggotaan' => $request->status_keanggotaan,
            ]);

            DB::commit();

            return redirect()
                ->route('anggota.keanggotaan.show', $anggota->id)
                ->with('success', 'Data keanggotaan berhasil diperbarui dan tersimpan dalam history.');

        } catch (\Exception $e) {
            DB::rollback();
            
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Terjadi kesalahan saat memperbarui data: ' . $e->getMessage());
        }
    }

    /**
     * Get history data for AJAX request
     */
    public function getHistory($id, Request $request)
    {
        $anggota = Anggota::findOrFail($id);
        
        $query = $anggota->jenjangHistory()
            ->with(['refJenjang', 'refGrupUpa', 'dpc', 'dpd', 'upa', 'user']);

        // Filter by date range if provided
        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('tanggal_perubahan', [
                $request->start_date,
                $request->end_date
            ]);
        }

        $history = $query->latest()->paginate(10);

        if ($request->ajax()) {
            return response()->json([
                'history' => $history,
                'html' => view('anggota.keanggotaan.partials.history-table', compact('history'))->render()
            ]);
        }

        return view('anggota.keanggotaan.partials.history-table', compact('history'));
    }

    /**
     * Export history to PDF
     */
    public function exportHistory($id)
    {
        $anggota = Anggota::with(['refJenjang', 'refGrupUpa', 'dpc', 'dpd', 'upa'])->findOrFail($id);
        
        $history = $anggota->jenjangHistory()
            ->with(['refJenjang', 'refGrupUpa', 'dpc', 'dpd', 'upa', 'user'])
            ->latest()
            ->get();

        $pdf = \PDF::loadView('anggota.keanggotaan.export.history-pdf', compact('anggota', 'history'));
        
        return $pdf->download("history_keanggotaan_{$anggota->nama_lengkap}_{$anggota->nomor_identitas_anggota}.pdf");
    }

    /**
     * Get UPA options based on Grup UPA
     */
    public function getUpaOptions(Request $request)
    {
        $grupUpaId = $request->grup_upa_id;
        
        if (!$grupUpaId) {
            return response()->json([]);
        }

        $upas = Upa::where('ref_grup_upa_id', $grupUpaId)
            ->orderBy('nama_upa')
            ->get(['upa_id', 'nama_upa']);

        return response()->json($upas);
    }
}
