<?php

namespace App\Http\Controllers;

use App\Models\Anggota;
use App\Models\StrukturOrganisasi;
use App\Models\RefKeahlian;
use App\Models\RefGrupUpa;
use App\Models\AnggotaAktivitas;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\AnggotaExport;
use App\Exports\AnggotaTemplateExport;
use App\Imports\AnggotaImport;
use App\Helpers\CacheHelper;

class AnggotaController extends Controller
{
    /**
     * Clear cache yang terkait dengan data anggota
     */
    private function clearAnggotaCache()
    {
        // Clear cache master data
        CacheHelper::clearMasterDataCache();
        
        // Clear cache hasil query (pattern matching)
        CacheHelper::clearPatternCache('*anggota_index_*');
    }
    
    /**
     * Resolve anggota dari slug
     */
    private function resolveAnggota($slug)
    {
        $anggota = Anggota::findBySlug($slug);
        
        if (!$anggota) {
            abort(404, 'Anggota tidak ditemukan');
        }
        
        return $anggota;
    }
    public function index(Request $request)
    {
        // Cache key berdasarkan parameter request
        $cacheKey = 'anggota_index_' . md5(serialize($request->all()));
        
        // Cache hasil query untuk 5 menit
        $anggota = cache()->remember($cacheKey, 300, function() use ($request) {
            // Optimasi query dengan select specific fields dan eager loading
            $query = Anggota::select([
                'id', 'nomor_identitas_anggota', 'nama_lengkap', 'jenis_kelamin', 
                'kota', 'ref_jenjang_id', 'upa_id', 'dpc_id', 'dpd_id', 
                'status_keanggotaan', 'email', 'foto', 'created_at'
            ])->with([
                'refJenjang:ref_jenjang_id,jenjang',
                'upa:upa_id,nama_upa',
                'dpc:dpc_id,nama_dpc',
                'dpd:dpd_id,nama_dpd',
                'keahlian:ref_keahlian_id,nama_keahlian'
            ]);
            
            // Filter berdasarkan pencarian
            if ($request->has('search') && $request->search) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('nama_lengkap', 'like', "%{$search}%")
                      ->orWhere('nomor_identitas_anggota', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%");
                });
            }
            
            // Filter berdasarkan status
            if ($request->has('status') && $request->status) {
                $query->where('status_keanggotaan', $request->status);
            }
            
            // Filter berdasarkan jenis kelamin
            if ($request->has('jenis_kelamin') && $request->jenis_kelamin) {
                $query->where('jenis_kelamin', $request->jenis_kelamin);
            }
            
            // Filter berdasarkan jenjang
            if ($request->has('jenjang') && $request->jenjang) {
                $query->where('ref_jenjang_id', $request->jenjang);
            }
            
            // Filter berdasarkan provinsi
            if ($request->has('provinsi') && $request->provinsi) {
                $query->where('provinsi', $request->provinsi);
            }
            
            // Filter berdasarkan DPD
            if ($request->has('dpd') && $request->dpd) {
                $query->where('dpd_id', $request->dpd);
            }
            
            // Filter berdasarkan DPC
            if ($request->has('dpc') && $request->dpc) {
                $query->where('dpc_id', $request->dpc);
            }
            
            return $query->orderBy('created_at', 'desc')->paginate(10);
        });
        
        // Get cached master data untuk dropdown
        $masterData = CacheHelper::getMasterData();
        
        return view('anggota.index', array_merge(
            ['anggota' => $anggota],
            $masterData
        ));
    }

    /**
     * Menampilkan data anggota aktif dengan filter otomatis
     */
    public function indexAktif(Request $request)
    {
        // Set filter status otomatis ke 'Aktif'
        $request->merge(['status' => 'Aktif']);
        
        // Cache key berdasarkan parameter request
        $cacheKey = 'anggota_aktif_' . md5(serialize($request->all()));
        
        // Cache hasil query untuk 5 menit
        $anggota = cache()->remember($cacheKey, 300, function() use ($request) {
            // Optimasi query dengan select specific fields dan eager loading
            $query = Anggota::select([
                'id', 'nomor_identitas_anggota', 'nama_lengkap', 'jenis_kelamin', 
                'kota', 'ref_jenjang_id', 'upa_id', 'dpc_id', 'dpd_id', 
                'status_keanggotaan', 'email', 'foto', 'created_at'
            ])->with([
                'refJenjang:ref_jenjang_id,jenjang',
                'upa:upa_id,nama_upa',
                'dpc:dpc_id,nama_dpc',
                'dpd:dpd_id,nama_dpd',
                'keahlian:ref_keahlian_id,nama_keahlian'
            ])->where('status_keanggotaan', 'Aktif');
            
            // Filter berdasarkan pencarian
            if ($request->has('search') && $request->search) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('nama_lengkap', 'like', "%{$search}%")
                      ->orWhere('nomor_identitas_anggota', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%");
                });
            }
            
            // Filter berdasarkan jenis kelamin
            if ($request->has('jenis_kelamin') && $request->jenis_kelamin) {
                $query->where('jenis_kelamin', $request->jenis_kelamin);
            }
            
            // Filter berdasarkan jenjang
            if ($request->has('jenjang') && $request->jenjang) {
                $query->where('ref_jenjang_id', $request->jenjang);
            }
            
            // Filter berdasarkan provinsi
            if ($request->has('provinsi') && $request->provinsi) {
                $query->where('provinsi', $request->provinsi);
            }
            
            // Filter berdasarkan DPD
            if ($request->has('dpd') && $request->dpd) {
                $query->where('dpd_id', $request->dpd);
            }
            
            // Filter berdasarkan DPC
            if ($request->has('dpc') && $request->dpc) {
                $query->where('dpc_id', $request->dpc);
            }
            
            return $query->orderBy('created_at', 'desc')->paginate(10);
        });
        
        // Get cached master data untuk dropdown
        $masterData = CacheHelper::getMasterData();
        
        return view('anggota.index', array_merge(
            [
                'anggota' => $anggota,
                'filterStatus' => 'Aktif',
                'pageTitle' => 'Data Anggota Aktif'
            ],
            $masterData
        ));
    }

    /**
     * Menampilkan data anggota non aktif dengan filter otomatis
     */
    public function indexNonAktif(Request $request)
    {
        // Set filter status otomatis ke 'Tidak Aktif'
        $request->merge(['status' => 'Tidak Aktif']);
        
        // Cache key berdasarkan parameter request
        $cacheKey = 'anggota_non_aktif_' . md5(serialize($request->all()));
        
        // Cache hasil query untuk 5 menit
        $anggota = cache()->remember($cacheKey, 300, function() use ($request) {
            // Optimasi query dengan select specific fields dan eager loading
            $query = Anggota::select([
                'id', 'nomor_identitas_anggota', 'nama_lengkap', 'jenis_kelamin', 
                'kota', 'ref_jenjang_id', 'upa_id', 'dpc_id', 'dpd_id', 
                'status_keanggotaan', 'email', 'foto', 'created_at'
            ])->with([
                'refJenjang:ref_jenjang_id,jenjang',
                'upa:upa_id,nama_upa',
                'dpc:dpc_id,nama_dpc',
                'dpd:dpd_id,nama_dpd',
                'keahlian:ref_keahlian_id,nama_keahlian'
            ])->where('status_keanggotaan', 'Tidak Aktif');
            
            // Filter berdasarkan pencarian
            if ($request->has('search') && $request->search) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('nama_lengkap', 'like', "%{$search}%")
                      ->orWhere('nomor_identitas_anggota', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%");
                });
            }
            
            // Filter berdasarkan jenis kelamin
            if ($request->has('jenis_kelamin') && $request->jenis_kelamin) {
                $query->where('jenis_kelamin', $request->jenis_kelamin);
            }
            
            // Filter berdasarkan jenjang
            if ($request->has('jenjang') && $request->jenjang) {
                $query->where('ref_jenjang_id', $request->jenjang);
            }
            
            // Filter berdasarkan provinsi
            if ($request->has('provinsi') && $request->provinsi) {
                $query->where('provinsi', $request->provinsi);
            }
            
            // Filter berdasarkan DPD
            if ($request->has('dpd') && $request->dpd) {
                $query->where('dpd_id', $request->dpd);
            }
            
            // Filter berdasarkan DPC
            if ($request->has('dpc') && $request->dpc) {
                $query->where('dpc_id', $request->dpc);
            }
            
            return $query->orderBy('created_at', 'desc')->paginate(10);
        });
        
        // Get cached master data untuk dropdown
        $masterData = CacheHelper::getMasterData();
        
        return view('anggota.index', array_merge(
            [
                'anggota' => $anggota,
                'filterStatus' => 'Tidak Aktif',
                'pageTitle' => 'Data Anggota Non Aktif'
            ],
            $masterData
        ));
    }

    public function create()
    {
        $keahlianList = RefKeahlian::active()->get();
        $grupUpaList = RefGrupUpa::active()->get();
        return view('anggota.create', compact('keahlianList', 'grupUpaList'));
    }

    /**
     * Get DPC by DPD ID
     */
    public function getDpcByDpd(Request $request)
    {
        $dpdId = $request->get('dpd_id');
        
        if (!$dpdId) {
            return response()->json([]);
        }
        
        $dpcList = \App\Models\Dpc::where('dpd_id', $dpdId)
                                   ->where('is_active', true)
                                   ->orderBy('nama_dpc')
                                   ->get(['dpc_id', 'nama_dpc']);
        
        return response()->json($dpcList);
    }

    /**
     * Get UPA by DPC ID
     */
    public function getUpaByDpc(Request $request)
    {
        $dpcId = $request->get('dpc_id');
        
        if (!$dpcId) {
            return response()->json([]);
        }
        
        $upaList = \App\Models\Upa::where('dpc_id', $dpcId)
                                  ->where('status', 'Aktif')
                                  ->orderBy('nama_upa')
                                  ->get(['upa_id', 'nama_upa']);
        
        return response()->json($upaList);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nomor_identitas_anggota' => 'required|string|unique:tbl_anggota',
            'nama_lengkap' => 'required|string|max:255',
            'tempat_lahir' => 'required|string|max:255',
            'tanggal_lahir' => 'required|date',
            'jenis_kelamin' => 'required|in:L,P',
            'alamat_jalan' => 'required|string',
            'rt' => 'required|string|max:5',
            'rw' => 'required|string|max:5',
            'kelurahan' => 'required|string|max:255',
            'kecamatan' => 'required|string|max:255',
            'kota' => 'required|string|max:255',
            'provinsi' => 'required|string|max:255',
            'kode_pos' => 'nullable|string|max:5|regex:/^[0-9]{5}$/',
            'no_telepon' => 'nullable|string|max:15',
            'email' => 'nullable|email|unique:tbl_anggota',
            'ref_jenjang_id' => 'required|exists:tbl_ref_jenjang,ref_jenjang_id',
            'dpc_id' => 'nullable|exists:tbl_dpc,dpc_id',
            'dpd_id' => 'nullable|exists:tbl_dpd,dpd_id',
            'upa_id' => 'nullable|exists:tbl_upa,upa_id',
            'tanggal_bergabung' => 'required|date',
            'penghasilan_per_bulan' => 'nullable|numeric|min:0',
            'muzaki_mustahiq' => 'nullable|in:Muzaki,Mustahiq',
            'nomor_identitas_pasangan' => 'nullable|string|max:255',
            'catatan_profil' => 'nullable|string',
            'keahlian_lainnya' => 'nullable|string|max:1000',
            'keahlian_lainnya_check' => 'nullable|boolean',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:4096',
            'keahlian_ids' => 'nullable|array',
            'keahlian_ids.*' => 'exists:tbl_ref_keahlian,ref_keahlian_id',
            'aktivitas_luar_ids' => 'nullable|array',
            'aktivitas_luar_ids.*' => 'exists:aktivitas_luars,id',
            'aktivitas_luar_lainnya_check' => 'nullable|boolean',
            'aktivitas_luar_lainnya_text' => 'nullable|string|max:1000'
        ]);

        // Handle foto upload
        if ($request->hasFile('foto')) {
            $path = $request->file('foto')->store('anggota-photos', 'public');
            $validated['foto'] = $path;
        }

        // Handle keahlian lainnya checkbox
        if (!$request->has('keahlian_lainnya_check') || !$request->keahlian_lainnya_check) {
            $validated['keahlian_lainnya'] = null;
        }

        $anggota = Anggota::create($validated);

        // Sync keahlian
        if ($request->has('keahlian_ids')) {
            $anggota->keahlian()->sync($request->keahlian_ids);
        }

        // Simpan aktivitas luar
        if ($request->has('aktivitas_luar_ids') && is_array($request->aktivitas_luar_ids)) {
            $anggota->aktivitasLuar()->sync($request->aktivitas_luar_ids);
        }
        
        // Simpan aktivitas custom (lainnya)
        if ($request->has('aktivitas_luar_lainnya_check') && $request->aktivitas_luar_lainnya_check && $request->aktivitas_luar_lainnya_text) {
            // Hapus aktivitas custom lama
            $anggota->aktivitas()->delete();
            
            // Simpan aktivitas custom baru
            $anggota->aktivitas()->create([
                'uraian_aktivitas' => $request->aktivitas_luar_lainnya_text
            ]);
        } else {
            // Jika tidak ada aktivitas custom, hapus yang lama
            $anggota->aktivitas()->delete();
        }

        // Clear cache setelah data berubah
        $this->clearAnggotaCache();
        
        $slug = \App\Helpers\UrlHelper::generateAnggotaUrl($anggota->id, $anggota->nama_lengkap);
        return redirect()->route('anggota.show', $slug)
                        ->with('success', 'Data anggota berhasil ditambahkan!');
    }

    public function show($slug)
    {
        try {
            $anggota = $this->resolveAnggota($slug);
            
            // Cache key untuk anggota show
            $cacheKey = 'anggota_show_' . $anggota->id;
            
            // Load relationships dengan caching
            $anggota = cache()->remember($cacheKey, 300, function() use ($anggota) {
                $anggota->load([
                    'refJenjang',
                    'refGrupUpa',
                    'dpc',
                    'dpd',
                    'keahlian',
                    'aktivitas',
                    'aktivitasLuar',
                    'jenjangHistory' => function($query) {
                        $query->with(['refJenjang', 'refGrupUpa', 'dpc', 'dpd', 'upa', 'user'])
                              ->orderBy('tanggal_perubahan', 'desc')
                              ->limit(5); // Limit history untuk dashboard
                    },
                    'mutasi' => function($query) {
                        $query->orderBy('tanggal_mutasi', 'desc')->limit(10); // Limit mutasi
                    },
                    'riwayatPerpindahan' => function($query) {
                        $query->with(['upaAsal', 'upaTujuan', 'dpcAsal', 'dpcTujuan', 'pembimbingLama', 'pembimbingBaru'])
                              ->orderBy('tanggal_efektif', 'desc')
                              ->limit(10); // Limit riwayat perpindahan
                    },
                    'absensiKegiatan' => function($query) {
                        $query->with('kegiatan')->orderBy('created_at', 'desc')->limit(10); // Limit absensi
                    }
                ]);
                return $anggota;
            });

            return view('anggota.show', compact('anggota'));

        } catch (\Exception $e) {
            \Log::error('Error loading anggota show: ' . $e->getMessage());
            
            // Fallback: Load without relationships
            if (isset($anggota)) {
                return view('anggota.show', compact('anggota'));
            } else {
                abort(404, 'Anggota tidak ditemukan');
            }
        }
    }
    public function edit($slug)
    {
        $anggota = $this->resolveAnggota($slug);
        $keahlianList = RefKeahlian::active()->get();
        $grupUpaList = RefGrupUpa::active()->get();
        $anggota->load(['keahlian', 'aktivitas', 'aktivitasLuar', 'refGrupUpa', 'dpc', 'dpd']);
        return view('anggota.edit', compact('anggota', 'keahlianList', 'grupUpaList'));
    }

    public function update(Request $request, $slug)
    {
        $anggota = $this->resolveAnggota($slug);
        
        $validated = $request->validate([
            'nomor_identitas_anggota' => 'required|string|unique:tbl_anggota,nomor_identitas_anggota,' . $anggota->id,
            'nama_lengkap' => 'required|string|max:255',
            'tempat_lahir' => 'required|string|max:255',
            'tanggal_lahir' => 'required|date',
            'jenis_kelamin' => 'required|in:L,P',
            'alamat_jalan' => 'required|string',
            'rt' => 'required|string|max:5',
            'rw' => 'required|string|max:5',
            'kelurahan' => 'required|string|max:255',
            'kecamatan' => 'required|string|max:255',
            'kota' => 'required|string|max:255',
            'provinsi' => 'required|string|max:255',
            'kode_pos' => 'nullable|string|max:5|regex:/^[0-9]{5}$/',
            'no_telepon' => 'nullable|string|max:15',
            'email' => 'nullable|email|unique:tbl_anggota,email,' . $anggota->id,
            'ref_jenjang_id' => 'required|exists:tbl_ref_jenjang,ref_jenjang_id',
            'dpc_id' => 'nullable|exists:tbl_dpc,dpc_id',
            'dpd_id' => 'nullable|exists:tbl_dpd,dpd_id',
            'upa_id' => 'nullable|exists:tbl_upa,upa_id',
            'tanggal_bergabung' => 'required|date',
            'status_keanggotaan' => 'required|in:Aktif,Tidak Aktif,Diberhentikan,Pindah',
            'penghasilan_per_bulan' => 'nullable|numeric|min:0',
            'muzaki_mustahiq' => 'nullable|in:Muzaki,Mustahiq',
            'nomor_identitas_pasangan' => 'nullable|string|max:255',
            'catatan_profil' => 'nullable|string',
            'keahlian_lainnya' => 'nullable|string|max:1000',
            'keahlian_lainnya_check' => 'nullable|boolean',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:4096',
            'keahlian_ids' => 'nullable|array',
            'keahlian_ids.*' => 'exists:tbl_ref_keahlian,ref_keahlian_id',
            'aktivitas_luar_ids' => 'nullable|array',
            'aktivitas_luar_ids.*' => 'exists:aktivitas_luars,id',
            'aktivitas_luar_lainnya_check' => 'nullable|boolean',
            'aktivitas_luar_lainnya_text' => 'nullable|string|max:1000'
        ]);

        // Handle foto upload
        if ($request->hasFile('foto')) {
            // Delete old photo
            if ($anggota->foto) {
                Storage::disk('public')->delete($anggota->foto);
            }
            
            $path = $request->file('foto')->store('anggota-photos', 'public');
            $validated['foto'] = $path;
        }

        // Handle keahlian lainnya checkbox
        if (!$request->has('keahlian_lainnya_check') || !$request->keahlian_lainnya_check) {
            $validated['keahlian_lainnya'] = null;
        }

        $anggota->update($validated);

        // Sync keahlian
        if ($request->has('keahlian_ids')) {
            $anggota->keahlian()->sync($request->keahlian_ids);
        } else {
            $anggota->keahlian()->detach();
        }

        // Update aktivitas luar
        if ($request->has('aktivitas_luar_ids') && is_array($request->aktivitas_luar_ids)) {
            $anggota->aktivitasLuar()->sync($request->aktivitas_luar_ids);
        } else {
            $anggota->aktivitasLuar()->detach();
        }
        
        // Update aktivitas custom (lainnya)
        if ($request->has('aktivitas_luar_lainnya_check') && $request->aktivitas_luar_lainnya_check && $request->aktivitas_luar_lainnya_text) {
            // Hapus aktivitas custom lama
            $anggota->aktivitas()->delete();
            
            // Simpan aktivitas custom baru
            $anggota->aktivitas()->create([
                'uraian_aktivitas' => $request->aktivitas_luar_lainnya_text
            ]);
        } else {
            // Jika tidak ada aktivitas custom, hapus yang lama
            $anggota->aktivitas()->delete();
        }

        // Clear cache setelah data berubah
        $this->clearAnggotaCache();
        
        $slug = \App\Helpers\UrlHelper::generateAnggotaUrl($anggota->id, $anggota->nama_lengkap);
        return redirect()->route('anggota.show', $slug)
                        ->with('success', 'Data anggota berhasil diperbarui.');
    }

    public function destroy($slug)
    {
        $anggota = $this->resolveAnggota($slug);
        
        // Delete photo if exists
        if ($anggota->foto) {
            Storage::disk('public')->delete($anggota->foto);
        }
        
        $anggota->delete();
        
        // Clear cache setelah data berubah
        $this->clearAnggotaCache();

        return redirect()->route('anggota.index')
                        ->with('success', 'Data anggota berhasil dihapus.');
    }

    

    public function import2(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv'
        ]);

        try {
            Excel::import(new AnggotaImport, $request->file('file'));
            
            return redirect()->route('anggota.index')
                            ->with('success', 'Data anggota berhasil diimport.');
        } catch (\Exception $e) {
            return redirect()->back()
                            ->with('error', 'Gagal mengimport data: ' . $e->getMessage());
        }
    }

    public function statistik()
    {
        $stats = [
            'total_anggota' => Anggota::count(),
            'anggota_aktif' => Anggota::where('status_keanggotaan', 'Aktif')->count(),
            'anggota_tidak_aktif' => Anggota::where('status_keanggotaan', 'Tidak Aktif')->count(),
            'total_muzaki' => Anggota::where('muzaki_mustahiq', 'Muzaki')->count(),
            'total_mustahiq' => Anggota::where('muzaki_mustahiq', 'Mustahiq')->count(),
        ];

        // Statistik per provinsi
        $anggotaPerProvinsi = Anggota::selectRaw('provinsi, count(*) as total')
                                    ->groupBy('provinsi')
                                    ->orderBy('total', 'desc')
                                    ->get();

        // Statistik per jenjang
        $anggotaPerJenjang = Anggota::join('tbl_ref_jenjang', 'tbl_anggota.ref_jenjang_id', '=', 'tbl_ref_jenjang.ref_jenjang_id')
                                   ->selectRaw('tbl_ref_jenjang.jenjang, count(*) as total')
                                   ->groupBy('tbl_ref_jenjang.jenjang')
                                   ->orderBy('total', 'desc')
                                   ->get();

        // Statistik umur
        $anggotaPerUmur = Anggota::selectRaw("
                CASE 
                    WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) < 25 THEN 'Dibawah 25'
                    WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) BETWEEN 25 AND 35 THEN '25-35'
                    WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) BETWEEN 36 AND 45 THEN '36-45'
                    WHEN TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) BETWEEN 46 AND 55 THEN '46-55'
                    ELSE 'Diatas 55'
                END as kelompok_umur,
                count(*) as total
            ")
            ->groupBy('kelompok_umur')
            ->orderBy('total', 'desc')
            ->get();

        return view('anggota.statistik', compact(
            'stats', 'anggotaPerProvinsi', 'anggotaPerJenjang', 'anggotaPerUmur'
        ));
    }

    // ===== ALTERNATIVE: Simple Show Method (Fallback) =====
        public function showSimple(Anggota $anggota)
        {
            // Simple version without relationships for debugging
            return view('anggota.show-simple', compact('anggota'));
        }


    public function export(Request $request)
    {
        try {
            // Get filter parameters from request
            $filters = $request->only(['search', 'status', 'jenis_kelamin', 'jenjang', 'dpc', 'dpd', 'provinsi']);
            
            // Debug: Log filters untuk troubleshooting
            \Log::info('Export filters:', $filters);
            
            // Build query with filters
            $query = Anggota::with(['refJenjang', 'refGrupUpa', 'dpc', 'dpd']);
            
            // Apply filters
            if (!empty($filters['search'])) {
                $search = $filters['search'];
                $query->where(function($q) use ($search) {
                    $q->where('nama_lengkap', 'like', "%{$search}%")
                      ->orWhere('nomor_identitas_anggota', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%");
                });
            }
            
            if (!empty($filters['status'])) {
                $query->where('status_keanggotaan', $filters['status']);
            }
            
            if (!empty($filters['jenis_kelamin'])) {
                $query->where('jenis_kelamin', $filters['jenis_kelamin']);
            }
            
            if (!empty($filters['jenjang'])) {
                $query->where('ref_jenjang_id', $filters['jenjang']);
            }
            
            if (!empty($filters['dpc'])) {
                $query->where('dpc_id', $filters['dpc']);
            }
            
            if (!empty($filters['dpd'])) {
                $query->where('dpd_id', $filters['dpd']);
            }
            
            if (!empty($filters['provinsi'])) {
                $query->where('provinsi', $filters['provinsi']);
            }
            
            $anggota = $query->get();
            
            // Export Excel
            return Excel::download(new AnggotaExport($anggota), 'data_anggota_' . date('Y-m-d_H-i-s') . '.xlsx');
            
        } catch (\Exception $e) {
            \Log::error('Export error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Terjadi kesalahan saat export: ' . $e->getMessage());
        }
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv|max:10240', // 10MB max
            'action' => 'in:validate,import' // validate only or actual import
        ]);

        try {
            $file = $request->file('file');
            $action = $request->get('action', 'import');
            
            // Create import instance
            $import = new AnggotaImport();
            
            if ($action === 'validate') {
                // Validation only - don't save to database
                return $this->validateImportFile($file, $import);
            } else {
                // Actual import
                return $this->processImport($file, $import);
            }
            
        } catch (\Exception $e) {
            return redirect()->back()
                            ->with('error', 'Gagal mengimport data: ' . $e->getMessage());
        }
    }

    private function validateImportFile($file, AnggotaImport $import)
    {
        try {
            // Load the file to check for errors without importing
            $data = Excel::toArray($import, $file);
            
            if (empty($data[0])) {
                return redirect()->back()
                                ->with('error', 'File kosong atau tidak memiliki data.');
            }
            
            $rows = collect($data[0]);
            $headers = array_keys($rows->first());
            
            // Check required headers
            $requiredHeaders = [
                'nomor_identitas_anggota', 'nama_lengkap', 'tempat_lahir', 
                'tanggal_lahir', 'jenis_kelamin', 'alamat_jalan', 'rt', 'rw',
                'kelurahan', 'kecamatan', 'kota', 'provinsi', 'jenjang_anggota',
                'tanggal_bergabung'
            ];
            
            $missingHeaders = array_diff($requiredHeaders, $headers);
            if (!empty($missingHeaders)) {
                return redirect()->back()
                                ->with('error', 'Header yang hilang: ' . implode(', ', $missingHeaders));
            }
            
            // Preview first 5 rows
            $preview = $rows->take(5);
            
            return redirect()->back()
                            ->with('success', 'File valid! Ditemukan ' . $rows->count() . ' baris data.')
                            ->with('preview_data', $preview->toArray());
            
        } catch (\Exception $e) {
            return redirect()->back()
                            ->with('error', 'Error validasi: ' . $e->getMessage());
        }
    }

    private function processImport($file, AnggotaImport $import)
    {
        // Import with error handling
        Excel::import($import, $file);
        
        $summary = $import->getImportSummary();
        
        if ($summary['imported'] > 0) {
            $message = "Import berhasil! {$summary['imported']} anggota berhasil ditambahkan.";
            
            if ($summary['skipped'] > 0) {
                $message .= " {$summary['skipped']} baris diabaikan karena error.";
            }
            
            $result = redirect()->route('anggota.index')
                               ->with('success', $message);
            
            if (!empty($summary['errors'])) {
                $result->with('import_errors', $summary['errors']);
            }
            
            return $result;
        } else {
            return redirect()->back()
                            ->with('error', 'Tidak ada data yang berhasil diimport.')
                            ->with('import_errors', $summary['errors']);
        }
    }

    public function downloadTemplate()
    {
        return Excel::download(
            new AnggotaTemplateExport(), 
            'template-import-anggota-' . date('Y-m-d') . '.xlsx'
        );
    }

    public function importPreview(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv|max:10240'
        ]);

        try {
            $file = $request->file('file');
            $data = Excel::toArray(new AnggotaImport(), $file);
            
            if (empty($data[0])) {
                return response()->json([
                    'success' => false,
                    'message' => 'File kosong atau tidak memiliki data.'
                ]);
            }
            
            $rows = collect($data[0])->take(10); // Preview 10 rows only
            
            return response()->json([
                'success' => true,
                'message' => 'Preview data berhasil dimuat',
                'data' => [
                    'headers' => array_keys($rows->first()),
                    'rows' => $rows->toArray(),
                    'total_rows' => count($data[0])
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    public function getImportStatus()
    {
        // For large imports, you might want to implement a job queue
        // and return the status of the import job
        return response()->json([
            'status' => 'completed',
            'progress' => 100,
            'message' => 'Import selesai'
        ]);
    }
    
    public function print($slug)
    {
        $anggota = $this->resolveAnggota($slug);
        
        // Load relationships untuk print
        $anggota->load([
            'refJenjang',
            'refGrupUpa',
            'dpc',
            'dpd',
            'keahlian',
            'aktivitas',
            'aktivitasLuar'
        ]);
        
        return view('anggota.print', compact('anggota'));
    }
    
    public function updateFoto(Request $request, $slug)
    {
        try {
            $anggota = $this->resolveAnggota($slug);
            
            $request->validate([
                'foto' => 'required|image|mimes:jpeg,png,jpg,gif|max:4096' // 4MB max
            ]);
            
            // Hapus foto lama jika ada
            if ($anggota->foto && Storage::disk('public')->exists($anggota->foto)) {
                Storage::disk('public')->delete($anggota->foto);
            }
            
            // Upload foto baru
            $file = $request->file('foto');
            $filename = 'foto_' . $anggota->id . '_' . time() . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs('foto_anggota', $filename, 'public');
            
            // Update database
            $anggota->update(['foto' => $path]);
            
            return redirect()->back()->with('success', 'Foto profil berhasil diubah!');
            
        } catch (\Exception $e) {
            \Log::error('Error updating foto: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Terjadi kesalahan saat mengubah foto: ' . $e->getMessage());
        }
    }

}