<?php

namespace App\Exports;

use App\Models\Anggota;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class StatistikAnggotaExport implements FromQuery, WithHeadings, WithMapping, WithStyles, WithColumnWidths, WithChunkReading
{
    protected $query;

    public function __construct($query)
    {
        $this->query = $query;
    }

    public function query()
    {
        return $this->query->with(['refJenjang', 'dpc', 'dpd']);
    }

    public function headings(): array
    {
        return [
            'No',
            'NIA',
            'Nama Lengkap',
            'Jenis Kelamin',
            'Umur',
            'Jenjang',
            'Status Keanggotaan',
            'Email',
            'No Telepon',
            'Tempat Lahir',
            'Tanggal Lahir',
            'Alamat',
            'Kota',
            'Provinsi',
            'Kode Pos',
            'DPD',
            'DPC',
            'Tanggal Bergabung'
        ];
    }

    public function map($anggota): array
    {
        static $no = 1;
        
        return [
            $no++,
            $anggota->nomor_identitas_anggota,
            $anggota->nama_lengkap,
            $anggota->jenis_kelamin == 'L' ? 'Laki-laki' : 'Perempuan',
            $anggota->umur ?? $this->calculateAge($anggota->tanggal_lahir),
            $anggota->refJenjang->jenjang ?? $anggota->jenjang_anggota ?? '-',
            $anggota->status_keanggotaan ?? $anggota->status ?? '-',
            $anggota->email,
            $anggota->no_telepon,
            $anggota->tempat_lahir,
            $anggota->tanggal_lahir ? \Carbon\Carbon::parse($anggota->tanggal_lahir)->format('d/m/Y') : '-',
            $anggota->alamat,
            $anggota->kota,
            $anggota->provinsi,
            $anggota->kode_pos,
            $anggota->dpd->nama_dpd ?? '-',
            $anggota->dpc->nama_dpc ?? '-',
            $anggota->created_at ? $anggota->created_at->format('d/m/Y') : '-'
        ];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 5,   // No
            'B' => 15,  // NIA
            'C' => 25,  // Nama Lengkap
            'D' => 12,  // Jenis Kelamin
            'E' => 8,   // Umur
            'F' => 12,  // Jenjang
            'G' => 15,  // Status Keanggotaan
            'H' => 25,  // Email
            'I' => 15,  // No Telepon
            'J' => 15,  // Tempat Lahir
            'K' => 12,  // Tanggal Lahir
            'L' => 30,  // Alamat
            'M' => 15,  // Kota
            'N' => 15,  // Provinsi
            'O' => 10,  // Kode Pos
            'P' => 20,  // DPD
            'Q' => 20,  // DPC
            'R' => 15,  // Tanggal Bergabung
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            // Style untuk header
            1 => [
                'font' => [
                    'bold' => true,
                    'color' => ['rgb' => 'FFFFFF']
                ],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'FF5722']
                ],
                'alignment' => [
                    'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
                    'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER
                ]
            ],
            // Style untuk data
            'A:R' => [
                'alignment' => [
                    'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                        'color' => ['rgb' => 'CCCCCC']
                    ]
                ]
            ]
        ];
    }

    public function chunkSize(): int
    {
        return 1000; // Process 1000 records at a time
    }

    private function calculateAge($tanggalLahir)
    {
        if (!$tanggalLahir) {
            return '-';
        }
        
        try {
            return \Carbon\Carbon::parse($tanggalLahir)->age;
        } catch (\Exception $e) {
            return '-';
        }
    }
}
