// Mobile App JavaScript
import axios from 'axios';

// Mobile App Class
class MobileApp {
    constructor() {
        this.apiBaseUrl = '/api/mobile';
        this.setupAxios();
        this.init();
    }

    setupAxios() {
        // Setup CSRF token
        const token = document.querySelector('meta[name="csrf-token"]');
        if (token) {
            axios.defaults.headers.common['X-CSRF-TOKEN'] = token.getAttribute('content');
        }
        
        // Setup base URL
        axios.defaults.baseURL = window.location.origin;
        
        // Setup response interceptor
        axios.interceptors.response.use(
            response => response,
            error => {
                console.error('API Error:', error);
                this.showAlert('Terjadi kesalahan pada server', 'danger');
                return Promise.reject(error);
            }
        );
    }

    init() {
        // Initialize mobile features
        this.setupBottomNavigation();
        this.setupTouchEvents();
        this.setupPullToRefresh();
        this.setupOfflineDetection();
        
        // Initialize specific modules
        if (document.getElementById('attendance-module')) {
            this.initAttendanceModule();
        }
        
        if (document.getElementById('mutasi-module')) {
            this.initMutasiModule();
        }
    }

    // Bottom Navigation
    setupBottomNavigation() {
        const navItems = document.querySelectorAll('.mobile-nav-item');
        navItems.forEach(item => {
            item.addEventListener('click', (e) => {
                e.preventDefault();
                const href = item.querySelector('a')?.href;
                if (href) {
                    window.location.href = href;
                }
            });
        });
    }

    // Touch Events
    setupTouchEvents() {
        // Add touch feedback to buttons
        const buttons = document.querySelectorAll('.mobile-button, .mobile-touchable');
        buttons.forEach(button => {
            button.addEventListener('touchstart', () => {
                button.classList.add('active:scale-95');
            });
            
            button.addEventListener('touchend', () => {
                setTimeout(() => {
                    button.classList.remove('active:scale-95');
                }, 150);
            });
        });
    }

    // Pull to Refresh
    setupPullToRefresh() {
        let startY = 0;
        let currentY = 0;
        let isRefreshing = false;

        document.addEventListener('touchstart', (e) => {
            if (window.scrollY === 0) {
                startY = e.touches[0].clientY;
            }
        });

        document.addEventListener('touchmove', (e) => {
            if (window.scrollY === 0 && startY > 0) {
                currentY = e.touches[0].clientY;
                const pullDistance = currentY - startY;
                
                if (pullDistance > 50 && !isRefreshing) {
                    this.refreshPage();
                }
            }
        });

        document.addEventListener('touchend', () => {
            startY = 0;
            currentY = 0;
        });
    }

    // Offline Detection
    setupOfflineDetection() {
        window.addEventListener('online', () => {
            this.showAlert('Koneksi internet tersambung', 'success');
        });

        window.addEventListener('offline', () => {
            this.showAlert('Tidak ada koneksi internet', 'warning');
        });
    }

    // Attendance Module
    initAttendanceModule() {
        const checkinBtn = document.getElementById('checkin-btn');
        const checkoutBtn = document.getElementById('checkout-btn');
        const qrScanner = document.getElementById('qr-scanner');

        if (checkinBtn) {
            checkinBtn.addEventListener('click', () => this.handleCheckin());
        }

        if (checkoutBtn) {
            checkoutBtn.addEventListener('click', () => this.handleCheckout());
        }

        if (qrScanner) {
            this.initQRScanner();
        }
    }

    // Mutasi Module
    initMutasiModule() {
        const submitBtn = document.getElementById('submit-mutasi');
        const form = document.getElementById('mutasi-form');

        if (submitBtn && form) {
            submitBtn.addEventListener('click', (e) => {
                e.preventDefault();
                this.submitMutasi(form);
            });
        }
    }

    // Check-in Handler
    async handleCheckin() {
        const kegiatanId = document.getElementById('kegiatan-id')?.value;
        const anggotaId = document.getElementById('anggota-id')?.value;

        if (!kegiatanId || !anggotaId) {
            this.showAlert('Data tidak lengkap', 'danger');
            return;
        }

        try {
            this.showLoading(true);
            
            const response = await axios.post(`${this.apiBaseUrl}/attendance/checkin`, {
                kegiatan_id: kegiatanId,
                anggota_id: anggotaId
            });

            if (response.data.success) {
                this.showAlert('Check-in berhasil!', 'success');
                this.updateAttendanceStatus('checked-in');
            } else {
                this.showAlert(response.data.message || 'Check-in gagal', 'danger');
            }
        } catch (error) {
            console.error('Check-in error:', error);
            this.showAlert('Terjadi kesalahan saat check-in', 'danger');
        } finally {
            this.showLoading(false);
        }
    }

    // Check-out Handler
    async handleCheckout() {
        const kegiatanId = document.getElementById('kegiatan-id')?.value;
        const anggotaId = document.getElementById('anggota-id')?.value;

        if (!kegiatanId || !anggotaId) {
            this.showAlert('Data tidak lengkap', 'danger');
            return;
        }

        try {
            this.showLoading(true);
            
            const response = await axios.post(`${this.apiBaseUrl}/attendance/checkout`, {
                kegiatan_id: kegiatanId,
                anggota_id: anggotaId
            });

            if (response.data.success) {
                this.showAlert('Check-out berhasil!', 'success');
                this.updateAttendanceStatus('checked-out');
            } else {
                this.showAlert(response.data.message || 'Check-out gagal', 'danger');
            }
        } catch (error) {
            console.error('Check-out error:', error);
            this.showAlert('Terjadi kesalahan saat check-out', 'danger');
        } finally {
            this.showLoading(false);
        }
    }

    // Submit Mutasi
    async submitMutasi(form) {
        const formData = new FormData(form);
        
        try {
            this.showLoading(true);
            
            const response = await axios.post(`${this.apiBaseUrl}/mutasi/submit`, formData, {
                headers: {
                    'Content-Type': 'multipart/form-data'
                }
            });

            if (response.data.success) {
                this.showAlert('Pengajuan mutasi berhasil dikirim!', 'success');
                form.reset();
            } else {
                this.showAlert(response.data.message || 'Pengajuan mutasi gagal', 'danger');
            }
        } catch (error) {
            console.error('Mutasi error:', error);
            this.showAlert('Terjadi kesalahan saat mengirim pengajuan', 'danger');
        } finally {
            this.showLoading(false);
        }
    }

    // QR Scanner
    initQRScanner() {
        // Simple QR code scanner implementation
        const video = document.getElementById('qr-video');
        const canvas = document.getElementById('qr-canvas');
        const context = canvas.getContext('2d');

        if (navigator.mediaDevices && navigator.mediaDevices.getUserMedia) {
            navigator.mediaDevices.getUserMedia({ video: { facingMode: 'environment' } })
                .then(stream => {
                    video.srcObject = stream;
                    video.play();
                    this.scanQRCode(video, canvas, context);
                })
                .catch(error => {
                    console.error('Camera access denied:', error);
                    this.showAlert('Akses kamera diperlukan untuk scan QR code', 'warning');
                });
        } else {
            this.showAlert('Browser tidak mendukung akses kamera', 'warning');
        }
    }

    scanQRCode(video, canvas, context) {
        const scan = () => {
            if (video.readyState === video.HAVE_ENOUGH_DATA) {
                canvas.height = video.videoHeight;
                canvas.width = video.videoWidth;
                context.drawImage(video, 0, 0, canvas.width, canvas.height);
                
                // Here you would integrate with a QR code library
                // For now, we'll simulate QR detection
                this.simulateQRDetection();
            }
            requestAnimationFrame(scan);
        };
        scan();
    }

    simulateQRDetection() {
        // Simulate QR code detection after 3 seconds
        setTimeout(() => {
            const mockQRData = 'KEGIATAN_123_ANGGOTA_456';
            this.processQRCode(mockQRData);
        }, 3000);
    }

    processQRCode(qrData) {
        const parts = qrData.split('_');
        if (parts.length >= 3) {
            const kegiatanId = parts[1];
            const anggotaId = parts[3];
            
            document.getElementById('kegiatan-id').value = kegiatanId;
            document.getElementById('anggota-id').value = anggotaId;
            
            this.showAlert('QR Code berhasil dibaca!', 'success');
            this.handleCheckin();
        } else {
            this.showAlert('QR Code tidak valid', 'danger');
        }
    }

    // Utility Functions
    showAlert(message, type = 'info') {
        const alertContainer = document.getElementById('alert-container') || this.createAlertContainer();
        
        const alert = document.createElement('div');
        alert.className = `mobile-alert mobile-alert-${type} mobile-slide-up`;
        alert.innerHTML = `
            <div class="flex items-center justify-between">
                <span>${message}</span>
                <button onclick="this.parentElement.parentElement.remove()" class="ml-2 text-lg">&times;</button>
            </div>
        `;
        
        alertContainer.appendChild(alert);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (alert.parentElement) {
                alert.remove();
            }
        }, 5000);
    }

    createAlertContainer() {
        const container = document.createElement('div');
        container.id = 'alert-container';
        container.className = 'fixed top-4 left-4 right-4 z-50 space-y-2';
        document.body.appendChild(container);
        return container;
    }

    showLoading(show) {
        const loading = document.getElementById('mobile-loading') || this.createLoadingElement();
        loading.style.display = show ? 'flex' : 'none';
    }

    createLoadingElement() {
        const loading = document.createElement('div');
        loading.id = 'mobile-loading';
        loading.className = 'mobile-loading fixed inset-0 bg-black bg-opacity-50 z-50';
        loading.innerHTML = '<div class="mobile-spinner"></div>';
        document.body.appendChild(loading);
        return loading;
    }

    updateAttendanceStatus(status) {
        const statusElement = document.getElementById('attendance-status');
        if (statusElement) {
            statusElement.textContent = status === 'checked-in' ? 'Sudah Check-in' : 'Sudah Check-out';
            statusElement.className = `mobile-badge mobile-badge-${status === 'checked-in' ? 'success' : 'info'}`;
        }
    }

    refreshPage() {
        if (isRefreshing) return;
        
        isRefreshing = true;
        this.showAlert('Memperbarui data...', 'info');
        
        setTimeout(() => {
            window.location.reload();
        }, 1000);
    }
}

// Initialize Mobile App when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    new MobileApp();
});

// Export for use in other modules
export default MobileApp;
