<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\Upa;
use App\Models\Anggota;
use Illuminate\Support\Facades\DB;

class UpaPembimbingSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Ambil semua UPA yang aktif dan punya DPC
        $upaList = Upa::with('anggotaAktif')
            ->where('status', 'Aktif')
            ->whereNotNull('dpc_id')
            ->get();

        if ($upaList->isEmpty()) {
            $this->command->warn('Tidak ada UPA ditemukan!');
            return;
        }

        $this->command->info("Menambahkan pembimbing untuk setiap UPA...");
        $this->command->info("Total UPA: " . $upaList->count());

        $totalPembimbing = 0;

        foreach ($upaList as $upa) {
            // Cek apakah sudah ada pembimbing
            $existingPembimbing = DB::table('tbl_upa_pembimbing')
                ->where('upa_id', $upa->upa_id)
                ->where('status', 'Aktif')
                ->count();

            if ($existingPembimbing > 0) {
                $this->command->warn("  ↳ UPA {$upa->nama_upa} sudah punya {$existingPembimbing} pembimbing, skip...");
                continue;
            }

            // Ambil anggota dari UPA ini
            $anggotaList = $upa->anggotaAktif()->take(3)->get();

            if ($anggotaList->isEmpty() || $anggotaList->count() < 2) {
                $this->command->warn("  ↳ UPA {$upa->nama_upa} tidak punya cukup anggota (min 2), skip...");
                continue;
            }

            $this->command->info("  Menambahkan pembimbing untuk: {$upa->nama_upa}");

            $jabatan = ['Ketua', 'Wakil Ketua', 'Sekretaris'];
            $tanggalMulai = now()->subYears(rand(1, 3))->format('Y-m-d');

            foreach ($anggotaList as $index => $anggota) {
                // Skip jika sudah jadi pembimbing di UPA ini
                $isPembimbing = DB::table('tbl_upa_pembimbing')
                    ->where('upa_id', $upa->upa_id)
                    ->where('anggota_id', $anggota->id)
                    ->exists();

                if ($isPembimbing) {
                    continue;
                }

                $jabatanPembimbing = $jabatan[$index] ?? 'Anggota';

                // Insert pembimbing
                DB::table('tbl_upa_pembimbing')->insert([
                    'upa_id' => $upa->upa_id,
                    'anggota_id' => $anggota->id,
                    'jabatan' => $jabatanPembimbing,
                    'tanggal_mulai' => $tanggalMulai,
                    'tanggal_selesai' => null,
                    'status' => 'Aktif',
                    'catatan' => "{$jabatanPembimbing} UPA {$upa->nama_upa}",
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                $this->command->info("    ✓ {$anggota->nama_lengkap} sebagai {$jabatanPembimbing}");
                $totalPembimbing++;
            }
        }

        $this->command->info("\n✓ Total berhasil menambahkan {$totalPembimbing} pembimbing untuk {$upaList->count()} UPA");
    }
}

