<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\Upa;
use App\Models\Anggota;
use App\Models\RefJenjang;
use Illuminate\Support\Facades\DB;

class UpaAnggotaComprehensiveSeeder extends Seeder
{
    /**
     * Daftar nama depan untuk generate nama
     */
    private $namaDepan = [
        'Ahmad', 'Muhammad', 'Abdul', 'Abdullah', 'Ali', 'Umar', 'Hasan', 'Husain', 'Ibrahim', 'Ismail',
        'Yusuf', 'Ya\'qub', 'Musa', 'Harun', 'Dawud', 'Sulaiman', 'Zakariya', 'Yahya', 'Isa', 'Idris',
        'Fatimah', 'Aisyah', 'Khadijah', 'Zainab', 'Ruqayyah', 'Ummu Kultsum', 'Hafsah', 'Zainab', 'Asma', 'Safiyyah'
    ];

    private $namaBelakang = [
        'Rahman', 'Rahim', 'Hakim', 'Aziz', 'Karim', 'Latif', 'Jabbar', 'Malik', 'Salam', 'Mumin',
        'Muhsin', 'Khalil', 'Jamil', 'Majid', 'Rashid', 'Sabur', 'Shakur', 'Wahid', 'Wadud', 'Hamid'
    ];

    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Ambil semua UPA yang punya DPC
        $upaList = Upa::with('dpc')
            ->where('status', 'Aktif')
            ->whereNotNull('dpc_id')
            ->get();

        if ($upaList->isEmpty()) {
            $this->command->warn('Tidak ada UPA dengan DPC ditemukan! Jalankan UpaComprehensiveSeeder terlebih dahulu.');
            return;
        }

        // Ambil semua jenjang
        $jenjangList = RefJenjang::all();
        if ($jenjangList->isEmpty()) {
            $this->command->warn('Tidak ada jenjang ditemukan!');
            return;
        }

        $this->command->info("Membuat 10 anggota untuk setiap UPA...");
        $this->command->info("Total UPA: " . $upaList->count());

        $totalAnggota = 0;

        foreach ($upaList as $upa) {
            // Skip jika DPC tidak ada
            if (!$upa->dpc) {
                $this->command->warn("  ↳ UPA {$upa->nama_upa} tidak punya DPC, skip...");
                continue;
            }

            $this->command->info("  Membuat anggota untuk: {$upa->nama_upa} ({$upa->kode_upa})");

            for ($i = 1; $i <= 10; $i++) {
                // Generate NIA unik
                $nia = $this->generateNIA($upa, $i);

                // Skip jika NIA sudah ada
                if (Anggota::where('nomor_identitas_anggota', $nia)->exists()) {
                    $this->command->warn("    ↳ Anggota {$nia} sudah ada, skip...");
                    continue;
                }

                // Generate nama random
                $namaLengkap = $this->generateNama();
                $jenisKelamin = rand(0, 1) ? 'L' : 'P';

                // Pilih jenjang random
                $jenjang = $jenjangList->random();

                // Insert anggota
                DB::table('tbl_anggota')->insert([
                    'nomor_identitas_anggota' => $nia,
                    'nama_lengkap' => $namaLengkap,
                    'tempat_lahir' => 'Jakarta',
                    'tanggal_lahir' => now()->subYears(rand(25, 55))->format('Y-m-d'),
                    'jenis_kelamin' => $jenisKelamin,
                    'alamat_jalan' => "Jl. {$upa->dpc->nama_dpc} No. " . rand(1, 100),
                    'rt' => str_pad(rand(1, 15), 3, '0', STR_PAD_LEFT),
                    'rw' => str_pad(rand(1, 10), 3, '0', STR_PAD_LEFT),
                    'kelurahan' => $upa->dpc->nama_dpc,
                    'kecamatan' => $upa->dpc->nama_dpc,
                    'kota' => 'Jakarta Timur',
                    'provinsi' => 'DKI Jakarta',
                    'kode_pos' => '13' . str_pad(rand(100, 999), 3, '0', STR_PAD_LEFT),
                    'no_telepon' => '08' . rand(10, 99) . rand(10000000, 99999999),
                    'email' => strtolower(str_replace(' ', '', $namaLengkap)) . rand(1, 999) . '@email.com',
                    'ref_jenjang_id' => $jenjang->id,
                    'upa_id' => $upa->upa_id,
                    'dpc_id' => $upa->dpc_id,
                    'dpd_id' => $upa->dpc->dpd_id ?? 1,
                    'tanggal_bergabung' => now()->subYears(rand(1, 5))->format('Y-m-d'),
                    'status_keanggotaan' => 'Aktif',
                    'muzaki_mustahiq' => rand(0, 1) ? 'Muzaki' : 'Mustahiq',
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                $totalAnggota++;
            }

            $this->command->info("    ✓ Berhasil membuat 10 anggota");
        }

        $this->command->info("\n✓ Total berhasil membuat {$totalAnggota} anggota untuk {$upaList->count()} UPA");
    }

    /**
     * Generate NIA (Nomor Identitas Anggota)
     */
    private function generateNIA($upa, $nomor)
    {
        // Format: PKS-XXX-YYYY
        // XXX = kode DPC
        // YYYY = nomor urut
        
        $counter = DB::table('tbl_anggota')
            ->where('dpc_id', $upa->dpc_id)
            ->count();

        return 'PKS' . str_pad($upa->dpc_id, 3, '0', STR_PAD_LEFT) . '-' . date('Y') . str_pad($counter + $nomor, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Generate nama random
     */
    private function generateNama()
    {
        $namaDepan = $this->namaDepan[array_rand($this->namaDepan)];
        $namaBelakang = $this->namaBelakang[array_rand($this->namaBelakang)];
        
        return $namaDepan . ' ' . $namaBelakang;
    }
}

