<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DatabaseSeeder extends Seeder
{
    public function run()
    {
        // Disable foreign key checks selama seeding
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        
        try {
            $this->command->info('🌱 Starting database seeding...');
            
            // Seeder dengan urutan yang benar
            $this->call(UserSeeder::class);
            $this->seedStrukturOrganisasi();
            $this->seedAnggotaSample();
            $this->seedKegiatanSample();
            
            $this->command->info('✅ Database seeding completed successfully!');
            
        } catch (\Exception $e) {
            $this->command->error('❌ Seeding failed: ' . $e->getMessage());
            throw $e;
        } finally {
            // Re-enable foreign key checks
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        }
    }

    private function seedStrukturOrganisasi()
    {
        $this->command->info('Seeding struktur organisasi...');
        
        // STEP 1: Insert Provinsi dulu (parent_id = null)
        $provinsi_data = [
            [
                'id' => 1,
                'kode_struktur' => 'PROV-01', 
                'nama_struktur' => 'DPW Jawa Barat', 
                'tipe_struktur' => 'Provinsi', 
                'parent_id' => null,
                'status' => 'Aktif',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'id' => 2,
                'kode_struktur' => 'PROV-02', 
                'nama_struktur' => 'DPW DKI Jakarta', 
                'tipe_struktur' => 'Provinsi', 
                'parent_id' => null,
                'status' => 'Aktif',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'id' => 3,
                'kode_struktur' => 'PROV-03', 
                'nama_struktur' => 'DPW Jawa Tengah', 
                'tipe_struktur' => 'Provinsi', 
                'parent_id' => null,
                'status' => 'Aktif',
                'created_at' => now(),
                'updated_at' => now()
            ],
        ];

        // Insert provinsi satu per satu
        foreach ($provinsi_data as $provinsi) {
            DB::table('tbl_struktur_organisasi')->insert($provinsi);
        }
        
        $this->command->info('✓ Provinsi seeded successfully');

        // STEP 2: Insert Kabupaten/Kota (parent_id merujuk ke provinsi yang sudah ada)
        $kabupaten_data = [
            [
                'id' => 4,
                'kode_struktur' => 'KAB-01', 
                'nama_struktur' => 'DPC Bandung', 
                'tipe_struktur' => 'Kabupaten/Kota', 
                'parent_id' => 1, // DPW Jawa Barat
                'status' => 'Aktif',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'id' => 5,
                'kode_struktur' => 'KAB-02', 
                'nama_struktur' => 'DPC Bogor', 
                'tipe_struktur' => 'Kabupaten/Kota', 
                'parent_id' => 1, // DPW Jawa Barat
                'status' => 'Aktif',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'id' => 6,
                'kode_struktur' => 'KAB-03', 
                'nama_struktur' => 'DPC Depok', 
                'tipe_struktur' => 'Kabupaten/Kota', 
                'parent_id' => 1, // DPW Jawa Barat
                'status' => 'Aktif',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'id' => 7,
                'kode_struktur' => 'KAB-04', 
                'nama_struktur' => 'DPC Bekasi', 
                'tipe_struktur' => 'Kabupaten/Kota', 
                'parent_id' => 1, // DPW Jawa Barat
                'status' => 'Aktif',
                'created_at' => now(),
                'updated_at' => now()
            ],
        ];

        // Insert kabupaten satu per satu
        foreach ($kabupaten_data as $kabupaten) {
            DB::table('tbl_struktur_organisasi')->insert($kabupaten);
        }
        
        $this->command->info('✓ Kabupaten/Kota seeded successfully');

        // STEP 3: Insert Kecamatan (parent_id merujuk ke kabupaten yang sudah ada)
        $kecamatan_data = [
            [
                'id' => 8,
                'kode_struktur' => 'KEC-01', 
                'nama_struktur' => 'PAC Pancoran Mas', 
                'tipe_struktur' => 'Kecamatan', 
                'parent_id' => 6, // DPC Depok
                'status' => 'Aktif',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'id' => 9,
                'kode_struktur' => 'KEC-02', 
                'nama_struktur' => 'PAC Beji', 
                'tipe_struktur' => 'Kecamatan', 
                'parent_id' => 6, // DPC Depok
                'status' => 'Aktif',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'id' => 10,
                'kode_struktur' => 'KEC-03', 
                'nama_struktur' => 'PAC Cipayung', 
                'tipe_struktur' => 'Kecamatan', 
                'parent_id' => 6, // DPC Depok
                'status' => 'Aktif',
                'created_at' => now(),
                'updated_at' => now()
            ],
        ];

        // Insert kecamatan satu per satu
        foreach ($kecamatan_data as $kecamatan) {
            DB::table('tbl_struktur_organisasi')->insert($kecamatan);
        }
        
        $this->command->info('✓ Kecamatan seeded successfully');
        $this->command->info('✅ Struktur organisasi seeding completed');
    }

    private function seedAnggotaSample()
    {
        $this->command->info('Seeding anggota sample...');
        
        $anggota_sample = [
            [
                'nomor_identitas_anggota' => 'PKS-001-2024',
                'nama_lengkap' => 'Ahmad Sutanto',
                'tempat_lahir' => 'Jakarta',
                'tanggal_lahir' => '1985-03-15',
                'jenis_kelamin' => 'L',
                'alamat_jalan' => 'Jl. Margonda Raya No. 123',
                'rt' => '01',
                'rw' => '05',
                'kelurahan' => 'Kemiri Muka',
                'kecamatan' => 'Beji',
                'kota' => 'Depok',
                'provinsi' => 'Jawa Barat',
                'no_telepon' => '081234567890',
                'email' => 'ahmad.sutanto@email.com',
                'jenjang_anggota' => 'Anggota Biasa',
                'tanggal_bergabung' => '2020-01-15',
                'status_keanggotaan' => 'Aktif',
                'penghasilan_per_bulan' => 5000000,
                'muzaki_mustahiq' => 'Muzaki',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'nomor_identitas_anggota' => 'PKS-002-2024',
                'nama_lengkap' => 'Siti Nurhaliza',
                'tempat_lahir' => 'Bandung',
                'tanggal_lahir' => '1990-07-22',
                'jenis_kelamin' => 'P',
                'alamat_jalan' => 'Jl. Raya Cisalak No. 45',
                'rt' => '03',
                'rw' => '07',
                'kelurahan' => 'Cisalak',
                'kecamatan' => 'Cipayung',
                'kota' => 'Depok',
                'provinsi' => 'Jawa Barat',
                'no_telepon' => '081298765432',
                'email' => 'siti.nurhaliza@email.com',
                'jenjang_anggota' => 'Anggota Biasa',
                'tanggal_bergabung' => '2021-03-10',
                'status_keanggotaan' => 'Aktif',
                'penghasilan_per_bulan' => 3500000,
                'muzaki_mustahiq' => 'Mustahiq',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'nomor_identitas_anggota' => 'PKS-003-2024',
                'nama_lengkap' => 'Muhammad Ridwan',
                'tempat_lahir' => 'Bogor',
                'tanggal_lahir' => '1982-11-08',
                'jenis_kelamin' => 'L',
                'alamat_jalan' => 'Jl. Juanda No. 67',
                'rt' => '02',
                'rw' => '04',
                'kelurahan' => 'Pancoran Mas',
                'kecamatan' => 'Pancoran Mas',
                'kota' => 'Depok',
                'provinsi' => 'Jawa Barat',
                'no_telepon' => '081345678901',
                'email' => 'muhammad.ridwan@email.com',
                'jenjang_anggota' => 'Kader',
                'tanggal_bergabung' => '2019-06-20',
                'status_keanggotaan' => 'Aktif',
                'penghasilan_per_bulan' => 7500000,
                'muzaki_mustahiq' => 'Muzaki',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'nomor_identitas_anggota' => 'PKS-004-2024',
                'nama_lengkap' => 'Fatimah Zahra',
                'tempat_lahir' => 'Depok',
                'tanggal_lahir' => '1995-12-03',
                'jenis_kelamin' => 'P',
                'alamat_jalan' => 'Jl. Veteran No. 89',
                'rt' => '05',
                'rw' => '02',
                'kelurahan' => 'Beji',
                'kecamatan' => 'Beji',
                'kota' => 'Depok',
                'provinsi' => 'Jawa Barat',
                'no_telepon' => '081456789012',
                'email' => 'fatimah.zahra@email.com',
                'jenjang_anggota' => 'Anggota Biasa',
                'tanggal_bergabung' => '2022-09-15',
                'status_keanggotaan' => 'Aktif',
                'penghasilan_per_bulan' => 4200000,
                'muzaki_mustahiq' => 'Muzaki',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'nomor_identitas_anggota' => 'PKS-005-2024',
                'nama_lengkap' => 'Budi Santoso',
                'tempat_lahir' => 'Jakarta',
                'tanggal_lahir' => '1988-04-17',
                'jenis_kelamin' => 'L',
                'alamat_jalan' => 'Jl. Sawangan Raya No. 156',
                'rt' => '04',
                'rw' => '08',
                'kelurahan' => 'Cipayung',
                'kecamatan' => 'Cipayung',
                'kota' => 'Depok',
                'provinsi' => 'Jawa Barat',
                'no_telepon' => '081567890123',
                'email' => 'budi.santoso@email.com',
                'jenjang_anggota' => 'Kader',
                'tanggal_bergabung' => '2018-11-30',
                'status_keanggotaan' => 'Tidak Aktif',
                'penghasilan_per_bulan' => 6000000,
                'muzaki_mustahiq' => 'Muzaki',
                'created_at' => now(),
                'updated_at' => now()
            ]
        ];

        foreach ($anggota_sample as $anggota) {
            DB::table('tbl_anggota')->insert($anggota);
        }
        
        $this->command->info('✅ Anggota seeding completed');
    }

    private function seedKegiatanSample()
    {
        $this->command->info('Seeding kegiatan sample...');
        
        $kegiatan_sample = [
            [
                'kode_kegiatan' => 'KGT-001-2024',
                'nama_kegiatan' => 'Mabit Ramadhan 1445H',
                'jenis_kegiatan' => 'Mabit',
                'uraian_kegiatan' => 'Kegiatan malam bina iman dan takwa dalam rangka bulan Ramadhan',
                'tanggal_mulai' => '2024-03-15 19:00:00',
                'tanggal_selesai' => '2024-03-16 05:00:00',
                'tempat' => 'Masjid Al-Hikmah Depok',
                'alamat_tempat' => 'Jl. Margonda Raya, Depok, Jawa Barat',
                'struktur_penyelenggara_id' => 6, // DPC Depok
                'pic_kegiatan' => 'Ahmad Sutanto',
                'status_kegiatan' => 'Completed',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'kode_kegiatan' => 'KGT-002-2024',
                'nama_kegiatan' => 'Rapat Koordinasi DPC',
                'jenis_kegiatan' => 'Rapat',
                'uraian_kegiatan' => 'Rapat koordinasi bulanan pengurus DPC Depok',
                'tanggal_mulai' => '2024-08-10 14:00:00',
                'tanggal_selesai' => '2024-08-10 17:00:00',
                'tempat' => 'Kantor DPC PKS Depok',
                'alamat_tempat' => 'Jl. Juanda No. 45, Depok, Jawa Barat',
                'struktur_penyelenggara_id' => 6, // DPC Depok
                'pic_kegiatan' => 'Muhammad Ridwan',
                'status_kegiatan' => 'Completed',
                'created_at' => now(),
                'updated_at' => now()
            ],
            [
                'kode_kegiatan' => 'KGT-003-2024',
                'nama_kegiatan' => 'Bakti Sosial Peduli Banjir',
                'jenis_kegiatan' => 'Lainnya',
                'uraian_kegiatan' => 'Kegiatan bakti sosial untuk membantu korban banjir di wilayah Cipayung',
                'tanggal_mulai' => '2024-09-05 08:00:00',
                'tanggal_selesai' => '2024-09-05 15:00:00',
                'tempat' => 'Kelurahan Cipayung',
                'alamat_tempat' => 'Cipayung, Depok, Jawa Barat',
                'struktur_penyelenggara_id' => 10, // PAC Cipayung
                'pic_kegiatan' => 'Siti Nurhaliza',
                'status_kegiatan' => 'Completed',
                'created_at' => now(),
                'updated_at' => now()
            ]
        ];

        foreach ($kegiatan_sample as $kegiatan) {
            DB::table('tbl_kegiatan')->insert($kegiatan);
        }
        
        $this->command->info('✓ Kegiatan seeded successfully');

        // Seed Absensi untuk kegiatan
        $this->seedAbsensiSample();
    }

    private function seedAbsensiSample()
    {
        $this->command->info('Seeding absensi sample...');
        
        // Absensi untuk Mabit Ramadhan (kegiatan_id = 1)
        $absensi_mabit = [
            ['kegiatan_id' => 1, 'anggota_id' => 1, 'status_kehadiran' => 'Hadir', 'waktu_absen' => '2024-03-15 19:15:00'],
            ['kegiatan_id' => 1, 'anggota_id' => 2, 'status_kehadiran' => 'Hadir', 'waktu_absen' => '2024-03-15 19:30:00'],
            ['kegiatan_id' => 1, 'anggota_id' => 3, 'status_kehadiran' => 'Hadir', 'waktu_absen' => '2024-03-15 19:10:00'],
            ['kegiatan_id' => 1, 'anggota_id' => 4, 'status_kehadiran' => 'Izin'],
            ['kegiatan_id' => 1, 'anggota_id' => 5, 'status_kehadiran' => 'Tidak Hadir'],
        ];

        // Absensi untuk Rapat Koordinasi (kegiatan_id = 2)
        $absensi_rapat = [
            ['kegiatan_id' => 2, 'anggota_id' => 1, 'status_kehadiran' => 'Hadir', 'waktu_absen' => '2024-08-10 14:05:00'],
            ['kegiatan_id' => 2, 'anggota_id' => 3, 'status_kehadiran' => 'Hadir', 'waktu_absen' => '2024-08-10 13:55:00'],
            ['kegiatan_id' => 2, 'anggota_id' => 4, 'status_kehadiran' => 'Hadir', 'waktu_absen' => '2024-08-10 14:10:00'],
        ];

        // Absensi untuk Bakti Sosial (kegiatan_id = 3)
        $absensi_baksos = [
            ['kegiatan_id' => 3, 'anggota_id' => 2, 'status_kehadiran' => 'Hadir', 'waktu_absen' => '2024-09-05 08:15:00'],
            ['kegiatan_id' => 3, 'anggota_id' => 4, 'status_kehadiran' => 'Hadir', 'waktu_absen' => '2024-09-05 08:20:00'],
            ['kegiatan_id' => 3, 'anggota_id' => 1, 'status_kehadiran' => 'Sakit'],
        ];

        $all_absensi = array_merge($absensi_mabit, $absensi_rapat, $absensi_baksos);

        foreach ($all_absensi as $absensi) {
            DB::table('tbl_absensi_kegiatan')->insert(array_merge($absensi, [
                'metode_absen' => 'Manual',
                'created_at' => now(),
                'updated_at' => now()
            ]));
        }
        
        $this->command->info('✅ Absensi seeding completed');
    }
}