<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\Anggota;
use App\Models\RefGrupUpa;
use App\Models\RefJenjang;
use App\Models\Dpc;
use App\Models\Dpd;
use App\Models\RefKeahlian;
use Carbon\Carbon;

class AnggotaDemoSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Pastikan data master sudah ada
        $this->command->info('🌱 Seeding data anggota demo...');
        
        // Ambil data master
        $grupUpaList = RefGrupUpa::active()->get();
        $jenjangList = RefJenjang::active()->get();
        $dpcList = Dpc::active()->get();
        $dpdList = Dpd::active()->get();
        $keahlianList = RefKeahlian::active()->get();
        
        if ($grupUpaList->count() < 5) {
            $this->command->error('❌ Minimal 5 grup UPA diperlukan untuk demo');
            return;
        }
        
        // Data nama untuk demo
        $namaLaki = [
            'Ahmad Rizki Pratama', 'Muhammad Fauzi', 'Budi Santoso', 'Eko Prasetyo', 'Dedi Kurniawan',
            'Rizki Ramadhan', 'Andi Wijaya', 'Surya Nugraha', 'Fajar Hidayat', 'Rudi Hartono',
            'Joko Susilo', 'Agus Setiawan', 'Hendra Gunawan', 'Roni Saputra', 'Doni Prasetyo'
        ];
        
        $namaPerempuan = [
            'Siti Nurhaliza', 'Fatimah Zahra', 'Aisyah Putri', 'Maya Sari', 'Dewi Kartika',
            'Rina Wulandari', 'Sari Indah', 'Lina Marlina', 'Nurul Hidayah', 'Rita Sari',
            'Indah Permata', 'Kartika Dewi', 'Sinta Maharani', 'Dina Rahayu', 'Mira Sari'
        ];
        
        $kotaList = ['Jakarta Timur', 'Jakarta Barat', 'Jakarta Selatan', 'Jakarta Utara', 'Jakarta Pusat'];
        $kelurahanList = ['Kelurahan A', 'Kelurahan B', 'Kelurahan C', 'Kelurahan D', 'Kelurahan E'];
        $kecamatanList = ['Kecamatan A', 'Kecamatan B', 'Kecamatan C', 'Kecamatan D', 'Kecamatan E'];
        
        $anggotaData = [];
        $counter = 1;
        
        // Generate 30 anggota (6 per grup UPA)
        for ($grupIndex = 0; $grupIndex < 5; $grupIndex++) {
            $grupUpa = $grupUpaList[$grupIndex];
            
            for ($anggotaIndex = 0; $anggotaIndex < 6; $anggotaIndex++) {
                $isLaki = $anggotaIndex % 2 == 0; // Bergantian laki-perempuan
                $nama = $isLaki ? 
                    $namaLaki[array_rand($namaLaki)] : 
                    $namaPerempuan[array_rand($namaPerempuan)];
                
                // Random data
                $jenjang = $jenjangList->random();
                $dpc = $dpcList->random();
                $dpd = $dpdList->random();
                $kota = $kotaList[array_rand($kotaList)];
                $kelurahan = $kelurahanList[array_rand($kelurahanList)];
                $kecamatan = $kecamatanList[array_rand($kecamatanList)];
                
                // Generate NIA
                $nia = 'PKS-' . str_pad($counter, 3, '0', STR_PAD_LEFT) . '-' . date('Y');
                
                // Generate tanggal lahir (usia 20-50 tahun)
                $tahunLahir = rand(1974, 2004);
                $bulanLahir = rand(1, 12);
                $hariLahir = rand(1, 28);
                $tanggalLahir = Carbon::create($tahunLahir, $bulanLahir, $hariLahir);
                
                // Generate tanggal bergabung (1-5 tahun terakhir)
                $tahunBergabung = rand(2020, 2024);
                $bulanBergabung = rand(1, 12);
                $hariBergabung = rand(1, 28);
                $tanggalBergabung = Carbon::create($tahunBergabung, $bulanBergabung, $hariBergabung);
                
                $anggotaData[] = [
                    'nomor_identitas_anggota' => $nia,
                    'nama_lengkap' => $nama,
                    'tempat_lahir' => $kota,
                    'tanggal_lahir' => $tanggalLahir->format('Y-m-d'),
                    'jenis_kelamin' => $isLaki ? 'L' : 'P',
                    'alamat_jalan' => 'Jl. ' . $nama . ' No. ' . rand(1, 999),
                    'rt' => str_pad(rand(1, 20), 2, '0', STR_PAD_LEFT),
                    'rw' => str_pad(rand(1, 10), 2, '0', STR_PAD_LEFT),
                    'kelurahan' => $kelurahan,
                    'kecamatan' => $kecamatan,
                    'kota' => $kota,
                    'provinsi' => 'DKI Jakarta',
                    'no_telepon' => '08' . rand(1000000000, 9999999999),
                    'email' => strtolower(str_replace(' ', '.', $nama)) . '@email.com',
                    'ref_jenjang_id' => $jenjang->ref_jenjang_id,
                    'ref_grup_upa_id' => $grupUpa->ref_grup_upa_id,
                    'dpc_id' => $dpc->dpc_id,
                    'dpd_id' => $dpd->dpd_id,
                    'tanggal_bergabung' => $tanggalBergabung->format('Y-m-d'),
                    'status_keanggotaan' => rand(0, 10) < 8 ? 'Aktif' : 'Tidak Aktif', // 80% aktif
                    'penghasilan_per_bulan' => rand(3000000, 15000000),
                    'muzaki_mustahiq' => rand(0, 1) ? 'Muzaki' : 'Mustahiq',
                    'nomor_identitas_pasangan' => null,
                    'catatan_profil' => 'Anggota demo untuk testing sistem',
                    'keahlian_lainnya' => null,
                    'created_at' => now(),
                    'updated_at' => now()
                ];
                
                $counter++;
            }
        }
        
        // Insert data anggota
        foreach ($anggotaData as $anggota) {
            Anggota::create($anggota);
        }
        
        // Assign keahlian secara random
        $this->assignKeahlian($keahlianList);
        
        $this->command->info('✅ Data anggota demo berhasil dibuat!');
        $this->command->info('📊 Total: 30 anggota');
        $this->command->info('👥 Terdistribusi ke 5 grup UPA (6 anggota per grup)');
        $this->command->info('🎯 Status: 80% Aktif, 20% Tidak Aktif');
    }
    
    private function assignKeahlian($keahlianList)
    {
        $anggotaList = Anggota::all();
        
        foreach ($anggotaList as $anggota) {
            // Random 2-4 keahlian per anggota
            $jumlahKeahlian = rand(2, 4);
            $keahlianRandom = $keahlianList->random($jumlahKeahlian);
            
            foreach ($keahlianRandom as $keahlian) {
                $anggota->keahlian()->attach($keahlian->ref_keahlian_id);
            }
        }
    }
}