<?php

namespace App\Traits;

use App\Models\Dpc;
use App\Models\Dpd;

trait HasScopeAccess
{
    /**
     * Check if user can access specific DPC
     */
    public function canAccessDpc($dpcId)
    {
        $user = auth()->user();
        
        if (!$user) {
            return false;
        }

        // Superadmin dan Admin Sistem bisa akses semua
        if ($user->hasFullSystemAccess()) {
            return true;
        }

        // Admin DPC hanya bisa akses DPC nya sendiri
        if ($user->isAdminDpc()) {
            return $user->dpc_id == $dpcId;
        }

        // Admin DPD bisa akses semua DPC di bawah DPD nya
        if ($user->isAdminDpd()) {
            $dpcs = Dpc::where('dpd_id', $user->dpd_id)->pluck('dpc_id')->toArray();
            return in_array($dpcId, $dpcs);
        }

        // Anggota bisa akses DPC nya sendiri
        if ($user->isAnggota() && $user->dpc_id) {
            return $user->dpc_id == $dpcId;
        }

        return false;
    }

    /**
     * Check if user can access specific DPD
     */
    public function canAccessDpd($dpdId)
    {
        $user = auth()->user();
        
        if (!$user) {
            return false;
        }

        // Superadmin dan Admin Sistem bisa akses semua
        if ($user->hasFullSystemAccess()) {
            return true;
        }

        // Admin DPD hanya bisa akses DPD nya sendiri
        if ($user->isAdminDpd()) {
            return $user->dpd_id == $dpdId;
        }

        // Admin DPC bisa akses DPD yang menjadi parent dari DPC nya
        if ($user->isAdminDpc() && $user->dpc) {
            return $user->dpc->dpd_id == $dpdId;
        }

        // Anggota bisa akses DPD nya sendiri
        if ($user->isAnggota() && $user->dpd_id) {
            return $user->dpd_id == $dpdId;
        }

        return false;
    }

    /**
     * Apply scope filter to query based on user role
     */
    public function applyScopeFilter($query, $dpcColumn = 'dpc_id', $dpdColumn = 'dpd_id')
    {
        $user = auth()->user();
        
        if (!$user) {
            return $query->whereRaw('1 = 0'); // No access
        }

        // Superadmin dan Admin Sistem bisa akses semua
        if ($user->hasFullSystemAccess()) {
            return $query;
        }

        // Admin DPC hanya bisa akses data DPC nya sendiri
        if ($user->isAdminDpc() && $user->dpc_id) {
            return $query->where($dpcColumn, $user->dpc_id);
        }

        // Admin DPD bisa akses semua data di bawah DPD nya
        if ($user->isAdminDpd() && $user->dpd_id) {
            $dpcIds = Dpc::where('dpd_id', $user->dpd_id)->pluck('dpc_id')->toArray();
            return $query->where(function($q) use ($dpdColumn, $dpcColumn, $user, $dpcIds) {
                $q->where($dpdColumn, $user->dpd_id)
                  ->orWhereIn($dpcColumn, $dpcIds);
            });
        }

        // Anggota bisa akses data miliknya sendiri
        if ($user->isAnggota()) {
            if ($user->dpc_id) {
                return $query->where($dpcColumn, $user->dpc_id);
            }
            if ($user->dpd_id) {
                return $query->where($dpdColumn, $user->dpd_id);
            }
        }

        return $query->whereRaw('1 = 0'); // No access
    }

    /**
     * Check if user can manage users
     */
    public function canManageUsers()
    {
        $user = auth()->user();
        return $user && $user->canManageUsers();
    }

    /**
     * Check if user can approve mutations
     */
    public function canApproveMutations()
    {
        $user = auth()->user();
        return $user && ($user->isAdminDpd() || $user->hasFullSystemAccess());
    }

    /**
     * Check if user can approve placements
     */
    public function canApprovePlacements()
    {
        $user = auth()->user();
        return $user && ($user->isAdminDpd() || $user->hasFullSystemAccess());
    }

    /**
     * Check if user can edit placement composition
     */
    public function canEditPlacementComposition()
    {
        $user = auth()->user();
        return $user && ($user->isAdminDpd() || $user->hasFullSystemAccess());
    }

    /**
     * Check if user can do rekomposisi
     */
    public function canDoRekomposisi()
    {
        $user = auth()->user();
        return $user && ($user->isAdminDpd() || $user->hasFullSystemAccess());
    }

    /**
     * Check if user can create mutations
     */
    public function canCreateMutations()
    {
        $user = auth()->user();
        return $user && ($user->isAdminDpc() || $user->hasFullSystemAccess());
    }

    /**
     * Check if user can create placements
     */
    public function canCreatePlacements()
    {
        $user = auth()->user();
        return $user && ($user->isAdminDpc() || $user->hasFullSystemAccess());
    }
}

