<?php

namespace App\Services;

use App\Models\Anggota;
use App\Models\RiwayatPerpindahanAnggota;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PerpindahanAnggotaService
{
    /**
     * Proses perpindahan anggota
     * 
     * @param array $data
     * @param string $tipe
     * @return RiwayatPerpindahanAnggota
     */
    public function prosesPerpindahan(array $data, string $tipe): RiwayatPerpindahanAnggota
    {
        return DB::transaction(function () use ($data, $tipe) {
            // 1. Ambil data anggota saat ini
            $anggota = Anggota::findOrFail($data['anggota_id']);
            $upaAsalId = $anggota->upa_id;
            $dpcAsalId = $anggota->dpc_id;

            // 2. Generate nomor surat
            $nomorSurat = $this->generateNomorSurat($tipe);

            // 3. Simpan riwayat perpindahan
            $riwayat = RiwayatPerpindahanAnggota::create([
                'tipe_transaksi' => $tipe,
                'anggota_id' => $data['anggota_id'],
                'upa_asal_id' => $upaAsalId,
                'upa_tujuan_id' => $data['upa_tujuan_id'],
                'dpc_asal_id' => $dpcAsalId,
                'dpc_tujuan_id' => $data['dpc_tujuan_id'],
                'pembimbing_lama_id' => $data['pembimbing_lama_id'] ?? null,
                'pembimbing_baru_id' => $data['pembimbing_baru_id'] ?? null,
                'tanggal_efektif' => $data['tanggal_efektif'] ?? now(),
                'tahun_pelantikan' => $data['tahun_pelantikan'] ?? null,
                'nomor_surat' => $nomorSurat,
                'alasan' => $data['alasan'] ?? null,
                'catatan' => $data['catatan'] ?? null,
                'metadata_json' => $data['metadata_json'] ?? null,
                'status' => $data['status'] ?? 'completed',
                'created_by' => Auth::id(),
                'approved_by' => Auth::id(),
                'approved_at' => now(),
            ]);

            // 4. Update data anggota (pindah ke UPA baru)
            // PENTING: Ini mengubah data real anggota
            $anggota->upa_id = $data['upa_tujuan_id'];
            $anggota->dpc_id = $data['dpc_tujuan_id'];
            $anggota->save();

            return $riwayat->load([
                'anggota',
                'upaAsal',
                'upaTujuan',
                'dpcAsal',
                'dpcTujuan',
                'pembimbingLama',
                'pembimbingBaru'
            ]);
        });
    }

    /**
     * Generate nomor surat berdasarkan tipe
     * 
     * @param string $tipe
     * @return string
     */
    private function generateNomorSurat(string $tipe): string
    {
        $prefix = $this->getSuratPrefix($tipe);
        $tahun = Carbon::now()->format('Y');
        $bulan = Carbon::now()->format('m');

        // Hitung jumlah surat di bulan ini untuk tipe tertentu
        $count = RiwayatPerpindahanAnggota::where('tipe_transaksi', $tipe)
            ->whereYear('created_at', $tahun)
            ->whereMonth('created_at', $bulan)
            ->count() + 1;

        $nomorUrut = str_pad($count, 3, '0', STR_PAD_LEFT);

        // Format: PREFIX/NOMOR_URUT/BULAN/TAHUN
        // Contoh: MTS/001/10/2025 untuk Mutasi
        return "{$prefix}/{$nomorUrut}/{$bulan}/{$tahun}";
    }

    /**
     * Get prefix surat berdasarkan tipe
     * 
     * @param string $tipe
     * @return string
     */
    private function getSuratPrefix(string $tipe): string
    {
        $prefixes = [
            'mutasi' => 'MTS',
            'penempatan' => 'PNP',
            'perpindahan' => 'PRH',
            'rekomposisi' => 'RKP',
        ];

        return $prefixes[$tipe] ?? 'TRX';
    }

    /**
     * Get validasi rules berdasarkan tipe
     * 
     * @param string $tipe
     * @return array
     */
    public function getValidationRules(string $tipe): array
    {
        $baseRules = [
            'anggota_id' => 'required|exists:tbl_anggota,id',
            'upa_tujuan_id' => 'required|exists:tbl_upa,upa_id',
            'dpc_tujuan_id' => 'required|exists:tbl_dpc,dpc_id',
            'tanggal_efektif' => 'nullable|date',
            'catatan' => 'nullable|string',
        ];

        // Rules tambahan per tipe
        $additionalRules = [];
        
        switch ($tipe) {
            case 'mutasi':
                $additionalRules = [
                    'alasan' => 'required|string',
                    'pembimbing_lama_id' => 'nullable|exists:tbl_anggota,id',
                    'pembimbing_baru_id' => 'nullable|exists:tbl_anggota,id',
                ];
                break;
            
            case 'penempatan':
                $additionalRules = [
                    'tahun_pelantikan' => 'required|digits:4',
                    'pembimbing_lama_id' => 'nullable|exists:tbl_anggota,id',
                    'pembimbing_baru_id' => 'nullable|exists:tbl_anggota,id',
                ];
                break;
            
            case 'perpindahan':
                $additionalRules = [
                    'alasan' => 'nullable|string',
                ];
                break;
            
            case 'rekomposisi':
                $additionalRules = [
                    'alasan' => 'required|string',
                ];
                break;
        }

        return array_merge($baseRules, $additionalRules);
    }

    /**
     * Get required fields berdasarkan tipe untuk UI
     * 
     * @param string $tipe
     * @return array
     */
    public function getRequiredFields(string $tipe): array
    {
        $requiredFields = [
            'mutasi' => ['alasan', 'pembimbing_lama', 'pembimbing_baru'],
            'penempatan' => ['tahun_pelantikan', 'pembimbing_lama', 'pembimbing_baru'],
            'perpindahan' => [],
            'rekomposisi' => ['alasan'],
        ];

        return $requiredFields[$tipe] ?? [];
    }

    /**
     * Get label tipe transaksi
     * 
     * @param string $tipe
     * @return string
     */
    public function getTipeLabel(string $tipe): string
    {
        $labels = [
            'mutasi' => 'Mutasi Anggota',
            'penempatan' => 'Surat Penempatan',
            'perpindahan' => 'Perpindahan Anggota',
            'rekomposisi' => 'Rekomposisi UPA',
        ];

        return $labels[$tipe] ?? ucfirst($tipe);
    }

    /**
     * Get template surat berdasarkan tipe
     * 
     * @param string $tipe
     * @return string
     */
    public function getSuratTemplate(string $tipe): string
    {
        $templates = [
            'mutasi' => 'perpindahan-anggota.surat.mutasi',
            'penempatan' => 'perpindahan-anggota.surat.penempatan',
            'perpindahan' => 'perpindahan-anggota.surat.perpindahan',
            'rekomposisi' => 'perpindahan-anggota.surat.rekomposisi',
        ];

        return $templates[$tipe] ?? 'perpindahan-anggota.surat.default';
    }
}

