<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'role',
        'is_active',
        'dpc_id',
        'dpd_id',
        'anggota_id',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'is_active' => 'boolean',
        ];
    }

    /**
     * Check if user has specific role
     */
    public function hasRole($role)
    {
        return $this->role === $role;
    }

    /**
     * Check if user is superadmin
     */
    public function isSuperAdmin()
    {
        return $this->role === 'superadmin';
    }

    /**
     * Check if user is admin or superadmin (legacy method)
     */
    public function isAdmin()
    {
        return in_array($this->role, ['admin_sistem', 'superadmin']);
    }

    /**
     * Check if user is admin sistem
     */
    public function isAdminSistem()
    {
        return $this->role === 'admin_sistem';
    }

    /**
     * Check if user is admin DPC
     */
    public function isAdminDpc()
    {
        return $this->role === 'admin_dpc';
    }

    /**
     * Check if user is admin DPD
     */
    public function isAdminDpd()
    {
        return $this->role === 'admin_dpd';
    }

    /**
     * Check if user is anggota
     */
    public function isAnggota()
    {
        return $this->role === 'anggota';
    }

    /**
     * Check if user can access user management
     */
    public function canManageUsers()
    {
        return $this->role === 'superadmin';
    }

    /**
     * Check if user has full system access (except user management)
     */
    public function hasFullSystemAccess()
    {
        return in_array($this->role, ['admin_sistem', 'superadmin']);
    }

    /**
     * Get user's DPC scope
     */
    public function getDpcScope()
    {
        if ($this->isAdminDpc() && $this->dpc_id) {
            return $this->dpc_id;
        }
        if ($this->isAdminDpd() && $this->dpd_id) {
            // Admin DPD can access all DPCs under their DPD
            return Dpc::where('dpd_id', $this->dpd_id)->pluck('dpc_id')->toArray();
        }
        return null;
    }

    /**
     * Get user's DPD scope
     */
    public function getDpdScope()
    {
        if ($this->isAdminDpd() && $this->dpd_id) {
            return $this->dpd_id;
        }
        if ($this->isAdminDpc() && $this->dpc_id) {
            // Admin DPC's DPD is determined by their DPC
            return $this->dpc?->dpd_id;
        }
        return null;
    }

    /**
     * Check if user is active
     */
    public function isActive()
    {
        return $this->is_active;
    }

    /**
     * Get role badge color
     */
    public function getRoleBadgeColor()
    {
        return match($this->role) {
            'superadmin' => 'danger',
            'admin_sistem' => 'warning',
            'admin_dpc' => 'info',
            'admin_dpd' => 'primary',
            'anggota' => 'success',
            default => 'secondary'
        };
    }

    /**
     * Get role display name
     */
    public function getRoleDisplayName()
    {
        return match($this->role) {
            'superadmin' => 'Super Admin',
            'admin_sistem' => 'Admin Sistem',
            'admin_dpc' => 'Admin DPC',
            'admin_dpd' => 'Admin DPD',
            'anggota' => 'Anggota',
            default => 'Unknown'
        };
    }

    // ===== RELATIONSHIPS =====

    /**
     * Relasi ke DPC
     */
    public function dpc()
    {
        return $this->belongsTo(Dpc::class, 'dpc_id', 'dpc_id');
    }

    /**
     * Relasi ke DPD
     */
    public function dpd()
    {
        return $this->belongsTo(Dpd::class, 'dpd_id', 'dpd_id');
    }

    /**
     * Relasi ke Anggota
     */
    public function anggota()
    {
        return $this->belongsTo(Anggota::class, 'anggota_id', 'id');
    }
}
