<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Models\Anggota;
use App\Models\Upa;
use App\Models\Dpc;
use App\Models\User;

class RiwayatPerpindahanAnggota extends Model
{
    protected $table = 'tbl_riwayat_perpindahan_anggota';

    protected $fillable = [
        'tipe_transaksi',
        'anggota_id',
        'upa_asal_id',
        'upa_tujuan_id',
        'dpc_asal_id',
        'dpc_tujuan_id',
        'pembimbing_lama_id',
        'pembimbing_baru_id',
        'tanggal_efektif',
        'tahun_pelantikan',
        'nomor_surat',
        'alasan',
        'catatan',
        'metadata_json',
        'status',
        'created_by',
        'approved_by',
        'approved_at',
        'rekomposisi_batch_id',
    ];

    protected $casts = [
        'tanggal_efektif' => 'date',
        'approved_at' => 'datetime',
        'metadata_json' => 'array',
    ];

    // ===== RELATIONSHIPS =====

    /**
     * Relasi ke Anggota yang dipindahkan
     */
    public function anggota(): BelongsTo
    {
        return $this->belongsTo(Anggota::class, 'anggota_id', 'id');
    }

    /**
     * Relasi ke UPA Asal
     */
    public function upaAsal(): BelongsTo
    {
        return $this->belongsTo(Upa::class, 'upa_asal_id', 'upa_id');
    }

    /**
     * Relasi ke UPA Tujuan
     */
    public function upaTujuan(): BelongsTo
    {
        return $this->belongsTo(Upa::class, 'upa_tujuan_id', 'upa_id');
    }

    /**
     * Relasi ke DPC Asal
     */
    public function dpcAsal(): BelongsTo
    {
        return $this->belongsTo(Dpc::class, 'dpc_asal_id', 'dpc_id');
    }

    /**
     * Relasi ke DPC Tujuan
     */
    public function dpcTujuan(): BelongsTo
    {
        return $this->belongsTo(Dpc::class, 'dpc_tujuan_id', 'dpc_id');
    }

    /**
     * Relasi ke Pembimbing Lama
     */
    public function pembimbingLama(): BelongsTo
    {
        return $this->belongsTo(Anggota::class, 'pembimbing_lama_id', 'id');
    }

    /**
     * Relasi ke Pembimbing Baru
     */
    public function pembimbingBaru(): BelongsTo
    {
        return $this->belongsTo(Anggota::class, 'pembimbing_baru_id', 'id');
    }

    /**
     * Relasi ke User yang membuat transaksi
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by', 'id');
    }

    /**
     * Relasi ke User yang approve transaksi
     */
    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by', 'id');
    }

    // ===== SCOPES =====

    /**
     * Scope untuk filter berdasarkan tipe transaksi
     */
    public function scopeByTipe($query, $tipe)
    {
        return $query->where('tipe_transaksi', $tipe);
    }

    /**
     * Scope untuk filter berdasarkan anggota
     */
    public function scopeByAnggota($query, $anggotaId)
    {
        return $query->where('anggota_id', $anggotaId);
    }

    /**
     * Scope untuk filter status completed
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope untuk filter status pending
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    // ===== ACCESSORS & MUTATORS =====

    /**
     * Get formatted tipe transaksi
     */
    public function getTipeTransaksiLabelAttribute()
    {
        $labels = [
            'mutasi' => 'Mutasi Anggota',
            'penempatan' => 'Surat Penempatan',
            'perpindahan' => 'Perpindahan Anggota',
            'rekomposisi' => 'Rekomposisi UPA',
        ];

        return $labels[$this->tipe_transaksi] ?? $this->tipe_transaksi;
    }

    /**
     * Get status badge color
     */
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'draft' => 'secondary',
            'pending' => 'warning',
            'approved' => 'info',
            'completed' => 'success',
            'cancelled' => 'danger',
        ];

        return $badges[$this->status] ?? 'secondary';
    }
}
