<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Cache;

class OptimizedAnggota extends Model
{
    protected $table = 'tbl_anggota';
    
    protected $fillable = [
        'nomor_identitas_anggota',
        'nama_lengkap',
        'jenis_kelamin',
        'kota',
        'ref_jenjang_id',
        'ref_grup_upa_id',
        'dpc_id',
        'dpd_id',
        'status_keanggotaan',
        'email',
        'foto',
        'created_at'
    ];

    protected $casts = [
        'created_at' => 'datetime',
    ];

    // Optimized relationships with specific columns
    public function refJenjang()
    {
        return $this->belongsTo(RefJenjang::class, 'ref_jenjang_id', 'ref_jenjang_id')
                    ->select('ref_jenjang_id', 'jenjang');
    }

    public function refGrupUpa()
    {
        return $this->belongsTo(RefGrupUpa::class, 'ref_grup_upa_id', 'ref_grup_upa_id')
                    ->select('ref_grup_upa_id', 'nama_grup_upa');
    }

    public function dpc()
    {
        return $this->belongsTo(Dpc::class, 'dpc_id', 'dpc_id')
                    ->select('dpc_id', 'nama_dpc');
    }

    public function dpd()
    {
        return $this->belongsTo(Dpd::class, 'dpd_id', 'dpd_id')
                    ->select('dpd_id', 'nama_dpd');
    }

    // Optimized scopes
    public function scopeOptimizedList(Builder $query)
    {
        return $query->select([
            'id', 'nomor_identitas_anggota', 'nama_lengkap', 'jenis_kelamin', 
            'kota', 'ref_jenjang_id', 'ref_grup_upa_id', 'dpc_id', 'dpd_id', 
            'status_keanggotaan', 'email', 'foto', 'created_at'
        ])->with([
            'refJenjang',
            'refGrupUpa',
            'dpc',
            'dpd'
        ]);
    }

    public function scopeWithCache(Builder $query, string $key, int $ttl = 3600)
    {
        return Cache::remember($key, $ttl, function () use ($query) {
            return $query->get();
        });
    }

    public function scopePaginatedWithCache(Builder $query, string $key, int $perPage = 15, int $ttl = 300)
    {
        return Cache::remember($key, $ttl, function () use ($query, $perPage) {
            return $query->paginate($perPage);
        });
    }

    // Static methods for common queries
    public static function getStatusList()
    {
        return Cache::remember('anggota_status_list_optimized', 3600, function () {
            return static::select('status_keanggotaan')
                        ->distinct()
                        ->pluck('status_keanggotaan');
        });
    }

    public static function getProvinsiList()
    {
        return Cache::remember('anggota_provinsi_list_optimized', 3600, function () {
            return static::select('provinsi')
                        ->distinct()
                        ->whereNotNull('provinsi')
                        ->pluck('provinsi');
        });
    }

    // Optimized search method
    public function scopeSearch(Builder $query, string $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('nama_lengkap', 'like', "%{$search}%")
              ->orWhere('nomor_identitas_anggota', 'like', "%{$search}%")
              ->orWhere('email', 'like', "%{$search}%");
        });
    }

    // Optimized filter methods
    public function scopeFilterByStatus(Builder $query, string $status)
    {
        return $query->where('status_keanggotaan', $status);
    }

    public function scopeFilterByJenjang(Builder $query, int $jenjangId)
    {
        return $query->where('ref_jenjang_id', $jenjangId);
    }

    public function scopeFilterByProvinsi(Builder $query, string $provinsi)
    {
        return $query->where('provinsi', $provinsi);
    }

    public function scopeFilterByDpc(Builder $query, int $dpcId)
    {
        return $query->where('dpc_id', $dpcId);
    }

    public function scopeFilterByDpd(Builder $query, int $dpdId)
    {
        return $query->where('dpd_id', $dpdId);
    }

    // Performance monitoring
    public static function getPerformanceStats()
    {
        return Cache::remember('anggota_performance_stats', 3600, function () {
            return [
                'total_count' => static::count(),
                'active_count' => static::where('status_keanggotaan', 'Aktif')->count(),
                'last_updated' => now(),
            ];
        });
    }
}


