<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Carbon\Carbon;

class AnggotaJenjangHistory extends Model
{
    use HasFactory;

    protected $table = 'tbl_anggota_jenjang_history';

    protected $fillable = [
        'anggota_id',
        'ref_jenjang_id',
        'dpc_id',
        'dpd_id',
        'upa_id',
        'tanggal_perubahan',
        'keterangan',
        'user_id',
        'status_keanggotaan'
    ];

    protected $casts = [
        'tanggal_perubahan' => 'date',
    ];

    // ===== RELATIONSHIPS =====
    public function anggota()
    {
        return $this->belongsTo(Anggota::class, 'anggota_id');
    }

    public function refJenjang()
    {
        return $this->belongsTo(RefJenjang::class, 'ref_jenjang_id', 'ref_jenjang_id');
    }


    public function dpc()
    {
        return $this->belongsTo(Dpc::class, 'dpc_id', 'dpc_id');
    }

    public function dpd()
    {
        return $this->belongsTo(Dpd::class, 'dpd_id', 'dpd_id');
    }

    public function upa()
    {
        return $this->belongsTo(Upa::class, 'upa_id', 'upa_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    // ===== SCOPES =====
    public function scopeByAnggota($query, $anggotaId)
    {
        return $query->where('anggota_id', $anggotaId);
    }

    public function scopeByTanggal($query, $tanggal)
    {
        return $query->where('tanggal_perubahan', $tanggal);
    }

    public function scopeLatest($query)
    {
        return $query->orderBy('tanggal_perubahan', 'desc');
    }

    // ===== ACCESSORS =====
    public function getTanggalPerubahanFormattedAttribute()
    {
        return $this->tanggal_perubahan ? $this->tanggal_perubahan->format('d/m/Y') : null;
    }

    public function getKeteranganSingkatAttribute()
    {
        if (!$this->keterangan) return null;
        
        return strlen($this->keterangan) > 50 
            ? substr($this->keterangan, 0, 50) . '...' 
            : $this->keterangan;
    }

    // ===== STATIC METHODS =====
    public static function createHistory($anggotaId, $data, $keterangan = null, $userId = null)
    {
        return static::create([
            'anggota_id' => $anggotaId,
            'ref_jenjang_id' => $data['ref_jenjang_id'] ?? null,
            'ref_grup_upa_id' => $data['ref_grup_upa_id'] ?? null,
            'dpc_id' => $data['dpc_id'] ?? null,
            'dpd_id' => $data['dpd_id'] ?? null,
            'upa_id' => $data['upa_id'] ?? null,
            'tanggal_perubahan' => $data['tanggal_perubahan'] ?? now()->toDateString(),
            'status_keanggotaan' => $data['status_keanggotaan'] ?? null,
            'keterangan' => $keterangan,
            'user_id' => $userId ?? auth()->id(),
        ]);
    }

    public static function getLatestHistory($anggotaId)
    {
        return static::byAnggota($anggotaId)
            ->latest()
            ->first();
    }

    public static function getHistoryByDateRange($anggotaId, $startDate, $endDate)
    {
        return static::byAnggota($anggotaId)
            ->whereBetween('tanggal_perubahan', [$startDate, $endDate])
            ->latest()
            ->get();
    }
}
