<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use App\Casts\Encrypted;

class Anggota extends Model
{
    use HasFactory;

    protected $table = 'tbl_anggota';

    protected $fillable = [
        'nomor_identitas_anggota',
        'nama_lengkap',
        'tempat_lahir',
        'tanggal_lahir',
        'jenis_kelamin',
        'alamat_jalan',
        'rt',
        'rw',
        'kelurahan',
        'kecamatan',
        'kota',
        'provinsi',
        'kode_pos',
        'no_telepon',
        'email',
        'ref_jenjang_id',
        'ref_grup_upa_id',
        'kode_upa',
        'upa_id',
        'dpc_id',
        'dpd_id',
        'tanggal_bergabung',
        'status_keanggotaan',
        'penghasilan_per_bulan',
        'muzaki_mustahiq',
        'nomor_identitas_pasangan',
        'catatan_profil',
        'keahlian_lainnya',
        'foto',
        // Encrypted fields
        'nomor_identitas_anggota_encrypted',
        'alamat_jalan_encrypted',
        'rt_encrypted',
        'rw_encrypted',
        'kelurahan_encrypted',
        'no_telepon_encrypted',
        'email_encrypted',
        'penghasilan_per_bulan_encrypted',
        'nomor_identitas_pasangan_encrypted'
    ];

    protected $casts = [
        'tanggal_lahir' => 'date',
        'tanggal_bergabung' => 'date',
        'penghasilan_per_bulan' => 'decimal:2',
        // Encrypted fields
        'nomor_identitas_anggota_encrypted' => Encrypted::class,
        'alamat_jalan_encrypted' => Encrypted::class,
        'rt_encrypted' => Encrypted::class,
        'rw_encrypted' => Encrypted::class,
        'kelurahan_encrypted' => Encrypted::class,
        'no_telepon_encrypted' => Encrypted::class,
        'email_encrypted' => Encrypted::class,
        'penghasilan_per_bulan_encrypted' => Encrypted::class,
        'nomor_identitas_pasangan_encrypted' => Encrypted::class
    ];

    // ===== RELATIONSHIPS =====
    public function mutasi()
    {
        return $this->hasMany(Mutasi::class, 'anggota_id');
    }

    public function perpindahan()
    {
        return $this->hasMany(PerpindahanAnggota::class, 'anggota_id');
    }

    public function riwayatPerpindahan()
    {
        return $this->hasMany(RiwayatPerpindahanAnggota::class, 'anggota_id');
    }

    public function absensiKegiatan()
    {
        return $this->hasMany(AbsensiKegiatan::class, 'anggota_id');
    }

    public function pesertaTakwim()
    {
        return $this->hasMany(PesertaTakwim::class, 'anggota_id');
    }

    // Relasi ke RefJenjang
    public function refJenjang()
    {
        return $this->belongsTo(RefJenjang::class, 'ref_jenjang_id', 'ref_jenjang_id');
    }

    // Relasi ke RefGrupUpa
    public function refGrupUpa()
    {
        return $this->belongsTo(RefGrupUpa::class, 'ref_grup_upa_id', 'ref_grup_upa_id');
    }


    // Relasi ke UPA
    public function upa()
    {
        return $this->belongsTo(Upa::class, 'upa_id', 'upa_id');
    }

    // Relasi ke DPC
    public function dpc()
    {
        return $this->belongsTo(Dpc::class, 'dpc_id', 'dpc_id');
    }

    // Relasi ke DPD
    public function dpd()
    {
        return $this->belongsTo(Dpd::class, 'dpd_id', 'dpd_id');
    }

    // Relasi ke Keahlian (many-to-many)
    public function keahlian()
    {
        return $this->belongsToMany(RefKeahlian::class, 'tbl_anggota_keahlian', 'anggota_id', 'ref_keahlian_id');
    }

    // Relasi ke Aktivitas (one-to-many)
    public function aktivitas()
    {
        return $this->hasMany(AnggotaAktivitas::class, 'anggota_id');
    }

    // Relasi ke History Keanggotaan (one-to-many)
    public function jenjangHistory()
    {
        return $this->hasMany(AnggotaJenjangHistory::class, 'anggota_id');
    }

    public function aktivitasLuar()
    {
        return $this->belongsToMany(\App\Models\AktivitasLuar::class, 'anggota_aktivitas_luar', 'anggota_id', 'aktivitas_luar_id');
    }


    // Relasi ke UPA sebagai pembimbing (many-to-many)
    public function upaPembimbing()
    {
        return $this->belongsToMany(Upa::class, 'tbl_upa_pembimbing', 'anggota_id', 'upa_id')
                    ->withPivot(['jabatan', 'tanggal_mulai', 'tanggal_selesai', 'status', 'catatan'])
                    ->withTimestamps();
    }

    // Relasi ke UPA aktif sebagai pembimbing
    public function upaPembimbingAktif()
    {
        return $this->belongsToMany(Upa::class, 'tbl_upa_pembimbing', 'anggota_id', 'upa_id')
                    ->wherePivot('status', 'Aktif')
                    ->withPivot(['jabatan', 'tanggal_mulai', 'tanggal_selesai', 'status', 'catatan'])
                    ->withTimestamps();
    }

    // ===== ENCRYPTED ACCESSORS =====
    public function getNomorIdentitasAnggotaAttribute()
    {
        // Prioritas: gunakan data asli jika ada, jika tidak ada baru gunakan data terenkripsi
        if (isset($this->attributes['nomor_identitas_anggota']) && $this->attributes['nomor_identitas_anggota']) {
            return $this->attributes['nomor_identitas_anggota'];
        }
        
        // Fallback: cek data terenkripsi
        if (isset($this->attributes['nomor_identitas_anggota_encrypted']) && $this->attributes['nomor_identitas_anggota_encrypted']) {
            try {
                return \Illuminate\Support\Facades\Crypt::decryptString($this->attributes['nomor_identitas_anggota_encrypted']);
            } catch (\Exception $e) {
                return null;
            }
        }
        
        return null;
    }

    public function getAlamatJalanAttribute()
    {
        if (isset($this->attributes['alamat_jalan']) && $this->attributes['alamat_jalan']) {
            return $this->attributes['alamat_jalan'];
        }
        
        if (isset($this->attributes['alamat_jalan_encrypted']) && $this->attributes['alamat_jalan_encrypted']) {
            try {
                return \Illuminate\Support\Facades\Crypt::decryptString($this->attributes['alamat_jalan_encrypted']);
            } catch (\Exception $e) {
                return null;
            }
        }
        return null;
    }

    public function getRtAttribute()
    {
        if (isset($this->attributes['rt']) && $this->attributes['rt']) {
            return $this->attributes['rt'];
        }
        
        if (isset($this->attributes['rt_encrypted']) && $this->attributes['rt_encrypted']) {
            try {
                return \Illuminate\Support\Facades\Crypt::decryptString($this->attributes['rt_encrypted']);
            } catch (\Exception $e) {
                return null;
            }
        }
        return null;
    }

    public function getRwAttribute()
    {
        if (isset($this->attributes['rw']) && $this->attributes['rw']) {
            return $this->attributes['rw'];
        }
        
        if (isset($this->attributes['rw_encrypted']) && $this->attributes['rw_encrypted']) {
            try {
                return \Illuminate\Support\Facades\Crypt::decryptString($this->attributes['rw_encrypted']);
            } catch (\Exception $e) {
                return null;
            }
        }
        return null;
    }

    public function getKelurahanAttribute()
    {
        if (isset($this->attributes['kelurahan']) && $this->attributes['kelurahan']) {
            return $this->attributes['kelurahan'];
        }
        
        if (isset($this->attributes['kelurahan_encrypted']) && $this->attributes['kelurahan_encrypted']) {
            try {
                return \Illuminate\Support\Facades\Crypt::decryptString($this->attributes['kelurahan_encrypted']);
            } catch (\Exception $e) {
                return null;
            }
        }
        return null;
    }

    public function getNoTeleponAttribute()
    {
        if (isset($this->attributes['no_telepon']) && $this->attributes['no_telepon']) {
            return $this->attributes['no_telepon'];
        }
        
        if (isset($this->attributes['no_telepon_encrypted']) && $this->attributes['no_telepon_encrypted']) {
            try {
                return \Illuminate\Support\Facades\Crypt::decryptString($this->attributes['no_telepon_encrypted']);
            } catch (\Exception $e) {
                return null;
            }
        }
        return null;
    }

    public function getEmailAttribute()
    {
        if (isset($this->attributes['email']) && $this->attributes['email']) {
            return $this->attributes['email'];
        }
        
        if (isset($this->attributes['email_encrypted']) && $this->attributes['email_encrypted']) {
            try {
                return \Illuminate\Support\Facades\Crypt::decryptString($this->attributes['email_encrypted']);
            } catch (\Exception $e) {
                return null;
            }
        }
        return null;
    }

    public function getPenghasilanPerBulanAttribute()
    {
        if (isset($this->attributes['penghasilan_per_bulan']) && $this->attributes['penghasilan_per_bulan']) {
            return $this->attributes['penghasilan_per_bulan'];
        }
        
        if (isset($this->attributes['penghasilan_per_bulan_encrypted']) && $this->attributes['penghasilan_per_bulan_encrypted']) {
            try {
                return \Illuminate\Support\Facades\Crypt::decryptString($this->attributes['penghasilan_per_bulan_encrypted']);
            } catch (\Exception $e) {
                return null;
            }
        }
        return null;
    }

    public function getNomorIdentitasPasanganAttribute()
    {
        if (isset($this->attributes['nomor_identitas_pasangan']) && $this->attributes['nomor_identitas_pasangan']) {
            return $this->attributes['nomor_identitas_pasangan'];
        }
        
        if (isset($this->attributes['nomor_identitas_pasangan_encrypted']) && $this->attributes['nomor_identitas_pasangan_encrypted']) {
            try {
                return \Illuminate\Support\Facades\Crypt::decryptString($this->attributes['nomor_identitas_pasangan_encrypted']);
            } catch (\Exception $e) {
                return null;
            }
        }
        return null;
    }

    // ===== ENCRYPTED MUTATORS =====
    public function setNomorIdentitasAnggotaAttribute($value)
    {
        $this->attributes['nomor_identitas_anggota'] = $value;
        $this->attributes['nomor_identitas_anggota_encrypted'] = $value ? \Illuminate\Support\Facades\Crypt::encryptString($value) : null;
    }

    public function setAlamatJalanAttribute($value)
    {
        $this->attributes['alamat_jalan'] = $value;
        $this->attributes['alamat_jalan_encrypted'] = $value ? \Illuminate\Support\Facades\Crypt::encryptString($value) : null;
    }

    public function setRtAttribute($value)
    {
        $this->attributes['rt'] = $value;
        $this->attributes['rt_encrypted'] = $value ? \Illuminate\Support\Facades\Crypt::encryptString($value) : null;
    }

    public function setRwAttribute($value)
    {
        $this->attributes['rw'] = $value;
        $this->attributes['rw_encrypted'] = $value ? \Illuminate\Support\Facades\Crypt::encryptString($value) : null;
    }

    public function setKelurahanAttribute($value)
    {
        $this->attributes['kelurahan'] = $value;
        $this->attributes['kelurahan_encrypted'] = $value ? \Illuminate\Support\Facades\Crypt::encryptString($value) : null;
    }

    public function setNoTeleponAttribute($value)
    {
        $cleanValue = preg_replace('/[^0-9+]/', '', $value);
        $this->attributes['no_telepon'] = $cleanValue;
        $this->attributes['no_telepon_encrypted'] = $cleanValue ? \Illuminate\Support\Facades\Crypt::encryptString($cleanValue) : null;
    }

    public function setEmailAttribute($value)
    {
        $cleanValue = strtolower(trim($value));
        $this->attributes['email'] = $cleanValue;
        $this->attributes['email_encrypted'] = $cleanValue ? \Illuminate\Support\Facades\Crypt::encryptString($cleanValue) : null;
    }

    public function setPenghasilanPerBulanAttribute($value)
    {
        $this->attributes['penghasilan_per_bulan'] = $value;
        $this->attributes['penghasilan_per_bulan_encrypted'] = $value ? \Illuminate\Support\Facades\Crypt::encryptString($value) : null;
    }

    public function setNomorIdentitasPasanganAttribute($value)
    {
        $this->attributes['nomor_identitas_pasangan'] = $value;
        $this->attributes['nomor_identitas_pasangan_encrypted'] = $value ? \Illuminate\Support\Facades\Crypt::encryptString($value) : null;
    }

    // ===== ACCESSORS (untuk view show) =====
    public function getNiaAttribute()
    {
        return $this->nomor_identitas_anggota;
    }

    public function getUmurAttribute()
    {
        if (!$this->tanggal_lahir) return null;
        
        // Cache umur calculation untuk menghindari perhitungan berulang
        return cache()->remember("anggota_umur_{$this->id}", 3600, function() {
            try {
                return Carbon::parse($this->tanggal_lahir)->age;
            } catch (\Exception $e) {
                return null;
            }
        });
    }

    public function getLamaKeanggotaanAttribute()
    {
        if (!$this->tanggal_bergabung) return null;
        
        // Cache lama keanggotaan calculation
        return cache()->remember("anggota_lama_keanggotaan_{$this->id}", 3600, function() {
            try {
                return Carbon::parse($this->tanggal_bergabung)->diffForHumans();
            } catch (\Exception $e) {
                return null;
            }
        });
    }

    public function getAlamatLengkapAttribute()
    {
        $alamatParts = array_filter([
            $this->alamat_jalan,
            $this->rt && $this->rw ? "RT {$this->rt}/RW {$this->rw}" : null,
            $this->kelurahan,
            $this->kecamatan,
            $this->kota,
            $this->provinsi
        ]);
        
        return implode(', ', $alamatParts);
    }

    // ===== HELPER METHODS (untuk view show) =====
    public function getTotalKehadiran()
    {
        try {
            return $this->absensiKegiatan()
                ->where('status_kehadiran', 'Hadir')
                ->count();
        } catch (\Exception $e) {
            return 0;
        }
    }

    public function getRiwayatMutasiTerakhir()
    {
        try {
            return $this->mutasi()
                ->latest('tanggal_mutasi')
                ->first();
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getStatusBadgeClass()
    {
        return match($this->status_keanggotaan) {
            'Aktif' => 'success',
            'Tidak Aktif' => 'warning',
            'Diberhentikan' => 'danger',
            'Pindah' => 'info',
            default => 'secondary'
        };
    }

    // ===== SCOPES =====
    public function scopeAktif($query)
    {
        return $query->where('status_keanggotaan', 'Aktif');
    }

    public function scopeByJenjang($query, $jenjang)
    {
        return $query->where('jenjang_anggota', $jenjang);
    }

    public function scopeByProvinsi($query, $provinsi)
    {
        return $query->where('provinsi', $provinsi);
    }

    public function scopeByKota($query, $kota)
    {
        return $query->where('kota', $kota);
    }

    public function scopeMuzaki($query)
    {
        return $query->where('muzaki_mustahiq', 'Muzaki');
    }

    public function scopeMustahiq($query)
    {
        return $query->where('muzaki_mustahiq', 'Mustahiq');
    }

    // ===== UTILITY METHODS =====
    public function getJenisKelaminLengkap()
    {
        return $this->jenis_kelamin == 'L' ? 'Laki-laki' : 'Perempuan';
    }

    public function hasPhoto()
    {
        return !empty($this->foto);
    }

    public function getPhotoUrl()
    {
        if ($this->hasPhoto()) {
            $url = \Storage::url($this->foto);
            // Add cache busting parameter to force browser reload when photo changes
            return $url . '?v=' . $this->updated_at->timestamp;
        }
        return null;
    }

    

    // ===== MUTATORS =====
    // Mutators sudah dipindahkan ke bagian ENCRYPTED MUTATORS di atas

    // ===== BOOT METHOD =====
    protected static function boot()
    {
        parent::boot();
        
        // Auto-generate NIA if not provided
        static::creating(function ($anggota) {
            if (empty($anggota->nomor_identitas_anggota)) {
                // Generate NIA berdasarkan DPC jika ada
                $dpcId = $anggota->dpc_id ?? null;
                $anggota->nomor_identitas_anggota = static::generateNIA($dpcId);
            }
        });
    }

    // ===== STATIC METHODS =====
    public static function generateNIA($dpcId = null)
    {
        $tahun = date('Y');
        
        if ($dpcId) {
            // Generate NIA per DPC
            $counter = static::where('dpc_id', $dpcId)
                            ->whereYear('created_at', $tahun)
                            ->count() + 1;
            $paddedCounter = str_pad($counter, 3, '0', STR_PAD_LEFT);
            
            // Get DPC code for NIA format
            $dpc = \App\Models\Dpc::find($dpcId);
            $dpcCode = $dpc ? $dpc->kode_dpc : 'DPC';
            
            return "PKS-{$dpcCode}-{$paddedCounter}-{$tahun}";
        } else {
            // Fallback: Generate NIA global (untuk backward compatibility)
            $counter = static::whereYear('created_at', $tahun)->count() + 1;
            $paddedCounter = str_pad($counter, 3, '0', STR_PAD_LEFT);
            
            return "PKS-{$paddedCounter}-{$tahun}";
        }
    }
    
    /**
     * Generate NIA untuk DPC tertentu
     */
    public static function generateNIAForDpc($dpcId)
    {
        return static::generateNIA($dpcId);
    }
    
    /**
     * Get next NIA number for DPC (untuk preview)
     */
    public static function getNextNIAForDpc($dpcId)
    {
        $tahun = date('Y');
        $counter = static::where('dpc_id', $dpcId)
                        ->whereYear('created_at', $tahun)
                        ->count() + 1;
        $paddedCounter = str_pad($counter, 3, '0', STR_PAD_LEFT);
        
        // Get DPC code for NIA format
        $dpc = \App\Models\Dpc::find($dpcId);
        $dpcCode = $dpc ? $dpc->kode_dpc : 'DPC';
        
        return "PKS-{$dpcCode}-{$paddedCounter}-{$tahun}";
    }

    // ===== HISTORY MANAGEMENT METHODS =====
    /**
     * Update data keanggotaan dan simpan ke history
     */
    public function updateKeanggotaanData($data, $keterangan = null)
    {
        // Simpan data lama ke history sebelum update
        $this->saveToHistory($keterangan);
        
        // Update data baru
        $this->update([
            'ref_jenjang_id' => $data['ref_jenjang_id'] ?? $this->ref_jenjang_id,
            'ref_grup_upa_id' => $data['ref_grup_upa_id'] ?? $this->ref_grup_upa_id,
            'dpc_id' => $data['dpc_id'] ?? $this->dpc_id,
            'dpd_id' => $data['dpd_id'] ?? $this->dpd_id,
            'upa_id' => $data['upa_id'] ?? $this->upa_id,
            'status_keanggotaan' => $data['status_keanggotaan'] ?? $this->status_keanggotaan,
        ]);
    }

    /**
     * Simpan data saat ini ke history
     */
    public function saveToHistory($keterangan = null)
    {
        return AnggotaJenjangHistory::createHistory($this->id, [
            'ref_jenjang_id' => $this->ref_jenjang_id,
            'ref_grup_upa_id' => $this->ref_grup_upa_id,
            'dpc_id' => $this->dpc_id,
            'dpd_id' => $this->dpd_id,
            'upa_id' => $this->upa_id,
            'status_keanggotaan' => $this->status_keanggotaan,
            'tanggal_perubahan' => now()->toDateString(),
        ], $keterangan);
    }

    /**
     * Get latest history
     */
    public function getLatestHistory()
    {
        return $this->jenjangHistory()->latest()->first();
    }

    /**
     * Get history by date range
     */
    public function getHistoryByDateRange($startDate, $endDate)
    {
        return $this->jenjangHistory()
            ->whereBetween('tanggal_perubahan', [$startDate, $endDate])
            ->latest()
            ->get();
    }

    // ===== URL HELPER METHODS =====
    
    /**
     * Generate URL slug untuk anggota
     */
    public function getUrlSlugAttribute()
    {
        return anggota_url($this->id, $this->nama_lengkap);
    }
    
    /**
     * Generate route URL untuk show
     */
    public function getShowUrlAttribute()
    {
        return anggota_route('anggota.show', $this->id, $this->nama_lengkap);
    }
    
    /**
     * Generate route URL untuk edit
     */
    public function getEditUrlAttribute()
    {
        return anggota_route('anggota.edit', $this->id, $this->nama_lengkap);
    }
    
    /**
     * Generate route URL untuk delete
     */
    public function getDeleteUrlAttribute()
    {
        return anggota_route('anggota.destroy', $this->id, $this->nama_lengkap);
    }
    
    /**
     * Find anggota by URL slug
     */
    public static function findBySlug($slug)
    {
        $parsed = \App\Helpers\UrlHelper::parseAnggotaUrl($slug);
        
        if (!$parsed) {
            return null;
        }
        
        return static::find($parsed['id']);
    }

    // ===== DEBUGGING METHODS =====
    public function toArray()
    {
        $array = parent::toArray();
        
        // Add computed attributes for debugging
        $array['umur'] = $this->umur;
        $array['alamat_lengkap'] = $this->alamat_lengkap;
        $array['lama_keanggotaan'] = $this->lama_keanggotaan;
        
        return $array;
    }
}