<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Dpc;
use App\Models\Dpd;
use App\Models\Anggota;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class UserManagementController extends Controller
{
    public function __construct()
    {
        // Middleware akan diterapkan di routes
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $users = User::orderBy('created_at', 'desc')->paginate(10);
        return view('user-management.index', compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $roles = [
            'admin_sistem' => 'Admin Sistem',
            'admin_dpc' => 'Admin DPC',
            'admin_dpd' => 'Admin DPD',
            'anggota' => 'Anggota'
        ];

        // Superadmin bisa membuat superadmin juga
        if (auth()->user()->isSuperAdmin()) {
            $roles['superadmin'] = 'Super Admin';
        }

        // Get DPC dan DPD lists untuk dropdown
        $dpcs = Dpc::where('is_active', true)->orderBy('nama_dpc')->get();
        $dpds = Dpd::where('is_active', true)->orderBy('nama_dpd')->get();
        $anggotas = collect(); // Will be populated based on role selection via AJAX

        return view('user-management.create', compact('roles', 'dpcs', 'dpds', 'anggotas'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Validasi password dengan detail lengkap
        $password = $request->input('password');
        $passwordErrors = [];
        
        if (strlen($password) < 8) {
            $passwordErrors[] = '1. Minimal 8 karakter';
        }
        if (!preg_match('/[a-z]/', $password)) {
            $passwordErrors[] = '2. Harus mengandung huruf kecil';
        }
        if (!preg_match('/[A-Z]/', $password)) {
            $passwordErrors[] = '3. Harus mengandung huruf besar';
        }
        if (!preg_match('/[0-9]/', $password)) {
            $passwordErrors[] = '4. Harus mengandung angka';
        }
        if (!preg_match('/[@$!%*?&]/', $password)) {
            $passwordErrors[] = '5. Harus mengandung simbol khusus (@$!%*?&)';
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255|regex:/^[a-zA-Z\s]+$/',
            'email' => 'required|string|email|max:255|unique:users|regex:/^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/',
            'password' => 'required|string|min:8|max:255|confirmed|regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]/',
            'role' => ['required', Rule::in(['admin_sistem', 'superadmin', 'admin_dpc', 'admin_dpd', 'anggota'])],
            'dpc_id' => 'nullable|exists:tbl_dpc,dpc_id',
            'dpd_id' => 'nullable|exists:tbl_dpd,dpd_id',
            'anggota_id' => 'nullable|exists:tbl_anggota,id',
            'is_active' => 'boolean'
        ], [
            'name.required' => 'Nama wajib diisi.',
            'name.regex' => 'Nama hanya boleh mengandung huruf dan spasi.',
            'email.required' => 'Email wajib diisi.',
            'email.email' => 'Format email tidak valid.',
            'email.unique' => 'Email sudah digunakan.',
            'email.regex' => 'Format email tidak valid.',
            'password.required' => 'Password wajib diisi.',
            'password.min' => empty($passwordErrors) ? 'Password minimal 8 karakter.' : 'Password tidak memenuhi syarat: ' . implode(', ', $passwordErrors),
            'password.regex' => empty($passwordErrors) ? 'Password harus mengandung huruf besar, huruf kecil, angka, dan simbol khusus.' : 'Password tidak memenuhi syarat: ' . implode(', ', $passwordErrors),
            'role.required' => 'Role wajib dipilih.',
            'role.in' => 'Role tidak valid.',
            'dpc_id.exists' => 'DPC yang dipilih tidak valid.',
            'dpd_id.exists' => 'DPD yang dipilih tidak valid.',
            'anggota_id.exists' => 'Anggota yang dipilih tidak valid.',
        ]);
        
        // Jika ada error password, return dengan error message yang lengkap
        if (!empty($passwordErrors)) {
            return back()->withErrors([
                'password' => 'Password tidak memenuhi syarat berikut: ' . implode(', ', $passwordErrors)
            ])->withInput();
        }

        // Cek permission untuk membuat superadmin
        if ($validated['role'] === 'superadmin' && !auth()->user()->isSuperAdmin()) {
            return back()->withErrors(['role' => 'Anda tidak memiliki izin untuk membuat Super Admin.']);
        }

        // Validasi berdasarkan role
        if ($validated['role'] === 'admin_dpc' && empty($validated['dpc_id'])) {
            return back()->withErrors(['dpc_id' => 'DPC wajib dipilih untuk Admin DPC.']);
        }

        if ($validated['role'] === 'admin_dpd' && empty($validated['dpd_id'])) {
            return back()->withErrors(['dpd_id' => 'DPD wajib dipilih untuk Admin DPD.']);
        }

        if ($validated['role'] === 'anggota' && empty($validated['anggota_id'])) {
            return back()->withErrors(['anggota_id' => 'Anggota wajib dipilih untuk role Anggota.']);
        }

        // Untuk admin_dpc, set dpd_id dari dpc_id
        if ($validated['role'] === 'admin_dpc' && $validated['dpc_id']) {
            $dpc = Dpc::find($validated['dpc_id']);
            if ($dpc && $dpc->dpd_id) {
                $validated['dpd_id'] = $dpc->dpd_id;
            }
        }

        // Untuk anggota, set dpc_id dan dpd_id dari anggota_id
        if ($validated['role'] === 'anggota' && $validated['anggota_id']) {
            $anggota = Anggota::find($validated['anggota_id']);
            if ($anggota) {
                $validated['dpc_id'] = $anggota->dpc_id;
                $validated['dpd_id'] = $anggota->dpd_id;
            }
        }

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => $validated['role'],
            'dpc_id' => $validated['dpc_id'] ?? null,
            'dpd_id' => $validated['dpd_id'] ?? null,
            'anggota_id' => $validated['anggota_id'] ?? null,
            'is_active' => $request->has('is_active') ? true : false,
        ]);

        return redirect()->route('user-management.index')
            ->with('success', 'User berhasil dibuat!');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        return view('user-management.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $roles = [
            'admin_sistem' => 'Admin Sistem',
            'admin_dpc' => 'Admin DPC',
            'admin_dpd' => 'Admin DPD',
            'anggota' => 'Anggota'
        ];

        // Superadmin bisa mengubah role ke superadmin juga
        if (auth()->user()->isSuperAdmin()) {
            $roles['superadmin'] = 'Super Admin';
        }

        // Get DPC dan DPD lists untuk dropdown
        $dpcs = Dpc::where('is_active', true)->orderBy('nama_dpc')->get();
        $dpds = Dpd::where('is_active', true)->orderBy('nama_dpd')->get();
        
        // Get anggota list based on user's current role
        $anggotas = collect();
        if ($user->role === 'anggota' || old('role') === 'anggota') {
            $anggotas = Anggota::select('anggota_id', 'nama_lengkap', 'nomor_identitas_anggota')
                ->orderBy('nama_lengkap')
                ->get();
        }

        return view('user-management.edit', compact('user', 'roles', 'dpcs', 'dpds', 'anggotas'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'password' => 'nullable|string|min:8|confirmed|regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]/',
            'role' => ['required', Rule::in(['admin_sistem', 'superadmin', 'admin_dpc', 'admin_dpd', 'anggota'])],
            'dpc_id' => 'nullable|exists:tbl_dpc,dpc_id',
            'dpd_id' => 'nullable|exists:tbl_dpd,dpd_id',
            'anggota_id' => 'nullable|exists:tbl_anggota,id',
            'is_active' => 'boolean'
        ], [
            'password.regex' => 'Password harus mengandung huruf besar, huruf kecil, angka, dan simbol khusus.',
            'dpc_id.exists' => 'DPC yang dipilih tidak valid.',
            'dpd_id.exists' => 'DPD yang dipilih tidak valid.',
            'anggota_id.exists' => 'Anggota yang dipilih tidak valid.',
        ]);

        // Cek permission untuk mengubah ke superadmin
        if ($validated['role'] === 'superadmin' && !auth()->user()->isSuperAdmin()) {
            return back()->withErrors(['role' => 'Anda tidak memiliki izin untuk mengubah role ke Super Admin.']);
        }

        // Cek permission untuk mengubah superadmin lain
        if ($user->isSuperAdmin() && !auth()->user()->isSuperAdmin()) {
            return back()->withErrors(['role' => 'Anda tidak memiliki izin untuk mengubah Super Admin.']);
        }

        // Validasi berdasarkan role
        if ($validated['role'] === 'admin_dpc' && empty($validated['dpc_id'])) {
            return back()->withErrors(['dpc_id' => 'DPC wajib dipilih untuk Admin DPC.']);
        }

        if ($validated['role'] === 'admin_dpd' && empty($validated['dpd_id'])) {
            return back()->withErrors(['dpd_id' => 'DPD wajib dipilih untuk Admin DPD.']);
        }

        if ($validated['role'] === 'anggota' && empty($validated['anggota_id'])) {
            return back()->withErrors(['anggota_id' => 'Anggota wajib dipilih untuk role Anggota.']);
        }

        // Untuk admin_dpc, set dpd_id dari dpc_id
        if ($validated['role'] === 'admin_dpc' && $validated['dpc_id']) {
            $dpc = Dpc::find($validated['dpc_id']);
            if ($dpc && $dpc->dpd_id) {
                $validated['dpd_id'] = $dpc->dpd_id;
            }
        }

        // Untuk anggota, set dpc_id dan dpd_id dari anggota_id
        if ($validated['role'] === 'anggota' && $validated['anggota_id']) {
            $anggota = Anggota::find($validated['anggota_id']);
            if ($anggota) {
                $validated['dpc_id'] = $anggota->dpc_id;
                $validated['dpd_id'] = $anggota->dpd_id;
            }
        }

        // Clear scope fields jika role tidak memerlukannya
        if (!in_array($validated['role'], ['admin_dpc', 'admin_dpd', 'anggota'])) {
            $validated['dpc_id'] = null;
            $validated['dpd_id'] = null;
            $validated['anggota_id'] = null;
        }

        $user->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'role' => $validated['role'],
            'dpc_id' => $validated['dpc_id'] ?? null,
            'dpd_id' => $validated['dpd_id'] ?? null,
            'anggota_id' => $validated['anggota_id'] ?? null,
            'is_active' => $request->has('is_active') ? true : false,
        ]);

        // Update password jika diisi
        if (!empty($validated['password'])) {
            $user->update([
                'password' => Hash::make($validated['password'])
            ]);
        }

        return redirect()->route('user-management.index')
            ->with('success', 'User berhasil diperbarui!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        // Cek permission untuk menghapus superadmin
        if ($user->isSuperAdmin() && !auth()->user()->isSuperAdmin()) {
            return back()->withErrors(['error' => 'Anda tidak memiliki izin untuk menghapus Super Admin.']);
        }

        // Tidak bisa menghapus diri sendiri
        if ($user->id === auth()->id()) {
            return back()->withErrors(['error' => 'Anda tidak dapat menghapus akun sendiri.']);
        }

        $user->delete();

        return redirect()->route('user-management.index')
            ->with('success', 'User berhasil dihapus!');
    }

    /**
     * Toggle user active status
     */
    public function toggleStatus(User $user_management)
    {
        // Cek permission untuk mengubah status superadmin
        if ($user_management->isSuperAdmin() && !auth()->user()->isSuperAdmin()) {
            return back()->withErrors(['error' => 'Anda tidak memiliki izin untuk mengubah status Super Admin.']);
        }

        // Tidak bisa menonaktifkan diri sendiri
        if ($user_management->id === auth()->id()) {
            return back()->withErrors(['error' => 'Anda tidak dapat menonaktifkan akun sendiri.']);
        }

        $user_management->update([
            'is_active' => !$user_management->is_active
        ]);

        $status = $user_management->is_active ? 'diaktifkan' : 'dinonaktifkan';
        
        return back()->with('success', "User berhasil {$status}!");
    }
}
