<?php

namespace App\Http\Controllers;

use App\Models\UpaGroupActivity;
use App\Models\UpaGroupAttendance;
use App\Models\Upa;
use App\Models\Anggota;
use App\Models\UpaAgendaMaster;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\UpaGroupActivityExport;

class UpaGroupActivityController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = UpaGroupActivity::query();
        
        // Filter berdasarkan pencarian
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('kode_upa', 'like', "%{$search}%")
                  ->orWhere('nama_pembimbing', 'like', "%{$search}%")
                  ->orWhere('agenda_upa', 'like', "%{$search}%");
            });
        }
        
        // Filter berdasarkan status
        if ($request->has('status') && $request->status) {
            $query->where('status_terlaksana', $request->status);
        }
        
        // Filter berdasarkan tanggal
        if ($request->has('tanggal_dari') && $request->tanggal_dari) {
            $query->where('tanggal_upa', '>=', $request->tanggal_dari);
        }
        
        if ($request->has('tanggal_sampai') && $request->tanggal_sampai) {
            $query->where('tanggal_upa', '<=', $request->tanggal_sampai);
        }
        
        $activities = $query->with(['upa', 'attendances.anggota'])
                          ->orderBy('tanggal_upa', 'desc')
                          ->paginate(20);
        
        // Data untuk filter dropdown
        $statusList = ['Terlaksana', 'Tidak Terlaksana'];
        $upaList = Upa::select('kode_upa', 'nama_upa')->get();
        
        return view('upa-group-activity.index', compact('activities', 'statusList', 'upaList'));
    }

    /**
     * Export daftar aktivitas UPA ke Excel sesuai filter.
     */
    public function export(Request $request)
    {
        // Gunakan query yang sama dengan index() untuk konsistensi filter
        $query = UpaGroupActivity::query();

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('kode_upa', 'like', "%{$search}%")
                  ->orWhere('nama_pembimbing', 'like', "%{$search}%")
                  ->orWhere('agenda_upa', 'like', "%{$search}%");
            });
        }

        if ($request->has('status') && $request->status) {
            $query->where('status_terlaksana', $request->status);
        }

        if ($request->has('tanggal_dari') && $request->tanggal_dari) {
            $query->where('tanggal_upa', '>=', $request->tanggal_dari);
        }

        if ($request->has('tanggal_sampai') && $request->tanggal_sampai) {
            $query->where('tanggal_upa', '<=', $request->tanggal_sampai);
        }

        $activities = $query->with(['upa', 'attendances.anggota'])
                            ->orderBy('tanggal_upa', 'desc')
                            ->get();

        $filename = 'aktivitas_upa_' . date('Y-m-d_H-i-s') . '.xlsx';

        return Excel::download(new UpaGroupActivityExport($activities), $filename);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $upaList = Upa::where('status', 'Aktif')
                     ->orderBy('nama_upa')
                     ->get(['upa_id', 'kode_upa', 'nama_upa']);
        
        $agendaList = UpaAgendaMaster::active()
                                   ->orderBy('nama_agenda')
                                   ->get();
        
        return view('upa-group-activity.create', compact('upaList', 'agendaList'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'kode_upa' => 'required|string|max:50',
            'nama_pembimbing' => 'required|string|max:255',
            'tanggal_upa' => 'required|date',
            'pekan' => 'nullable|integer',
            'waktu_upa' => 'required|date_format:H:i',
            'kehadiran_pembimbing' => 'boolean',
            'agenda_upa' => 'nullable|string',
            'agenda_khusus' => 'nullable|string',
            'agenda_notes_json' => 'nullable|string',
            'status_terlaksana' => 'required|in:Terlaksana,Tidak Terlaksana',
            'catatan' => 'nullable|string',
            'anggota_kehadiran' => 'nullable|array',
            'anggota_kehadiran.*' => 'in:Hadir,Tidak Hadir'
        ]);

       /// print_r($validated);
       /// exit();

        // Jika ada agenda khusus, gunakan agenda khusus, jika tidak gunakan agenda master
        if (!empty($validated['agenda_khusus'])) {
            $validated['agenda_upa'] = $validated['agenda_khusus'];
        }
        unset($validated['agenda_khusus']);

        // Parse agenda_notes_json dan simpan ke agenda_notes
        if (!empty($validated['agenda_notes_json'])) {
            $decoded = json_decode($validated['agenda_notes_json'], true);
            if (is_array($decoded) && count($decoded) > 0) {
                $validated['agenda_notes'] = $decoded;
            }
        }
        unset($validated['agenda_notes_json']);

        // Debug log untuk melihat data yang akan disimpan
        \Log::info('Storing UPA Activity', [
            'pekan' => $validated['pekan'] ?? 'null',
            'agenda_notes' => $validated['agenda_notes'] ?? 'null',
            'agenda_notes_count' => isset($validated['agenda_notes']) ? count($validated['agenda_notes']) : 0
        ]);

        $activity = UpaGroupActivity::create($validated);

        // Simpan kehadiran anggota jika ada
        if ($request->has('anggota_kehadiran')) {
            $this->saveAttendance($activity, $request->anggota_kehadiran);
        }

        return redirect()->route('upa-group-activity.show', $activity)
                        ->with('success', 'Aktivitas UPA berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(UpaGroupActivity $upaGroupActivity)
    {
        $upaGroupActivity->load([
            'upa',
            'attendances.anggota' => function($query) {
                $query->orderBy('nama_lengkap');
            }
        ]);
        
        return view('upa-group-activity.show', compact('upaGroupActivity'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(UpaGroupActivity $upaGroupActivity)
    {
        $upaList = Upa::where('status', 'Aktif')
                     ->orderBy('nama_upa')
                     ->get(['upa_id', 'kode_upa', 'nama_upa']);
        
        $agendaList = UpaAgendaMaster::active()
                                   ->orderBy('nama_agenda')
                                   ->get();
        
        $upaGroupActivity->load(['attendances.anggota']);
        
        return view('upa-group-activity.edit', compact('upaGroupActivity', 'upaList', 'agendaList'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, UpaGroupActivity $upaGroupActivity)
    {
        $validated = $request->validate([
            'kode_upa' => 'required|string|max:50',
            'nama_pembimbing' => 'required|string|max:255',
            'tanggal_upa' => 'required|date',
            'pekan' => 'nullable|integer',
            'waktu_upa' => 'required|date_format:H:i',
            'kehadiran_pembimbing' => 'boolean',
            'agenda_upa' => 'nullable|string',
            'agenda_khusus' => 'nullable|string',
            'agenda_notes_json' => 'nullable|string',
            'status_terlaksana' => 'required|in:Terlaksana,Tidak Terlaksana',
            'catatan' => 'nullable|string',
            'anggota_kehadiran' => 'nullable|array',
            'anggota_kehadiran.*' => 'in:Hadir,Tidak Hadir'
        ]);

        // Jika ada agenda khusus, gunakan agenda khusus, jika tidak gunakan agenda master
        if (!empty($validated['agenda_khusus'])) {
            $validated['agenda_upa'] = $validated['agenda_khusus'];
        }
        unset($validated['agenda_khusus']);

        // Parse agenda_notes_json dan simpan ke agenda_notes
        if (!empty($validated['agenda_notes_json'])) {
            $decoded = json_decode($validated['agenda_notes_json'], true);
            if (is_array($decoded) && count($decoded) > 0) {
                $validated['agenda_notes'] = $decoded;
            }
        }
        unset($validated['agenda_notes_json']);

        // Debug log untuk melihat data yang akan diupdate
        \Log::info('Updating UPA Activity', [
            'activity_id' => $upaGroupActivity->activity_id,
            'pekan' => $validated['pekan'] ?? 'null',
            'agenda_notes' => $validated['agenda_notes'] ?? 'null',
            'agenda_notes_count' => isset($validated['agenda_notes']) ? count($validated['agenda_notes']) : 0
        ]);

        $upaGroupActivity->update($validated);

        // Update kehadiran anggota jika ada
        if ($request->has('anggota_kehadiran')) {
            $this->saveAttendance($upaGroupActivity, $request->anggota_kehadiran);
        }

        return redirect()->route('upa-group-activity.show', $upaGroupActivity)
                        ->with('success', 'Aktivitas UPA berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(UpaGroupActivity $upaGroupActivity)
    {
        $upaGroupActivity->delete();

        return redirect()->route('upa-group-activity.index')
                        ->with('success', 'Aktivitas UPA berhasil dihapus.');
    }

    /**
     * Get anggota UPA untuk kehadiran
     */
    public function getAnggotaUpa(Request $request)
    {
        $kodeUpa = $request->get('kode_upa');
        
        if (!$kodeUpa) {
            return response()->json([]);
        }

        // Cari UPA berdasarkan kode_upa
        $upa = Upa::where('kode_upa', $kodeUpa)->first();
        
        if (!$upa) {
            return response()->json([]);
        }

        // Ambil anggota yang terkait dengan UPA ini
        $anggota = Anggota::where('upa_id', $upa->upa_id)
                         ->where('status_keanggotaan', 'Aktif')
                         ->orderBy('nama_lengkap')
                         ->get(['id', 'nama_lengkap', 'no_telepon']);

        return response()->json($anggota);
    }

    /**
     * Get pembimbing UPA berdasarkan kode UPA
     */
    public function getPembimbingUpa(Request $request)
    {
        $kodeUpa = $request->get('kode_upa');
        
        if (!$kodeUpa) {
            return response()->json(['nama_pembimbing' => '']);
        }

        $upa = Upa::where('kode_upa', $kodeUpa)->first();
        
        if (!$upa) {
            return response()->json(['nama_pembimbing' => '']);
        }

        // Ambil pembimbing aktif dari UPA
        $pembimbingAktif = $upa->pembimbingAktif()->first();
        
        if ($pembimbingAktif) {
            return response()->json([
                'nama_pembimbing' => $pembimbingAktif->nama_lengkap,
                'telepon_pembimbing' => $pembimbingAktif->no_telepon ?? '',
                'jabatan' => $pembimbingAktif->pivot->jabatan ?? ''
            ]);
        }

        return response()->json(['nama_pembimbing' => '']);
    }

    /**
     * Simpan kehadiran anggota
     */
    private function saveAttendance($activity, $anggotaKehadiran)
    {
        // Hapus kehadiran lama
        $activity->attendances()->delete();
        
        // Dapatkan semua anggota UPA
        $upa = Upa::where('kode_upa', $activity->kode_upa)->first();
        if (!$upa) {
            return;
        }
        
        $allAnggota = Anggota::where('upa_id', $upa->upa_id)
                            ->where('status_keanggotaan', 'Aktif')
                            ->get();
        
        // Simpan kehadiran untuk semua anggota
        foreach ($allAnggota as $anggota) {
            $status = 'Tidak Hadir'; // Default tidak hadir
            
            // Jika ada di checkbox yang dicentang, maka hadir
            if (isset($anggotaKehadiran[$anggota->id]) && $anggotaKehadiran[$anggota->id] === 'Hadir') {
                $status = 'Hadir';
            }
            
            UpaGroupAttendance::create([
                'activity_id' => $activity->activity_id,
                'anggota_id' => $anggota->id,
                'status_kehadiran' => $status
            ]);
        }
    }
}
