<?php

namespace App\Http\Controllers;

use App\Models\Upa;
use App\Models\Anggota;
use App\Models\Dpc;
use App\Models\RefProvinsi;
use App\Models\RefKota;
use App\Models\RefKecamatan;
use App\Models\RefKelurahan;
use Illuminate\Http\Request;

class UpaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Upa::query();
        
        // Filter berdasarkan pencarian
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_upa', 'like', "%{$search}%")
                  ->orWhere('kode_upa', 'like', "%{$search}%")
                  ->orWhere('kota', 'like', "%{$search}%")
                  ->orWhere('provinsi', 'like', "%{$search}%")
                  ->orWhereHas('refKota', function($query) use ($search) {
                      $query->where('nama_kota', 'like', "%{$search}%");
                  })
                  ->orWhereHas('refProvinsi', function($query) use ($search) {
                      $query->where('nama_provinsi', 'like', "%{$search}%");
                  })
                  ->orWhereHas('dpc', function($query) use ($search) {
                      $query->where('nama_dpc', 'like', "%{$search}%");
                  });
            });
        }
        
        // Filter berdasarkan status
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }
        
        $upas = $query->with(['pembimbingAktif', 'dpc', 'refProvinsi', 'refKota', 'refKecamatan', 'refKelurahan'])->paginate(20);
        
        // Data untuk filter dropdown
        $statusList = Upa::select('status')
                         ->distinct()
                         ->pluck('status');
        
        return view('upa.index', compact('upas', 'statusList'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $anggotaList = Anggota::where('status_keanggotaan', 'Aktif')
                             ->orderBy('nama_lengkap')
                             ->get();
        
        $dpcList = Dpc::where('is_active', true)
                     ->orderBy('nama_dpc')
                     ->get();
        
        return view('upa.create', compact('anggotaList', 'dpcList'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'kode_upa' => 'required|string|max:50|unique:tbl_upa',
            'nama_upa' => 'required|string|max:255',
            'deskripsi' => 'nullable|string',
            'alamat' => 'nullable|string|max:500',
            'dpc_id' => 'required|exists:tbl_dpc,dpc_id',
            'provinsi_id' => 'required|exists:ref_provinsi,id',
            'kota_id' => 'required|exists:ref_kota,id',
            'kecamatan_id' => 'required|exists:ref_kecamatan,id',
            'kelurahan_id' => 'nullable|exists:ref_kelurahan,id',
            'kode_pos' => 'nullable|string|max:10',
            'no_telepon' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:100',
            'tanggal_berdiri' => 'nullable|date',
            'status' => 'required|in:Aktif,Tidak Aktif,Dibubarkan',
            'catatan' => 'nullable|string',
            'pembimbing_anggota_id' => 'required|exists:tbl_anggota,id',
            'pembimbing_tanggal_mulai' => 'nullable|date',
            'pembimbing_tanggal_selesai' => 'nullable|date|after_or_equal:pembimbing_tanggal_mulai',
            'pembimbing_catatan' => 'nullable|string'
        ]);

        $upa = Upa::create($validated);

        // Simpan pembimbing (hanya 1 pembimbing per UPA)
        if ($request->pembimbing_anggota_id) {
            // Hapus semua pembimbing lama terlebih dahulu
            $upa->pembimbing()->detach();
            
            // Tambah pembimbing baru
            $upa->pembimbing()->attach($request->pembimbing_anggota_id, [
                'jabatan' => 'Ketua',
                'tanggal_mulai' => $request->pembimbing_tanggal_mulai,
                'tanggal_selesai' => $request->pembimbing_tanggal_selesai,
                'status' => 'Aktif',
                'catatan' => $request->pembimbing_catatan
            ]);
        }

        return redirect()->route('upa.show', $upa->upa_id)
                        ->with('success', 'Data UPA berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Upa $upa)
    {
            $upa->load([
                'pembimbing' => function($query) {
                    $query->orderBy('pivot_jabatan')->orderBy('pivot_tanggal_mulai', 'desc');
                },
                'dpc',
                'refProvinsi',
                'refKota', 
                'refKecamatan',
                'refKelurahan',
                'anggotaAktif' => function($query) {
                    $query->orderBy('nama_lengkap');
                }
            ]);
        
        $anggotaList = \App\Models\Anggota::where('status_keanggotaan', 'Aktif')
                                         ->orderBy('nama_lengkap')
                                         ->get();
        
        return view('upa.show', compact('upa', 'anggotaList'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Upa $upa)
    {
        $anggotaList = Anggota::where('status_keanggotaan', 'Aktif')
                             ->orderBy('nama_lengkap')
                             ->get();
        
        $dpcList = Dpc::where('is_active', true)
                     ->orderBy('nama_dpc')
                     ->get();
        
        $upa->load('pembimbing');
        
        return view('upa.edit', compact('upa', 'anggotaList', 'dpcList'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Upa $upa)
    {
        $validated = $request->validate([
            'kode_upa' => 'required|string|max:50|unique:tbl_upa,kode_upa,' . $upa->upa_id . ',upa_id',
            'nama_upa' => 'required|string|max:255',
            'deskripsi' => 'nullable|string',
            'alamat' => 'nullable|string|max:500',
            'dpc_id' => 'required|exists:tbl_dpc,dpc_id',
            'provinsi_id' => 'required|exists:ref_provinsi,id',
            'kota_id' => 'required|exists:ref_kota,id',
            'kecamatan_id' => 'required|exists:ref_kecamatan,id',
            'kelurahan_id' => 'nullable|exists:ref_kelurahan,id',
            'kode_pos' => 'nullable|string|max:10',
            'no_telepon' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:100',
            'tanggal_berdiri' => 'nullable|date',
            'status' => 'required|in:Aktif,Tidak Aktif,Dibubarkan',
            'catatan' => 'nullable|string',
            'pembimbing' => 'nullable|array',
            'pembimbing.*.anggota_id' => 'required|exists:tbl_anggota,id',
            'pembimbing.*.jabatan' => 'required|in:Ketua,Wakil Ketua,Sekretaris,Bendahara,Anggota',
            'pembimbing.*.tanggal_mulai' => 'nullable|date',
            'pembimbing.*.tanggal_selesai' => 'nullable|date|after_or_equal:pembimbing.*.tanggal_mulai',
            'pembimbing.*.status' => 'required|in:Aktif,Tidak Aktif',
            'pembimbing.*.catatan' => 'nullable|string'
        ]);

        $upa->update($validated);

        // Update pembimbing (hanya 1 pembimbing per UPA)
        $pembimbingData = [];
        if ($request->has('pembimbing') && is_array($request->pembimbing)) {
            // Hanya ambil pembimbing pertama (jika ada lebih dari 1, yang lain diabaikan)
            $pembimbing = $request->pembimbing[0] ?? null;
            if ($pembimbing && !empty($pembimbing['anggota_id'])) {
                $pembimbingData[$pembimbing['anggota_id']] = [
                    'jabatan' => $pembimbing['jabatan'],
                    'tanggal_mulai' => $pembimbing['tanggal_mulai'],
                    'tanggal_selesai' => $pembimbing['tanggal_selesai'],
                    'status' => $pembimbing['status'] ?? 'Aktif',
                    'catatan' => $pembimbing['catatan']
                ];
            }
        }
        
        // Hapus semua pembimbing lama, lalu sync dengan yang baru
        $upa->pembimbing()->detach();
        if (!empty($pembimbingData)) {
            $upa->pembimbing()->sync($pembimbingData);
        }

        return redirect()->route('upa.show', $upa)
                        ->with('success', 'Data UPA berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Upa $upa)
    {
        $upa->delete();

        return redirect()->route('upa.index')
                        ->with('success', 'Data UPA berhasil dihapus.');
    }

    /**
     * Get provinsi list for dropdown
     */
    public function getProvinsi()
    {
        $provinsi = RefProvinsi::active()
            ->orderBy('nama_provinsi')
            ->get(['id', 'nama_provinsi']);

        return response()->json($provinsi);
    }

    /**
     * Get kota list by provinsi
     */
    public function getKota(Request $request)
    {
        $provinsiId = $request->get('provinsi_id');
        
        if (!$provinsiId) {
            return response()->json([]);
        }

        $kota = RefKota::active()
            ->where('provinsi_id', $provinsiId)
            ->orderBy('nama_kota')
            ->get(['id', 'nama_kota', 'tipe']);

        return response()->json($kota);
    }

    /**
     * Get kecamatan list by kota
     */
    public function getKecamatan(Request $request)
    {
        $kotaId = $request->get('kota_id');
        
        if (!$kotaId) {
            return response()->json([]);
        }

        $kecamatan = RefKecamatan::active()
            ->where('kota_id', $kotaId)
            ->orderBy('nama_kecamatan')
            ->get(['id', 'nama_kecamatan']);

        return response()->json($kecamatan);
    }

    /**
     * Get kelurahan list by kecamatan
     */
    /**
     * Add member to UPA
     */
    public function addMember(Request $request, Upa $upa)
    {
        $validated = $request->validate([
            'anggota_id' => 'required|exists:tbl_anggota,id',
            'jabatan' => 'required|in:Sekretaris,Bendahara,Anggota',
            'tanggal_mulai' => 'nullable|date',
            'tanggal_selesai' => 'nullable|date|after_or_equal:tanggal_mulai',
            'catatan' => 'nullable|string'
        ]);

        // Add member to UPA by updating upa_id and jabatan in tbl_anggota
        $anggota = \App\Models\Anggota::findOrFail($validated['anggota_id']);
        
        // Debug: Log the data being saved
        \Log::info('Adding member to UPA', [
            'anggota_id' => $validated['anggota_id'],
            'jabatan' => $validated['jabatan'],
            'upa_id' => $upa->upa_id
        ]);
        
        $anggota->update([
            'upa_id' => $upa->upa_id,
            'jabatan' => $validated['jabatan']
        ]);
        
        // Debug: Verify the data was saved
        $anggota->refresh();
        \Log::info('Member updated', [
            'anggota_id' => $anggota->id,
            'jabatan' => $anggota->jabatan,
            'upa_id' => $anggota->upa_id
        ]);

        return redirect()->back()
                        ->with('success', 'Anggota berhasil ditambahkan ke UPA');
    }

    /**
     * Update member jabatan in UPA
     */
    public function updateMemberJabatan(Request $request, Upa $upa)
    {
        $validated = $request->validate([
            'anggota_id' => 'required|exists:tbl_anggota,id',
            'jabatan' => 'required|in:Sekretaris,Bendahara,Anggota'
        ]);

        $anggota = \App\Models\Anggota::findOrFail($validated['anggota_id']);
        $anggota->update([
            'jabatan' => $validated['jabatan']
        ]);

        return redirect()->back()
                        ->with('success', 'Jabatan anggota berhasil diupdate');
    }

    public function getKelurahan(Request $request)
    {
        $kecamatanId = $request->get('kecamatan_id');
        
        if (!$kecamatanId) {
            return response()->json([]);
        }

        $kelurahan = RefKelurahan::active()
            ->where('kecamatan_id', $kecamatanId)
            ->orderBy('nama_kelurahan')
            ->get(['id', 'nama_kelurahan', 'tipe', 'kode_pos']);

        return response()->json($kelurahan);
    }

    /**
     * Update pembimbing UPA (hanya 1 pembimbing per UPA)
     */
    public function updatePembimbing(Request $request, Upa $upa)
    {
        $validated = $request->validate([
            'pembimbing_anggota_id' => 'required|exists:tbl_anggota,id',
            'pembimbing_tanggal_mulai' => 'nullable|date',
            'pembimbing_tanggal_selesai' => 'nullable|date|after_or_equal:pembimbing_tanggal_mulai',
            'pembimbing_catatan' => 'nullable|string'
        ]);

        // Hapus semua pembimbing lama
        $upa->pembimbing()->detach();
        
        // Tambah pembimbing baru
        $upa->pembimbing()->attach($validated['pembimbing_anggota_id'], [
            'jabatan' => 'Ketua',
            'tanggal_mulai' => $validated['pembimbing_tanggal_mulai'],
            'tanggal_selesai' => $validated['pembimbing_tanggal_selesai'],
            'status' => 'Aktif',
            'catatan' => $validated['pembimbing_catatan']
        ]);

        return redirect()->back()
                        ->with('success', 'Pembimbing UPA berhasil diperbarui.');
    }
}
