<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Takwim;
use App\Models\StrukturOrganisasi;
use App\Models\Anggota;
use App\Models\PesertaTakwim;
use Illuminate\Support\Facades\DB;

class TakwimController extends Controller
{
    public function index(Request $request)
    {
        $query = Takwim::with(['strukturPenyelenggara', 'pesertaTakwim']);
        
        // Filter berdasarkan pencarian
        if ($request->filled('search')) {
            $query->where('nama_takwim', 'like', '%' . $request->search . '%')
                  ->orWhere('deskripsi', 'like', '%' . $request->search . '%')
                  ->orWhere('koordinator', 'like', '%' . $request->search . '%');
        }
        
        // Filter berdasarkan penyelenggara
        if ($request->filled('penyelenggara')) {
            $query->whereHas('strukturPenyelenggara', function($q) use ($request) {
                $q->where('nama_struktur', 'like', '%' . $request->penyelenggara . '%');
            });
        }
        
        // Filter berdasarkan koordinator
        if ($request->filled('koordinator')) {
            $query->where('koordinator', 'like', '%' . $request->koordinator . '%');
        }
        
        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status_takwim', $request->status);
        }
        
        // Filter berdasarkan tanggal mulai
        if ($request->filled('tanggal_dari')) {
            $query->whereDate('tanggal_mulai', '>=', $request->tanggal_dari);
        }
        
        // Filter berdasarkan tanggal selesai
        if ($request->filled('tanggal_sampai')) {
            $query->whereDate('tanggal_selesai', '<=', $request->tanggal_sampai);
        }
        
        $takwim = $query->orderBy('tanggal_mulai', 'desc')->paginate(10);
            
        return view('takwim.index', compact('takwim'));
    }

    public function create()
    {
        $anggotas = Anggota::where('status_keanggotaan', 'Aktif')
                          ->select(['id', 'nama_lengkap', 'nomor_identitas_anggota', 'ref_jenjang_id'])
                          ->with(['refJenjang:ref_jenjang_id,jenjang'])
                          ->orderBy('nama_lengkap')
                          ->get();
        
        return view('takwim.create', compact('anggotas'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nama_takwim' => 'required|string|max:255',
            'deskripsi' => 'nullable|string',
            'tanggal_mulai' => 'required|date',
            'tanggal_selesai' => 'nullable|date|after_or_equal:tanggal_mulai',
            'penyelenggara' => 'required|string|max:255',
            'koordinator_anggota_id' => 'required|exists:tbl_anggota,id',
            'evaluasi' => 'nullable|string',
            'catatan_berkelanjutan' => 'nullable|string',
            'status_takwim' => 'required|in:Aktif,Selesai,Dibatalkan'
        ]);

        // Get koordinator name from anggota
        $koordinator = Anggota::find($request->koordinator_anggota_id);
        
        Takwim::create([
            'nama_takwim' => $request->nama_takwim,
            'deskripsi' => $request->deskripsi,
            'tanggal_mulai' => $request->tanggal_mulai,
            'tanggal_selesai' => $request->tanggal_selesai,
            'penyelenggara' => $request->penyelenggara,
            'koordinator' => $koordinator->nama_lengkap,
            'koordinator_anggota_id' => $request->koordinator_anggota_id,
            'evaluasi' => $request->evaluasi,
            'catatan_berkelanjutan' => $request->catatan_berkelanjutan,
            'status_takwim' => $request->status_takwim
        ]);

        return redirect()->route('takwim.index')
            ->with('success', 'Takwim berhasil ditambahkan.');
    }

    public function show(Takwim $takwim)
    {
        $takwim->load(['strukturPenyelenggara', 'pesertaTakwim.anggota']);
        $anggotaTersedia = Anggota::whereNotIn('id', function($query) use ($takwim) {
            $query->select('anggota_id')
                  ->from('tbl_peserta_takwim')
                  ->where('takwim_id', $takwim->id)
                  ->whereNull('tanggal_keluar');
        })->get();
        
        return view('takwim.show', compact('takwim', 'anggotaTersedia'));
    }

    public function edit(Takwim $takwim)
    {
        $anggotas = Anggota::where('status_keanggotaan', 'Aktif')
                          ->select(['id', 'nama_lengkap', 'nomor_identitas_anggota', 'ref_jenjang_id'])
                          ->with(['refJenjang:ref_jenjang_id,jenjang'])
                          ->orderBy('nama_lengkap')
                          ->get();
        
        return view('takwim.edit', compact('takwim', 'anggotas'));
    }

    public function update(Request $request, Takwim $takwim)
    {
        $request->validate([
            'nama_takwim' => 'required|string|max:255',
            'deskripsi' => 'nullable|string',
            'tanggal_mulai' => 'required|date',
            'tanggal_selesai' => 'nullable|date|after_or_equal:tanggal_mulai',
            'penyelenggara' => 'required|string|max:255',
            'koordinator_anggota_id' => 'required|exists:tbl_anggota,id',
            'evaluasi' => 'nullable|string',
            'catatan_berkelanjutan' => 'nullable|string',
            'status_takwim' => 'required|in:Aktif,Selesai,Dibatalkan'
        ]);

        // Get koordinator name from anggota
        $koordinator = Anggota::find($request->koordinator_anggota_id);
        
        $takwim->update([
            'nama_takwim' => $request->nama_takwim,
            'deskripsi' => $request->deskripsi,
            'tanggal_mulai' => $request->tanggal_mulai,
            'tanggal_selesai' => $request->tanggal_selesai,
            'penyelenggara' => $request->penyelenggara,
            'koordinator' => $koordinator->nama_lengkap,
            'koordinator_anggota_id' => $request->koordinator_anggota_id,
            'evaluasi' => $request->evaluasi,
            'catatan_berkelanjutan' => $request->catatan_berkelanjutan,
            'status_takwim' => $request->status_takwim
        ]);

        return redirect()->route('takwim.show', $takwim)
            ->with('success', 'Takwim berhasil diperbarui.');
    }

    public function destroy(Takwim $takwim)
    {
        DB::transaction(function() use ($takwim) {
            // Hapus peserta takwim terlebih dahulu
            $takwim->pesertaTakwim()->delete();
            // Hapus takwim
            $takwim->delete();
        });

        return redirect()->route('takwim.index')
            ->with('success', 'Takwim berhasil dihapus.');
    }

    public function addPeserta(Request $request, Takwim $takwim)
    {
        $request->validate([
            'anggota_id' => 'required|exists:tbl_anggota,id',
            'tanggal_bergabung' => 'required|date',
            'catatan' => 'nullable|string'
        ]);

        // Cek apakah anggota sudah terdaftar
        $existingPeserta = PesertaTakwim::where('takwim_id', $takwim->id)
            ->where('anggota_id', $request->anggota_id)
            ->whereNull('tanggal_keluar')
            ->first();

        if ($existingPeserta) {
            return back()->with('error', 'Anggota sudah terdaftar dalam takwim ini.');
        }

        PesertaTakwim::create([
            'takwim_id' => $takwim->id,
            'anggota_id' => $request->anggota_id,
            'tanggal_bergabung' => $request->tanggal_bergabung,
            'catatan' => $request->catatan
        ]);

        return back()->with('success', 'Peserta berhasil ditambahkan.');
    }

    public function removePeserta(Takwim $takwim, Anggota $anggota)
    {
        $peserta = PesertaTakwim::where('takwim_id', $takwim->id)
            ->where('anggota_id', $anggota->id)
            ->whereNull('tanggal_keluar')
            ->first();

        if ($peserta) {
            $peserta->update(['tanggal_keluar' => now()]);
            return back()->with('success', 'Peserta berhasil dikeluarkan.');
        }

        return back()->with('error', 'Peserta tidak ditemukan.');
    }
}
