<?php

namespace App\Http\Controllers;

use App\Models\Anggota;
use App\Models\Dpc;
use App\Models\RiwayatPerpindahanAnggota;
use App\Models\Upa;
use App\Services\PerpindahanAnggotaService;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class PerpindahanAnggotaController extends Controller
{
    protected $perpindahanService;

    public function __construct(PerpindahanAnggotaService $perpindahanService)
    {
        $this->perpindahanService = $perpindahanService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = RiwayatPerpindahanAnggota::with([
            'anggota',
            'upaAsal',
            'upaTujuan',
            'dpcAsal',
            'dpcTujuan'
        ])
        ->orderBy('created_at', 'desc');

        // Filter berdasarkan tipe
        if ($request->filled('tipe')) {
            $query->byTipe($request->tipe);
        }

        // Filter berdasarkan anggota
        if ($request->filled('anggota_id')) {
            $query->byAnggota($request->anggota_id);
        }

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $riwayat = $query->paginate(20);

        return view('perpindahan-anggota.index', compact('riwayat'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        // Default tipe adalah 'mutasi', tapi bisa di-override dari query string
        $tipe = $request->get('tipe', 'mutasi');

        // Validasi tipe
        if (!in_array($tipe, ['mutasi', 'penempatan', 'perpindahan', 'rekomposisi'])) {
            $tipe = 'mutasi';
        }

        // Data untuk form
        $dpcList = Dpc::where('is_active', true)
            ->orderBy('nama_dpc')
            ->get();

        $upaList = Upa::where('status', 'Aktif')
            ->orderBy('nama_upa')
            ->get();

        $anggotaList = Anggota::where('status_keanggotaan', 'Aktif')
            ->orderBy('nama_lengkap')
            ->get();


        // Get required fields untuk tipe ini
        $requiredFields = $this->perpindahanService->getRequiredFields($tipe);
        $tipeLabel = $this->perpindahanService->getTipeLabel($tipe);

        return view('perpindahan-anggota.create', compact(
            'tipe',
            'tipeLabel',
            'requiredFields',
            'dpcList',
            'upaList',
            'anggotaList'
        ));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $tipe = $request->input('tipe_transaksi', 'mutasi');

        // Validasi berdasarkan tipe
        $rules = $this->perpindahanService->getValidationRules($tipe);
        $validated = $request->validate($rules);

        try {
            // Proses perpindahan menggunakan service
            $riwayat = $this->perpindahanService->prosesPerpindahan($validated, $tipe);

            return redirect()
                ->route('perpindahan-anggota.show', $riwayat->id)
                ->with('success', 'Perpindahan anggota berhasil diproses!');

        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Gagal memproses perpindahan: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $riwayat = RiwayatPerpindahanAnggota::with([
            'anggota',
            'upaAsal',
            'upaTujuan',
            'dpcAsal',
            'dpcTujuan',
            'pembimbingLama',
            'pembimbingBaru',
            'creator',
            'approver'
        ])->findOrFail($id);

        return view('perpindahan-anggota.show', compact('riwayat'));
    }

    /**
     * Generate dan download surat PDF
     */
    public function downloadSurat(string $id)
    {
        $riwayat = RiwayatPerpindahanAnggota::with([
            'anggota',
            'upaAsal',
            'upaTujuan',
            'dpcAsal',
            'dpcTujuan',
            'pembimbingLama',
            'pembimbingBaru'
        ])->findOrFail($id);

        // Get template surat berdasarkan tipe
        $template = $this->perpindahanService->getSuratTemplate($riwayat->tipe_transaksi);

        // Generate PDF
        $pdf = Pdf::loadView($template, compact('riwayat'))
            ->setPaper('a4', 'portrait');

        $filename = 'Surat_' . ucfirst($riwayat->tipe_transaksi) . '_' . $riwayat->nomor_surat . '.pdf';
        $filename = str_replace('/', '_', $filename);

        return $pdf->download($filename);
    }

    /**
     * Get UPA by DPC for AJAX
     */
    public function getUpaByDpc(Request $request)
    {
        $dpcId = $request->dpc_id;
        
        $upas = Upa::where('dpc_id', $dpcId)
            ->where('status', 'Aktif')
            ->select('upa_id', 'kode_upa', 'nama_upa')
            ->orderBy('nama_upa')
            ->get();

        return response()->json($upas);
    }

    /**
     * Get Anggota by UPA for AJAX
     */
    public function getAnggotaByUpa(Request $request)
    {
        $upaId = $request->upa_id;
        
        $anggota = Anggota::where('upa_id', $upaId)
            ->where('status_keanggotaan', 'Aktif')
            ->select('id', 'nama_lengkap', 'nomor_identitas_anggota')
            ->orderBy('nama_lengkap')
            ->get();

        return response()->json($anggota);
    }

    /**
     * Get Pembimbing by UPA for AJAX
     */
    public function getPembimbingByUpa(Request $request)
    {
        $upaId = $request->upa_id;
        
        $pembimbing = Anggota::whereHas('upaPembimbing', function($query) use ($upaId) {
            $query->where('tbl_upa_pembimbing.upa_id', $upaId)
                  ->where('tbl_upa_pembimbing.status', 'Aktif');
        })
        ->with(['upaPembimbing' => function($query) use ($upaId) {
            $query->where('tbl_upa_pembimbing.upa_id', $upaId)
                  ->where('tbl_upa_pembimbing.status', 'Aktif')
                  ->select('tbl_upa.upa_id', 'tbl_upa_pembimbing.anggota_id', 'tbl_upa_pembimbing.jabatan');
        }])
        ->select('id', 'nama_lengkap', 'nomor_identitas_anggota', 'no_telepon')
        ->get()
        ->map(function($anggota) {
            $jabatan = $anggota->upaPembimbing->first()->pivot->jabatan ?? 'Pembimbing';
            return [
                'id' => $anggota->id,
                'nama_lengkap' => $anggota->nama_lengkap,
                'nomor_identitas_anggota' => $anggota->nomor_identitas_anggota,
                'no_telepon' => $anggota->no_telepon,
                'jabatan' => $jabatan
            ];
        });

        return response()->json($pembimbing);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $riwayat = RiwayatPerpindahanAnggota::findOrFail($id);
            
            // Hanya bisa delete jika status draft atau cancelled
            if (!in_array($riwayat->status, ['draft', 'cancelled'])) {
                return redirect()
                    ->back()
                    ->with('error', 'Hanya riwayat dengan status draft atau cancelled yang bisa dihapus!');
            }

            $riwayat->delete();

            return redirect()
                ->route('perpindahan-anggota.index')
                ->with('success', 'Riwayat perpindahan berhasil dihapus!');

        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->with('error', 'Gagal menghapus riwayat: ' . $e->getMessage());
        }
    }
}
