<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Artisan;

class PerformanceController extends Controller
{
    /**
     * Display performance dashboard
     */
    public function dashboard()
    {
        $stats = [
            'cache' => $this->getCacheStats(),
            'database' => $this->getDatabaseStats(),
            'memory' => $this->getMemoryStats(),
            'queries' => $this->getQueryStats(),
        ];

        return view('performance.dashboard', compact('stats'));
    }

    /**
     * Clear all caches
     */
    public function clearCache()
    {
        try {
            // Clear application cache
            Artisan::call('cache:clear');
            
            // Clear view cache
            Artisan::call('view:clear');
            
            // Clear config cache
            Artisan::call('config:clear');
            
            // Clear route cache
            Artisan::call('route:clear');
            
            // Clear specific caches
            $this->clearSpecificCaches();
            
            return response()->json([
                'success' => true,
                'message' => 'All caches cleared successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error clearing cache: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Warm up cache
     */
    public function warmCache()
    {
        try {
            // Warm up common caches
            $this->warmUpCommonCaches();
            
            return response()->json([
                'success' => true,
                'message' => 'Cache warmed up successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error warming cache: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get performance metrics
     */
    public function metrics()
    {
        $metrics = [
            'memory_usage' => memory_get_usage(true),
            'peak_memory' => memory_get_peak_usage(true),
            'execution_time' => microtime(true) - LARAVEL_START,
            'cache_hits' => $this->getCacheHits(),
            'cache_misses' => $this->getCacheMisses(),
        ];

        return response()->json($metrics);
    }

    private function getCacheStats()
    {
        $keys = [
            'anggota_status_list',
            'ref_jenjang_active',
            'anggota_provinsi_list'
        ];

        $stats = [];
        foreach ($keys as $key) {
            $stats[$key] = Cache::has($key);
        }

        return $stats;
    }

    private function getDatabaseStats()
    {
        try {
            $connection = \DB::connection();
            $pdo = $connection->getPdo();
            
            return [
                'driver' => $connection->getDriverName(),
                'version' => $pdo->getAttribute(\PDO::ATTR_SERVER_VERSION),
                'connection_status' => 'Connected'
            ];
        } catch (\Exception $e) {
            return [
                'connection_status' => 'Error: ' . $e->getMessage()
            ];
        }
    }

    private function getMemoryStats()
    {
        return [
            'current' => $this->formatBytes(memory_get_usage(true)),
            'peak' => $this->formatBytes(memory_get_peak_usage(true)),
            'limit' => ini_get('memory_limit')
        ];
    }

    private function getQueryStats()
    {
        // This would require query logging to be enabled
        return [
            'total_queries' => 0,
            'slow_queries' => 0,
            'avg_query_time' => 0
        ];
    }

    private function clearSpecificCaches()
    {
        $keys = [
            'anggota_status_list',
            'ref_jenjang_active',
            'anggota_provinsi_list',
            'anggota_performance_stats'
        ];

        foreach ($keys as $key) {
            Cache::forget($key);
        }
    }

    private function warmUpCommonCaches()
    {
        // Warm up status list
        \App\Models\Anggota::getStatusList();
        
        // Warm up jenjang list
        \App\Models\RefJenjang::active()->get();
        
        // Warm up provinsi list
        \App\Models\Anggota::getProvinsiList();
    }

    private function getCacheHits()
    {
        // This would require cache hit/miss tracking
        return 0;
    }

    private function getCacheMisses()
    {
        // This would require cache hit/miss tracking
        return 0;
    }

    private function formatBytes($bytes, $precision = 2)
    {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
}


