<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\Kegiatan;
use App\Models\AbsensiKegiatan;
use App\Models\Mutasi;
use App\Models\Anggota;
use App\Models\Dpc;
use App\Models\StrukturOrganisasi;
use Carbon\Carbon;

class MobileController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Mobile Dashboard
     */
    public function dashboard(Request $request)
    {
        // Get anggota ID from session or request parameter
        $anggotaId = $request->session()->get('mobile_anggota_id') ?? $request->get('anggota_id');
        
        if (!$anggotaId) {
            // Show anggota selection page
            $anggotas = Anggota::where('status_keanggotaan', 'Aktif')
                ->select(['id', 'nama_lengkap', 'nomor_identitas_anggota'])
                ->orderBy('nama_lengkap')
                ->limit(20)
                ->get();
            
            return view('mobile.anggota-selection', compact('anggotas'));
        }
        
        $anggota = Anggota::find($anggotaId);
        
        if (!$anggota) {
            return redirect()->route('mobile.dashboard')->with('error', 'Data anggota tidak ditemukan');
        }
        
        // Store selected anggota in session
        $request->session()->put('mobile_anggota_id', $anggota->id);

        // Get upcoming activities
        $upcomingKegiatan = Kegiatan::where('tanggal_kegiatan', '>=', today())
            ->where('status_kegiatan', '!=', 'Dibatalkan')
            ->orderBy('tanggal_kegiatan')
            ->limit(5)
            ->get();

        // Get recent attendance
        $recentAttendance = AbsensiKegiatan::where('anggota_id', $anggota->id)
            ->with('kegiatan')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        // Get pending mutasi
        $pendingMutasi = Mutasi::where('anggota_id', $anggota->id)
            ->where('status_mutasi', 'Pending')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('mobile.dashboard', compact('anggota', 'upcomingKegiatan', 'recentAttendance', 'pendingMutasi'));
    }

    /**
     * Select Anggota for Mobile Session
     */
    public function selectAnggota(Request $request)
    {
        $request->validate([
            'anggota_id' => 'required|exists:tbl_anggota,id'
        ]);
        
        $anggota = Anggota::findOrFail($request->anggota_id);
        
        // Store selected anggota in session
        $request->session()->put('mobile_anggota_id', $anggota->id);
        
        return redirect()->route('mobile.dashboard')->with('success', 'Anggota berhasil dipilih');
    }


    /**
     * Mobile Attendance List
     */
    public function attendance(Request $request)
    {
        $anggotaId = $request->session()->get('mobile_anggota_id');
        
        if (!$anggotaId) {
            return redirect()->route('mobile.dashboard')->with('error', 'Silakan pilih anggota terlebih dahulu');
        }
        
        $anggota = Anggota::find($anggotaId);
        
        if (!$anggota) {
            return redirect()->route('mobile.dashboard')->with('error', 'Data anggota tidak ditemukan');
        }

        // Get activities where user is participant
        $kegiatan = Kegiatan::whereHas('anggotaPartisipan', function($query) use ($anggota) {
            $query->where('anggota_id', $anggota->id);
        })
        ->orWhere('status_kegiatan', '!=', 'Dibatalkan')
        ->orderBy('tanggal_kegiatan', 'desc')
        ->paginate(10);

        return view('mobile.attendance', compact('kegiatan', 'anggota'));
    }

    /**
     * Mobile Attendance Detail
     */
    public function attendanceDetail(Request $request, Kegiatan $kegiatan)
    {
        $anggotaId = $request->session()->get('mobile_anggota_id');
        
        if (!$anggotaId) {
            return redirect()->route('mobile.dashboard')->with('error', 'Silakan pilih anggota terlebih dahulu');
        }
        
        $anggota = Anggota::find($anggotaId);
        
        if (!$anggota) {
            return redirect()->route('mobile.dashboard')->with('error', 'Data anggota tidak ditemukan');
        }

        // Check if user has attendance record
        $attendance = AbsensiKegiatan::where('kegiatan_id', $kegiatan->id)
            ->where('anggota_id', $anggota->id)
            ->first();

        // Check if user is participant
        $isParticipant = $kegiatan->anggotaPartisipan()->where('anggota_id', $anggota->id)->exists();

        return view('mobile.attendance-detail', compact('kegiatan', 'anggota', 'attendance', 'isParticipant'));
    }

    /**
     * Mobile Check-in
     */
    public function checkin(Request $request)
    {
        $request->validate([
            'kegiatan_id' => 'required|exists:tbl_kegiatan,id',
            'anggota_id' => 'required|exists:tbl_anggota,id'
        ]);

        try {
            DB::beginTransaction();

            $kegiatan = Kegiatan::findOrFail($request->kegiatan_id);
            $anggota = Anggota::findOrFail($request->anggota_id);

            // Check if already checked in
            $existingCheckin = AbsensiKegiatan::where('kegiatan_id', $kegiatan->id)
                ->where('anggota_id', $anggota->id)
                ->first();

            if ($existingCheckin) {
                return response()->json([
                    'success' => false,
                    'message' => 'Anda sudah melakukan check-in sebelumnya'
                ]);
            }

            // Check if activity is still ongoing
            $now = Carbon::now();
            $activityStart = Carbon::parse($kegiatan->tanggal_mulai . ' ' . $kegiatan->waktu_mulai);
            $activityEnd = Carbon::parse($kegiatan->tanggal_selesai . ' ' . $kegiatan->waktu_selesai);

            if ($now->lt($activityStart->subMinutes(30))) {
                return response()->json([
                    'success' => false,
                    'message' => 'Check-in belum dibuka. Buka 30 menit sebelum kegiatan dimulai.'
                ]);
            }

            if ($now->gt($activityEnd)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Kegiatan sudah selesai'
                ]);
            }

            // Create attendance record
            AbsensiKegiatan::create([
                'kegiatan_id' => $kegiatan->id,
                'anggota_id' => $anggota->id,
                'waktu_checkin' => $now,
                'status' => 'hadir'
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Check-in berhasil!',
                'data' => [
                    'waktu_checkin' => $now->format('d/m/Y H:i:s'),
                    'status' => 'hadir'
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Mobile Check-out
     */
    public function checkout(Request $request)
    {
        $request->validate([
            'kegiatan_id' => 'required|exists:tbl_kegiatan,id',
            'anggota_id' => 'required|exists:tbl_anggota,id'
        ]);

        try {
            DB::beginTransaction();

            $kegiatan = Kegiatan::findOrFail($request->kegiatan_id);
            $anggota = Anggota::findOrFail($request->anggota_id);

            // Check if already checked in
            $attendance = AbsensiKegiatan::where('kegiatan_id', $kegiatan->id)
                ->where('anggota_id', $anggota->id)
                ->first();

            if (!$attendance) {
                return response()->json([
                    'success' => false,
                    'message' => 'Anda belum melakukan check-in'
                ]);
            }

            if ($attendance->waktu_checkout) {
                return response()->json([
                    'success' => false,
                    'message' => 'Anda sudah melakukan check-out'
                ]);
            }

            // Update attendance with checkout time
            $attendance->update([
                'waktu_checkout' => Carbon::now(),
                'status' => 'hadir'
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Check-out berhasil!',
                'data' => [
                    'waktu_checkout' => $attendance->waktu_checkout->format('d/m/Y H:i:s'),
                    'status' => 'hadir'
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Mobile Mutasi List
     */
    public function mutasi(Request $request)
    {
        $anggotaId = $request->session()->get('mobile_anggota_id');
        
        if (!$anggotaId) {
            return redirect()->route('mobile.dashboard')->with('error', 'Silakan pilih anggota terlebih dahulu');
        }
        
        $anggota = Anggota::find($anggotaId);
        
        if (!$anggota) {
            return redirect()->route('mobile.dashboard')->with('error', 'Data anggota tidak ditemukan');
        }

        $mutasi = Mutasi::where('anggota_id', $anggota->id)
            ->with(['strukturDari', 'strukturKe'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('mobile.mutasi', compact('mutasi', 'anggota'));
    }

    /**
     * Mobile Create Mutasi
     */
    public function createMutasi(Request $request)
    {
        $anggotaId = $request->session()->get('mobile_anggota_id');
        
        if (!$anggotaId) {
            return redirect()->route('mobile.dashboard')->with('error', 'Silakan pilih anggota terlebih dahulu');
        }
        
        $anggota = Anggota::find($anggotaId);
        
        if (!$anggota) {
            return redirect()->route('mobile.dashboard')->with('error', 'Data anggota tidak ditemukan');
        }

        // Get DPC list for selection
        $dpcList = Dpc::orderBy('nama_dpc')->get();

        return view('mobile.mutasi-create', compact('anggota', 'dpcList'));
    }

    /**
     * Mobile Submit Mutasi
     */
    public function submitMutasi(Request $request)
    {
        $anggotaId = $request->session()->get('mobile_anggota_id');
        
        if (!$anggotaId) {
            return response()->json([
                'success' => false,
                'message' => 'Silakan pilih anggota terlebih dahulu'
            ]);
        }
        
        $anggota = Anggota::find($anggotaId);
        
        if (!$anggota) {
            return response()->json([
                'success' => false,
                'message' => 'Data anggota tidak ditemukan'
            ]);
        }

        $request->validate([
            'jenis_mutasi' => 'required|string',
            'struktur_ke' => 'required|exists:tbl_struktur_organisasi,id',
            'alasan_mutasi' => 'required|string|max:500',
            'alamat_tujuan' => 'nullable|string|max:500',
            'catatan' => 'nullable|string|max:500'
        ]);

        try {
            DB::beginTransaction();

            $mutasiData = [
                'anggota_id' => $anggota->id,
                'jenis_mutasi' => $request->jenis_mutasi,
                'struktur_dari' => $anggota->struktur_organisasi_id,
                'struktur_ke' => $request->struktur_ke,
                'tanggal_mutasi' => now()->toDateString(),
                'alasan_mutasi' => $request->alasan_mutasi,
                'alamat_tujuan' => $request->alamat_tujuan,
                'catatan' => $request->catatan,
                'status_mutasi' => 'Pending'
            ];

            $mutasi = Mutasi::create($mutasiData);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Pengajuan mutasi berhasil dikirim!',
                'data' => [
                    'id' => $mutasi->id,
                    'jenis_mutasi' => $mutasi->jenis_mutasi,
                    'status' => $mutasi->status_mutasi
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Mobile Show Mutasi
     */
    public function showMutasi(Request $request, Mutasi $mutasi)
    {
        $anggotaId = $request->session()->get('mobile_anggota_id');
        
        if (!$anggotaId) {
            return redirect()->route('mobile.dashboard')->with('error', 'Silakan pilih anggota terlebih dahulu');
        }
        
        $anggota = Anggota::find($anggotaId);
        
        if (!$anggota || $mutasi->anggota_id !== $anggota->id) {
            return redirect()->route('mobile.mutasi')->with('error', 'Data mutasi tidak ditemukan');
        }

        $mutasi->load(['strukturDari', 'strukturKe', 'disetujuiOleh']);

        return view('mobile.mutasi-show', compact('mutasi', 'anggota'));
    }

    /**
     * Mobile Profile
     */
    public function profile(Request $request)
    {
        $anggotaId = $request->session()->get('mobile_anggota_id');
        
        if (!$anggotaId) {
            return redirect()->route('mobile.dashboard')->with('error', 'Silakan pilih anggota terlebih dahulu');
        }
        
        $anggota = Anggota::find($anggotaId);
        
        if (!$anggota) {
            return redirect()->route('mobile.dashboard')->with('error', 'Data anggota tidak ditemukan');
        }

        $anggota->load(['strukturOrganisasi', 'dpc', 'refJenjang']);

        return view('mobile.profile', compact('anggota'));
    }

    /**
     * Mobile Update Profile
     */
    public function updateProfile(Request $request)
    {
        $anggotaId = $request->session()->get('mobile_anggota_id');
        
        if (!$anggotaId) {
            return response()->json([
                'success' => false,
                'message' => 'Silakan pilih anggota terlebih dahulu'
            ]);
        }
        
        $anggota = Anggota::find($anggotaId);
        
        if (!$anggota) {
            return response()->json([
                'success' => false,
                'message' => 'Data anggota tidak ditemukan'
            ]);
        }

        $request->validate([
            'no_telepon' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:100',
            'alamat' => 'nullable|string|max:500'
        ]);

        try {
            $anggota->update($request->only(['no_telepon', 'email', 'alamat']));

            return response()->json([
                'success' => true,
                'message' => 'Profil berhasil diperbarui!'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ]);
        }
    }
}
