<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Dpc;
use App\Models\Upa;
use App\Models\Anggota;
use App\Models\RefJenjang;
use Illuminate\Support\Facades\DB;

class KomposisiController extends Controller
{
    /**
     * Menampilkan halaman utama komposisi DPC dan UPA
     */
    public function index(Request $request)
    {
        $query = Dpc::with(['dpd', 'upa' => function($query) {
            $query->active();
        }])
        ->active();

        // Filter DPD
        if ($request->filled('dpd_id')) {
            $query->where('dpd_id', $request->dpd_id);
        }

        // Filter DPC
        if ($request->filled('dpc_id')) {
            $query->where('dpc_id', $request->dpc_id);
        }

        // Filter UPA
        if ($request->filled('upa_id')) {
            $query->whereHas('upa', function($q) use ($request) {
                $q->where('upa_id', $request->upa_id);
            });
        }

        // Filter Status DPC
        if ($request->filled('status_dpc')) {
            if ($request->status_dpc == 'aktif') {
                $query->where('is_active', true);
            } elseif ($request->status_dpc == 'tidak_aktif') {
                $query->where('is_active', false);
            }
        }

        // (Status UPA filter removed)

        // Filter Jenjang Anggota
        if ($request->filled('jenjang_id')) {
            $query->whereHas('anggota', function($q) use ($request) {
                $q->where('ref_jenjang_id', $request->jenjang_id)
                  ->where('status_keanggotaan', 'Aktif');
            });
        }

        // (Min. jumlah anggota filter removed)

        $dpcList = $query->orderBy('nama_dpc')->get();

        // Data untuk filter
        $dpdList = \App\Models\Dpd::orderBy('nama_dpd')->get();
        $allDpcList = \App\Models\Dpc::orderBy('nama_dpc')->get();
        $allUpaList = \App\Models\Upa::orderBy('nama_upa')->get();
        $jenjangList = \App\Models\RefJenjang::active()->orderBy('jenjang')->get();
        
        $statusDpcOptions = [
            'aktif' => 'Aktif',
            'tidak_aktif' => 'Tidak Aktif'
        ];
        
        // (Status UPA options removed)

        $statistik = $this->getStatistikKomposisi($request);

        return view('komposisi.index', compact('dpcList', 'statistik', 'dpdList', 'allDpcList', 'allUpaList', 'jenjangList', 'statusDpcOptions'));
    }

    /**
     * Get DPC by DPD ID (AJAX)
     */
    public function getDpcByDpd(Request $request)
    {
        $dpdId = $request->get('dpd_id');
        
        if ($dpdId) {
            $dpcList = Dpc::where('dpd_id', $dpdId)
                ->orderBy('nama_dpc')
                ->get(['dpc_id', 'nama_dpc']);
        } else {
            $dpcList = Dpc::orderBy('nama_dpc')
                ->get(['dpc_id', 'nama_dpc']);
        }
        
        return response()->json($dpcList);
    }

    /**
     * Get UPA by DPC ID (AJAX)
     */
    public function getUpaByDpc(Request $request)
    {
        $dpcId = $request->get('dpc_id');
        
        if ($dpcId) {
            $upaList = Upa::where('dpc_id', $dpcId)
                ->orderBy('nama_upa')
                ->get(['upa_id', 'nama_upa']);
        } else {
            $upaList = Upa::orderBy('nama_upa')
                ->get(['upa_id', 'nama_upa']);
        }
        
        return response()->json($upaList);
    }

    /**
     * Menampilkan detail komposisi DPC tertentu
     */
    public function showDpc($dpcId)
    {
        $dpc = Dpc::with(['dpd', 'upa' => function($query) {
            $query->active()->with(['pembimbingAktif', 'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            }]);
        }])
        ->findOrFail($dpcId);

        $statistikDpc = $this->getStatistikDpc($dpcId);

        return view('komposisi.show-dpc', compact('dpc', 'statistikDpc'));
    }

    /**
     * Menampilkan detail komposisi UPA tertentu
     */
    public function showUpa($upaId)
    {
        $upa = Upa::with([
            'dpc.dpd',
            'pembimbingAktif' => function($query) {
                $query->with('refJenjang');
            },
            'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            },
            'refProvinsi',
            'refKota',
            'refKecamatan',
            'refKelurahan'
        ])
        ->findOrFail($upaId);

        $statistikUpa = $this->getStatistikUpa($upaId);

        return view('komposisi.show-upa', compact('upa', 'statistikUpa'));
    }

    /**
     * Print komposisi DPC
     */
    public function printDpc($dpcId)
    {
        $dpc = Dpc::with(['dpd', 'upa' => function($query) {
            $query->active()->with(['pembimbingAktif', 'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            }]);
        }])
        ->findOrFail($dpcId);

        $statistikDpc = $this->getStatistikDpc($dpcId);

        return view('komposisi.print-dpc', compact('dpc', 'statistikDpc'));
    }

    /**
     * Print komposisi UPA
     */
    public function printUpa($upaId)
    {
        $upa = Upa::with([
            'dpc.dpd',
            'pembimbingAktif' => function($query) {
                $query->with('refJenjang');
            },
            'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            },
            'refProvinsi',
            'refKota',
            'refKecamatan',
            'refKelurahan'
        ])
        ->findOrFail($upaId);

        $statistikUpa = $this->getStatistikUpa($upaId);

        return view('komposisi.print-upa', compact('upa', 'statistikUpa'));
    }

    /**
     * Print komposisi semua DPC
     */
    public function printAll(Request $request)
    {
        $query = Dpc::with(['dpd', 'upa' => function($query) {
            $query->active()->with(['pembimbingAktif', 'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            }]);
        }])
        ->active();

        // Apply filters if provided
        if ($request->filled('dpd_id')) {
            $query->where('dpd_id', $request->dpd_id);
        }

        if ($request->filled('dpc_id')) {
            $query->where('dpc_id', $request->dpc_id);
        }

        if ($request->filled('upa_id')) {
            $query->whereHas('upa', function($q) use ($request) {
                $q->where('upa_id', $request->upa_id);
            });
        }

        if ($request->filled('status_dpc')) {
            if ($request->status_dpc == 'aktif') {
                $query->where('is_active', true);
            } elseif ($request->status_dpc == 'tidak_aktif') {
                $query->where('is_active', false);
            }
        }

        // (Status UPA filter removed)

        if ($request->filled('jenjang_id')) {
            $query->whereHas('anggota', function($q) use ($request) {
                $q->where('ref_jenjang_id', $request->jenjang_id)
                  ->where('status_keanggotaan', 'Aktif');
            });
        }

        // (Min. jumlah anggota filter removed)

        $dpcList = $query->orderBy('nama_dpc')->get();

        // Data untuk filter info
        $dpdList = \App\Models\Dpd::orderBy('nama_dpd')->get();
        $jenjangList = \App\Models\RefJenjang::active()->orderBy('jenjang')->get();
        
        $statistik = $this->getStatistikKomposisi($request);

        return view('komposisi.print-all', compact('dpcList', 'statistik', 'dpdList', 'jenjangList'));
    }

    /**
     * Print komposisi dalam format Surat Keputusan
     */
    public function printSk(Request $request)
    {
        $query = Dpc::with(['dpd', 'upa' => function($query) {
            $query->active()->with(['pembimbingAktif', 'anggotaAktif' => function($query) {
                $query->with('refJenjang');
            }]);
        }])
        ->active();

        // Apply filters if provided
        if ($request->filled('dpd_id')) {
            $query->where('dpd_id', $request->dpd_id);
        }

        if ($request->filled('dpc_id')) {
            $query->where('dpc_id', $request->dpc_id);
        }

        if ($request->filled('upa_id')) {
            $query->whereHas('upa', function($q) use ($request) {
                $q->where('upa_id', $request->upa_id);
            });
        }

        if ($request->filled('status_dpc')) {
            if ($request->status_dpc == 'aktif') {
                $query->where('is_active', true);
            } elseif ($request->status_dpc == 'tidak_aktif') {
                $query->where('is_active', false);
            }
        }

        // (Status UPA filter removed)

        if ($request->filled('jenjang_id')) {
            $query->whereHas('anggota', function($q) use ($request) {
                $q->where('ref_jenjang_id', $request->jenjang_id)
                  ->where('status_keanggotaan', 'Aktif');
            });
        }

        // (Min. jumlah anggota filter removed)

        $dpcList = $query->orderBy('nama_dpc')->get();

        return view('komposisi.print-sk', compact('dpcList'));
    }

    /**
     * Get statistik komposisi keseluruhan
     */
    private function getStatistikKomposisi($request = null)
    {
        $dpcQuery = Dpc::active();
        $upaQuery = Upa::active();
        $anggotaQuery = Anggota::where('status_keanggotaan', 'Aktif');

        // Apply filters if provided
        if ($request) {
            if ($request->filled('dpd_id')) {
                $dpcQuery->where('dpd_id', $request->dpd_id);
                $upaQuery->whereHas('dpc', function($q) use ($request) {
                    $q->where('dpd_id', $request->dpd_id);
                });
                $anggotaQuery->whereHas('dpc', function($q) use ($request) {
                    $q->where('dpd_id', $request->dpd_id);
                });
            }

            if ($request->filled('dpc_id')) {
                $dpcQuery->where('dpc_id', $request->dpc_id);
                $upaQuery->where('dpc_id', $request->dpc_id);
                $anggotaQuery->where('dpc_id', $request->dpc_id);
            }

            if ($request->filled('upa_id')) {
                $dpcQuery->whereHas('upa', function($q) use ($request) {
                    $q->where('upa_id', $request->upa_id);
                });
                $upaQuery->where('upa_id', $request->upa_id);
                $anggotaQuery->whereHas('upa', function($q) use ($request) {
                    $q->where('upa_id', $request->upa_id);
                });
            }

            if ($request->filled('status_dpc')) {
                if ($request->status_dpc == 'aktif') {
                    $dpcQuery->where('is_active', true);
                } elseif ($request->status_dpc == 'tidak_aktif') {
                    $dpcQuery->where('is_active', false);
                }
                $upaQuery->whereHas('dpc', function($q) use ($request) {
                    if ($request->status_dpc == 'aktif') {
                        $q->where('is_active', true);
                    } elseif ($request->status_dpc == 'tidak_aktif') {
                        $q->where('is_active', false);
                    }
                });
                $anggotaQuery->whereHas('dpc', function($q) use ($request) {
                    if ($request->status_dpc == 'aktif') {
                        $q->where('is_active', true);
                    } elseif ($request->status_dpc == 'tidak_aktif') {
                        $q->where('is_active', false);
                    }
                });
            }

            // (Status UPA filter removed)

            if ($request->filled('jenjang_id')) {
                $anggotaQuery->where('ref_jenjang_id', $request->jenjang_id);
            }
        }

        $totalDpc = $dpcQuery->count();
        $totalUpa = $upaQuery->count();
        $totalAnggota = $anggotaQuery->count();

        // Statistik per jenjang
        $statistikJenjang = $anggotaQuery->clone()
            ->join('tbl_ref_jenjang', 'tbl_anggota.ref_jenjang_id', '=', 'tbl_ref_jenjang.ref_jenjang_id')
            ->select('tbl_ref_jenjang.jenjang', DB::raw('COUNT(*) as total'))
            ->groupBy('tbl_ref_jenjang.jenjang')
            ->orderBy('tbl_ref_jenjang.jenjang')
            ->get();

        // Statistik per DPC
        $statistikDpc = $dpcQuery->clone()
            ->withCount(['anggota' => function($query) {
                $query->where('status_keanggotaan', 'Aktif');
            }])
            ->withCount(['upa' => function($query) {
                $query->active();
            }])
            ->get();

        return [
            'total_dpc' => $totalDpc,
            'total_upa' => $totalUpa,
            'total_anggota' => $totalAnggota,
            'statistik_jenjang' => $statistikJenjang,
            'statistik_dpc' => $statistikDpc
        ];
    }

    /**
     * Get statistik DPC tertentu
     */
    private function getStatistikDpc($dpcId)
    {
        $totalUpa = Upa::where('dpc_id', $dpcId)->active()->count();
        $totalAnggota = Anggota::where('dpc_id', $dpcId)
            ->where('status_keanggotaan', 'Aktif')
            ->count();

        // Statistik per jenjang di DPC ini
        $statistikJenjang = Anggota::where('dpc_id', $dpcId)
            ->where('status_keanggotaan', 'Aktif')
            ->join('tbl_ref_jenjang', 'tbl_anggota.ref_jenjang_id', '=', 'tbl_ref_jenjang.ref_jenjang_id')
            ->select('tbl_ref_jenjang.jenjang', DB::raw('COUNT(*) as total'))
            ->groupBy('tbl_ref_jenjang.jenjang')
            ->orderBy('tbl_ref_jenjang.jenjang')
            ->get();

        // Statistik per UPA di DPC ini
        $statistikUpa = Upa::where('dpc_id', $dpcId)
            ->active()
            ->withCount(['anggota' => function($query) {
                $query->where('status_keanggotaan', 'Aktif');
            }])
            ->get();

        return [
            'total_upa' => $totalUpa,
            'total_anggota' => $totalAnggota,
            'statistik_jenjang' => $statistikJenjang,
            'statistik_upa' => $statistikUpa
        ];
    }

    /**
     * Get statistik UPA tertentu
     */
    private function getStatistikUpa($upaId)
    {
        $totalAnggota = Anggota::where('upa_id', $upaId)
            ->where('status_keanggotaan', 'Aktif')
            ->count();

        // Statistik per jenjang di UPA ini
        $statistikJenjang = Anggota::where('upa_id', $upaId)
            ->where('status_keanggotaan', 'Aktif')
            ->join('tbl_ref_jenjang', 'tbl_anggota.ref_jenjang_id', '=', 'tbl_ref_jenjang.ref_jenjang_id')
            ->select('tbl_ref_jenjang.jenjang', DB::raw('COUNT(*) as total'))
            ->groupBy('tbl_ref_jenjang.jenjang')
            ->orderBy('tbl_ref_jenjang.jenjang')
            ->get();

        // Data pembimbing aktif
        $pembimbingAktif = Anggota::whereHas('upaPembimbing', function($query) use ($upaId) {
            $query->where('upa_id', $upaId)
                  ->where('status', 'Aktif');
        })
        ->with('refJenjang')
        ->get();

        return [
            'total_anggota' => $totalAnggota,
            'statistik_jenjang' => $statistikJenjang,
            'pembimbing_aktif' => $pembimbingAktif
        ];
    }
}
