<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Kegiatan;
use App\Models\KegiatanPartisipanHierarchical;
use App\Models\Dpd;
use App\Models\Dpc;
use App\Models\Upa;
use App\Models\Anggota;
use App\Models\JenisKegiatan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Helpers\CacheHelper;

class KegiatanHierarchicalController extends Controller
{
    /**
     * Clear cache yang terkait dengan data master
     */
    private function clearMasterDataCache()
    {
        CacheHelper::clearMasterDataCache();
    }

    /**
     * API untuk mendapatkan DPC berdasarkan DPD
     */
    public function getDpcByDpd(Request $request)
    {
        try {
            $dpdId = $request->input('dpd_id');
            
            if (!$dpdId) {
                return response()->json(['data' => []]);
            }

            $dpc = Dpc::where('dpd_id', $dpdId)
                     ->where('is_active', true)
                     ->select(['dpc_id', 'nama_dpc', 'kode_dpc'])
                     ->orderBy('nama_dpc')
                     ->get();

            return response()->json(['data' => $dpc]);
        } catch (\Exception $e) {
            \Log::error('Error in getDpcByDpd: ' . $e->getMessage());
            return response()->json(['error' => 'Error loading DPC data'], 500);
        }
    }

    /**
     * API untuk mendapatkan UPA berdasarkan DPC
     */
    public function getUpaByDpc(Request $request)
    {
        try {
            $dpcId = $request->input('dpc_id');
            
            if (!$dpcId) {
                return response()->json(['data' => []]);
            }

            $upa = Upa::where('dpc_id', $dpcId)
                     ->where('status', 'Aktif')
                     ->select(['upa_id', 'nama_upa', 'kode_upa'])
                     ->orderBy('nama_upa')
                     ->get();

            return response()->json(['data' => $upa]);
        } catch (\Exception $e) {
            \Log::error('Error in getUpaByDpc: ' . $e->getMessage());
            return response()->json(['error' => 'Error loading UPA data'], 500);
        }
    }

    /**
     * API untuk mendapatkan Anggota berdasarkan UPA
     */
    public function getAnggotaByUpa(Request $request)
    {
        try {
            $upaId = $request->input('upa_id');
            
            if (!$upaId) {
                return response()->json(['data' => []]);
            }

            $anggota = Anggota::where('upa_id', $upaId)
                            ->where('status_keanggotaan', 'Aktif')
                            ->select(['id', 'nama_lengkap', 'nomor_identitas_anggota', 'jenis_kelamin'])
                            ->orderBy('nama_lengkap')
                            ->get();

            return response()->json(['data' => $anggota]);
        } catch (\Exception $e) {
            \Log::error('Error in getAnggotaByUpa: ' . $e->getMessage());
            return response()->json(['error' => 'Error loading Anggota data'], 500);
        }
    }

    /**
     * API untuk mendapatkan semua anggota berdasarkan hierarki yang dipilih
     */
    public function getAnggotaByHierarchy(Request $request)
    {
        try {
            $dpdId = $request->input('dpd_id');
            $dpcId = $request->input('dpc_id');
            $upaId = $request->input('upa_id');
            $anggotaId = $request->input('anggota_id');

            $query = Anggota::where('status_keanggotaan', 'Aktif')
                          ->select(['id', 'nama_lengkap', 'nomor_identitas_anggota', 'jenis_kelamin', 'upa_id', 'dpc_id', 'dpd_id'])
                          ->with([
                              'upa:upa_id,nama_upa',
                              'dpc:dpc_id,nama_dpc',
                              'dpd:dpd_id,nama_dpd'
                          ]);

            // Apply filters based on hierarchy
            if ($dpdId) {
                $query->where('dpd_id', $dpdId);
            }

            if ($dpcId) {
                $query->where('dpc_id', $dpcId);
            }

            if ($upaId) {
                $query->where('upa_id', $upaId);
            }

            if ($anggotaId) {
                $query->where('id', $anggotaId);
            }

            $anggota = $query->orderBy('nama_lengkap')->get();

            return response()->json(['data' => $anggota]);
        } catch (\Exception $e) {
            \Log::error('Error in getAnggotaByHierarchy: ' . $e->getMessage());
            return response()->json(['error' => 'Error loading Anggota data'], 500);
        }
    }

    public function create()
    {
        // Get all active DPD for selection
        $dpds = cache()->remember('dpd_active_list', 3600, function() {
            return Dpd::active()->select(['dpd_id', 'nama_dpd', 'kode_dpd'])->orderBy('nama_dpd')->get();
        });
        
        // Cache data yang jarang berubah
        $users = cache()->remember('users_list', 300, function() {
            return \App\Models\User::select(['id', 'name'])->orderBy('name')->get();
        });
        
        $jenisKegiatans = cache()->remember('jenis_kegiatan_active', 600, function() {
            return JenisKegiatan::active()->select(['ref_jenis_kegiatan_id', 'nama_jenis_kegiatan'])->orderBy('nama_jenis_kegiatan')->get();
        });
        
        return view('kegiatan.create-hierarchical', compact('dpds', 'users', 'jenisKegiatans'));
    }

    public function store(Request $request)
    {
        // Determine validation rules based on event type
        $eventType = $request->input('event_type', 'single');
        
        $rules = [
            'nama_kegiatan' => 'required|string|max:255',
            'deskripsi' => 'nullable|string',
            'event_type' => 'required|in:single,multi',
            'tempat' => 'required|string|max:500',
            'ref_jenis_kegiatan_id' => 'required',
            'jenis_kegiatan_lainnya' => 'required_if:ref_jenis_kegiatan_id,lainnya|nullable|string|max:255',
            'target_peserta' => 'nullable|integer|min:1',
            'catatan' => 'nullable|string',
            'pic_anggota_id' => 'nullable|exists:tbl_anggota,id',
            'partisipan_hierarchical' => 'nullable|array',
            'partisipan_hierarchical.*.selection_type' => 'required|in:dpd,dpc,upa,anggota',
            'partisipan_hierarchical.*.status_partisipasi' => 'required|in:Wajib,Opsional,Undangan',
        ];
        
        // Add conditional validation rules based on event type
        if ($eventType === 'single') {
            $rules['tanggal_kegiatan'] = 'required|date';
            $rules['waktu_mulai'] = 'required';
            $rules['waktu_selesai'] = 'nullable|after:waktu_mulai';
        } else {
            $rules['tanggal_mulai'] = 'required|date';
            $rules['tanggal_selesai'] = 'required|date|after_or_equal:tanggal_mulai';
            $rules['waktu_mulai_multi'] = 'required';
            $rules['waktu_selesai_multi'] = 'nullable|after:waktu_mulai_multi';
        }
        
        $messages = [
            'nama_kegiatan.required' => 'Nama kegiatan wajib diisi.',
            'tanggal_kegiatan.required' => 'Tanggal kegiatan wajib diisi.',
            'waktu_mulai.required' => 'Waktu mulai wajib diisi.',
            'waktu_selesai.after' => 'Waktu selesai harus setelah waktu mulai.',
            'tempat.required' => 'Tempat kegiatan wajib diisi.',
            'jenis_kegiatan.required' => 'Jenis kegiatan wajib dipilih.',
            'target_peserta.min' => 'Target peserta minimal 1 orang.',
        ];

        // Custom validation untuk jenis kegiatan
        if ($request->ref_jenis_kegiatan_id === 'lainnya') {
            if (empty($request->jenis_kegiatan_lainnya)) {
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['jenis_kegiatan_lainnya' => 'Jenis kegiatan lainnya wajib diisi.']);
            }
        } else {
            // Validasi bahwa ref_jenis_kegiatan_id ada di database
            if (!JenisKegiatan::where('ref_jenis_kegiatan_id', $request->ref_jenis_kegiatan_id)->exists()) {
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['ref_jenis_kegiatan_id' => 'Jenis kegiatan yang dipilih tidak valid.']);
            }
        }

        $request->validate($rules, $messages);

        try {
            DB::beginTransaction();

            // Prepare data untuk create based on event type
            $kegiatanData = $request->only([
                'nama_kegiatan', 'deskripsi', 'tempat', 'target_peserta', 
                'catatan', 'pic_anggota_id'
            ]);
            
            // Handle date/time fields based on event type
            if ($eventType === 'single') {
                $kegiatanData['tanggal_kegiatan'] = $request->tanggal_kegiatan;
                $kegiatanData['tanggal_mulai'] = $request->tanggal_kegiatan;
                $kegiatanData['tanggal_selesai'] = $request->tanggal_kegiatan;
                $kegiatanData['waktu_mulai'] = $request->waktu_mulai;
                $kegiatanData['waktu_selesai'] = $request->waktu_selesai;
            } else {
                $kegiatanData['tanggal_mulai'] = $request->tanggal_mulai;
                $kegiatanData['tanggal_selesai'] = $request->tanggal_selesai;
                $kegiatanData['tanggal_kegiatan'] = $request->tanggal_mulai; // Set as start date for compatibility
                $kegiatanData['waktu_mulai'] = $request->waktu_mulai_multi;
                $kegiatanData['waktu_selesai'] = $request->waktu_selesai_multi;
            }
            
            // Handle jenis kegiatan
            if ($request->ref_jenis_kegiatan_id === 'lainnya') {
                $kegiatanData['ref_jenis_kegiatan_id'] = null;
                $kegiatanData['jenis_kegiatan_lainnya'] = $request->jenis_kegiatan_lainnya;
            } else {
                $kegiatanData['ref_jenis_kegiatan_id'] = $request->ref_jenis_kegiatan_id;
                $kegiatanData['jenis_kegiatan_lainnya'] = null;
            }

            // Create kegiatan
            $kegiatan = Kegiatan::create($kegiatanData);

            // Add hierarchical partisipan jika ada
            if ($request->filled('partisipan_hierarchical')) {
                foreach ($request->partisipan_hierarchical as $partisipan) {
                    $partisipanData = [
                        'kegiatan_id' => $kegiatan->id,
                        'selection_type' => $partisipan['selection_type'],
                        'status_partisipasi' => $partisipan['status_partisipasi'],
                        'status_kehadiran' => 'Belum Konfirmasi'
                    ];

                    // Set the appropriate ID based on selection type
                    switch ($partisipan['selection_type']) {
                        case 'dpd':
                            $partisipanData['dpd_id'] = $partisipan['dpd_id'];
                            break;
                        case 'dpc':
                            $partisipanData['dpc_id'] = $partisipan['dpc_id'];
                            break;
                        case 'upa':
                            $partisipanData['upa_id'] = $partisipan['upa_id'];
                            break;
                        case 'anggota':
                            $partisipanData['anggota_id'] = $partisipan['anggota_id'];
                            break;
                    }

                    KegiatanPartisipanHierarchical::create($partisipanData);
                }
            }

            DB::commit();
            
            return redirect()->route('kegiatan.show', $kegiatan->id)
                           ->with('success', 'Kegiatan berhasil dibuat!');
                           
        } catch (\Exception $e) {
            DB::rollback();
            
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Gagal membuat kegiatan: ' . $e->getMessage());
        }
    }
}
