<?php

namespace App\Http\Controllers;

use App\Models\Dpc;
use App\Models\Dpd;
use Illuminate\Http\Request;

class DpcController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Dpc::with('dpd');
        
        // Filter berdasarkan pencarian
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_dpc', 'like', "%{$search}%")
                  ->orWhere('kode_dpc', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%");
            });
        }
        
        // Filter berdasarkan status aktif
        if ($request->has('status') && $request->status !== '') {
            if ($request->status == 'aktif') {
                $query->where('is_active', true);
            } elseif ($request->status == 'tidak_aktif') {
                $query->where('is_active', false);
            }
        }
        
        // Filter berdasarkan DPD
        if ($request->has('dpd_id') && $request->dpd_id) {
            $query->where('dpd_id', $request->dpd_id);
        }
        
        $dpcs = $query->orderBy('nama_dpc')->paginate(20);
        $dpds = Dpd::active()->orderBy('nama_dpd')->get();
        
        return view('dpc.index', compact('dpcs', 'dpds'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $dpds = Dpd::active()->orderBy('nama_dpd')->get();
        return view('dpc.create', compact('dpds'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'nama_dpc' => 'required|string|max:100',
            'kode_dpc' => 'required|string|max:10|unique:tbl_dpc,kode_dpc',
            'dpd_id' => 'nullable|exists:tbl_dpd,dpd_id',
            'deskripsi' => 'nullable|string',
            'is_active' => 'boolean'
        ], [
            'nama_dpc.required' => 'Nama DPC harus diisi',
            'nama_dpc.max' => 'Nama DPC maksimal 100 karakter',
            'kode_dpc.required' => 'Kode DPC harus diisi',
            'kode_dpc.max' => 'Kode DPC maksimal 10 karakter',
            'kode_dpc.unique' => 'Kode DPC sudah digunakan',
            'dpd_id.exists' => 'DPD tidak valid'
        ]);

        // Set default is_active jika tidak ada
        if (!isset($validated['is_active'])) {
            $validated['is_active'] = false;
        }

        Dpc::create($validated);

        return redirect()->route('dpc.index')
                        ->with('success', 'DPC berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Dpc $dpc)
    {
        // Load relationships dengan eager loading untuk performa
        $dpc->load([
            'dpd', 
            'upa' => function($query) {
                $query->with(['anggota', 'pembimbingAktif'])
                      ->orderBy('nama_upa');
            }, 
            'anggota'
        ]);
        
        return view('dpc.show', compact('dpc'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Dpc $dpc)
    {
        $dpds = Dpd::active()->orderBy('nama_dpd')->get();
        return view('dpc.edit', compact('dpc', 'dpds'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Dpc $dpc)
    {
        $validated = $request->validate([
            'nama_dpc' => 'required|string|max:100',
            'kode_dpc' => 'required|string|max:10|unique:tbl_dpc,kode_dpc,' . $dpc->dpc_id . ',dpc_id',
            'dpd_id' => 'nullable|exists:tbl_dpd,dpd_id',
            'deskripsi' => 'nullable|string',
            'is_active' => 'boolean'
        ], [
            'nama_dpc.required' => 'Nama DPC harus diisi',
            'nama_dpc.max' => 'Nama DPC maksimal 100 karakter',
            'kode_dpc.required' => 'Kode DPC harus diisi',
            'kode_dpc.max' => 'Kode DPC maksimal 10 karakter',
            'kode_dpc.unique' => 'Kode DPC sudah digunakan',
            'dpd_id.exists' => 'DPD tidak valid'
        ]);

        // Set default is_active jika tidak ada
        if (!isset($validated['is_active'])) {
            $validated['is_active'] = false;
        }

        $dpc->update($validated);

        return redirect()->route('dpc.index')
                        ->with('success', 'DPC berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Dpc $dpc)
    {
        // Check jika DPC masih memiliki anggota atau UPA
        if ($dpc->anggota()->count() > 0 || $dpc->upa()->count() > 0) {
            return redirect()->route('dpc.index')
                           ->with('error', 'DPC tidak dapat dihapus karena masih memiliki anggota atau UPA terkait.');
        }

        $dpc->delete();

        return redirect()->route('dpc.index')
                        ->with('success', 'DPC berhasil dihapus.');
    }

    /**
     * Toggle status aktif/tidak aktif
     */
    public function toggleStatus(Dpc $dpc)
    {
        $dpc->update(['is_active' => !$dpc->is_active]);
        
        $status = $dpc->is_active ? 'diaktifkan' : 'dinonaktifkan';
        
        return redirect()->route('dpc.index')
                        ->with('success', "DPC berhasil {$status}.");
    }
}

