<?php

namespace App\Helpers;

use App\Models\Anggota;
use App\Models\RefJenjang;
use App\Models\Dpd;
use App\Models\Dpc;

class CacheHelper
{
    /**
     * Get cached master data untuk dropdown
     * Cache lebih lama karena data master jarang berubah
     */
    public static function getMasterData()
    {
        return [
            'statusList' => cache()->remember('anggota_status_list', 1800, function() {
                return Anggota::select('status_keanggotaan')
                             ->distinct()
                             ->pluck('status_keanggotaan');
            }),
            
            'jenjangList' => cache()->remember('ref_jenjang_active', 3600, function() {
                return RefJenjang::active()->select('ref_jenjang_id', 'jenjang')->get();
            }),
            
            'provinsiList' => cache()->remember('anggota_provinsi_list', 1800, function() {
                return Anggota::select('provinsi')
                             ->distinct()
                             ->whereNotNull('provinsi')
                             ->pluck('provinsi');
            }),
            
            'dpdList' => cache()->remember('dpd_active_list', 3600, function() {
                return Dpd::active()->select('dpd_id', 'nama_dpd')->get();
            }),
            
            'dpcList' => cache()->remember('dpc_active_list', 3600, function() {
                return Dpc::active()->select('dpc_id', 'nama_dpc')->get();
            })
        ];
    }
    
    /**
     * Clear semua cache master data
     */
    public static function clearMasterDataCache()
    {
        cache()->forget('anggota_status_list');
        cache()->forget('ref_jenjang_active');
        cache()->forget('anggota_provinsi_list');
        cache()->forget('dpd_active_list');
        cache()->forget('dpc_active_list');
        cache()->forget('jenis_kegiatan_active');
        cache()->forget('users_list');
    }
    
    /**
     * Clear cache dengan pattern matching
     */
    public static function clearPatternCache($pattern)
    {
        // Hanya jalankan jika menggunakan Redis driver
        if (config('cache.default') === 'redis' && method_exists(cache()->getStore(), 'getRedis')) {
            $keys = cache()->getRedis()->keys($pattern);
            foreach ($keys as $key) {
                cache()->forget(str_replace(config('cache.prefix') . ':', '', $key));
            }
        } else {
            // Untuk driver lain, clear cache yang umum
            cache()->flush();
        }
    }
    
    /**
     * Clear cache untuk model tertentu
     */
    public static function clearModelCache($model, $id = null)
    {
        $modelName = strtolower(class_basename($model));
        
        if ($id) {
            // Clear cache untuk model spesifik
            cache()->forget("{$modelName}_umur_{$id}");
            cache()->forget("{$modelName}_lama_keanggotaan_{$id}");
        } else {
            // Clear semua cache untuk model
            self::clearPatternCache("*{$modelName}*");
        }
    }
    
    /**
     * Warm up cache untuk data yang sering digunakan
     */
    public static function warmUpCache()
    {
        // Warm up master data
        self::getMasterData();
        
        // Warm up dashboard stats
        cache()->remember('dashboard_stats', 300, function() {
            return [
                'total_anggota' => \App\Models\Anggota::count(),
                'anggota_aktif' => \App\Models\Anggota::where('status_keanggotaan', 'Aktif')->count(),
            ];
        });
        
        return true;
    }
}
