<?php

namespace App\Exports;

use App\Models\YaumiyahRecord;
use App\Models\YaumiyahActivity;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithTitle;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class YaumiyahRecordExport implements FromCollection, WithHeadings, WithMapping, WithStyles, WithColumnWidths, WithTitle
{
    protected $filters;

    public function __construct($filters = [])
    {
        $this->filters = $filters;
    }

    public function collection()
    {
        $query = YaumiyahRecord::with(['anggota', 'activity']);

        // Apply filters
        if (isset($this->filters['anggota_id']) && $this->filters['anggota_id']) {
            $query->where('anggota_id', $this->filters['anggota_id']);
        }

        if (isset($this->filters['week']) && $this->filters['week']) {
            $query->where('week_number', $this->filters['week']);
        }

        if (isset($this->filters['year']) && $this->filters['year']) {
            $query->where('year', $this->filters['year']);
        }

        return $query->orderBy('year', 'desc')
            ->orderBy('week_number', 'desc')
            ->orderBy('anggota_id')
            ->get();
    }

    public function headings(): array
    {
        return [
            'No',
            'NIA',
            'Nama Anggota',
            'Pekan',
            'Tahun',
            'Aktivitas',
            'Nilai',
            'Unit',
            'Target/Minggu',
            'Input Oleh',
            'Tanggal Input'
        ];
    }

    public function map($record): array
    {
        static $counter = 0;
        $counter++;

        $activity = $record->activity;
        $anggota = $record->anggota;

        return [
            $counter,
            $anggota ? $anggota->nomor_identitas_anggota : '-',
            $anggota ? $anggota->nama_lengkap : '-',
            $record->week_number,
            $record->year,
            $activity ? $activity->name : '-',
            $record->value,
            $activity ? $activity->unit : '-',
            $activity ? $activity->target_per_week : '-',
            $record->created_by_name ?? '-',
            $record->created_at ? \Carbon\Carbon::parse($record->created_at)->format('d/m/Y H:i') : '-',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            // Style the first row as bold text with background
            1 => [
                'font' => ['bold' => true, 'size' => 12, 'color' => ['rgb' => 'FFFFFF']],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4']
                ],
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
            ],
        ];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 8,   // No
            'B' => 15,  // NIA
            'C' => 30,  // Nama Anggota
            'D' => 10,  // Pekan
            'E' => 12,  // Tahun
            'F' => 40,  // Aktivitas
            'G' => 12,  // Nilai
            'H' => 15,  // Unit
            'I' => 18,  // Target/Minggu
            'J' => 20,  // Input Oleh
            'K' => 20,  // Tanggal Input
        ];
    }

    public function title(): string
    {
        $title = 'Data Yaumiyah';
        
        if (isset($this->filters['year']) && $this->filters['year']) {
            $title .= ' Tahun ' . $this->filters['year'];
        }
        
        if (isset($this->filters['week']) && $this->filters['week']) {
            $title .= ' Pekan ' . $this->filters['week'];
        }

        return $title;
    }
}
