<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class PerformanceMonitor extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'performance:monitor {--clear : Clear performance cache}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Monitor and optimize application performance';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        if ($this->option('clear')) {
            $this->clearPerformanceCache();
            return;
        }

        $this->info('Performance Monitoring Report');
        $this->line('============================');

        // Check cache performance
        $this->checkCachePerformance();
        
        // Check database performance
        $this->checkDatabasePerformance();
        
        // Check view performance
        $this->checkViewPerformance();
        
        // Provide optimization recommendations
        $this->provideRecommendations();
    }

    private function checkCachePerformance()
    {
        $this->info("\n📊 Cache Performance:");
        
        $cacheKeys = [
            'anggota_status_list',
            'ref_jenjang_active',
            'anggota_provinsi_list'
        ];
        
        $hitCount = 0;
        $totalCount = count($cacheKeys);
        
        foreach ($cacheKeys as $key) {
            if (Cache::has($key)) {
                $hitCount++;
                $this->line("✓ {$key} - Cached");
            } else {
                $this->line("✗ {$key} - Not cached");
            }
        }
        
        $hitRate = ($hitCount / $totalCount) * 100;
        $this->line("Cache Hit Rate: {$hitRate}%");
        
        if ($hitRate < 80) {
            $this->warn("⚠️  Low cache hit rate. Consider warming up cache.");
        }
    }

    private function checkDatabasePerformance()
    {
        $this->info("\n🗄️  Database Performance:");
        
        // Check if indexes exist (simplified check)
        $tables = ['tbl_anggota', 'tbl_ref_jenjang', 'tbl_dpc', 'tbl_dpd'];
        
        foreach ($tables as $table) {
            try {
                $indexes = DB::select("SHOW INDEX FROM {$table}");
                $indexCount = count($indexes);
                $this->line("✓ {$table} - {$indexCount} indexes");
            } catch (\Exception $e) {
                $this->line("✗ {$table} - Error checking indexes");
            }
        }
    }

    private function checkViewPerformance()
    {
        $this->info("\n🎨 View Performance:");
        
        $viewPath = resource_path('views');
        $bladeFiles = glob($viewPath . '/**/*.blade.php');
        
        $totalFiles = count($bladeFiles);
        $largeFiles = 0;
        
        foreach ($bladeFiles as $file) {
            $size = filesize($file);
            if ($size > 50000) { // Files larger than 50KB
                $largeFiles++;
                $this->warn("⚠️  Large view file: " . basename($file) . " ({$size} bytes)");
            }
        }
        
        $this->line("Total view files: {$totalFiles}");
        $this->line("Large files (>50KB): {$largeFiles}");
        
        if ($largeFiles > 0) {
            $this->warn("⚠️  Consider splitting large view files for better performance.");
        }
    }

    private function provideRecommendations()
    {
        $this->info("\n💡 Optimization Recommendations:");
        
        $recommendations = [
            "1. Enable OPcache in PHP for better performance",
            "2. Use Redis for caching instead of file-based cache",
            "3. Implement database query caching",
            "4. Minify CSS and JavaScript assets",
            "5. Use CDN for static assets",
            "6. Implement lazy loading for images",
            "7. Enable Gzip compression",
            "8. Use database indexes for frequently queried columns",
            "9. Implement pagination for large datasets",
            "10. Use eager loading to prevent N+1 queries"
        ];
        
        foreach ($recommendations as $recommendation) {
            $this->line($recommendation);
        }
    }

    private function clearPerformanceCache()
    {
        $this->info('Clearing performance cache...');
        
        $cacheKeys = [
            'anggota_status_list',
            'ref_jenjang_active',
            'anggota_provinsi_list'
        ];
        
        foreach ($cacheKeys as $key) {
            Cache::forget($key);
        }
        
        // Clear view cache
        \Artisan::call('view:clear');
        
        // Clear config cache
        \Artisan::call('config:clear');
        
        // Clear route cache
        \Artisan::call('route:clear');
        
        $this->info('Performance cache cleared successfully!');
    }
}


