<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class OptimizeAssets extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'assets:optimize {--force : Force optimization even if files exist}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Optimize CSS and JS assets for better performance';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting asset optimization...');

        // Optimize CSS
        $this->optimizeCSS();
        
        // Optimize JS
        $this->optimizeJS();
        
        // Generate critical CSS
        $this->generateCriticalCSS();
        
        $this->info('Asset optimization completed!');
    }

    private function optimizeCSS()
    {
        $this->info('Optimizing CSS files...');
        
        $cssPath = public_path('css');
        if (!File::exists($cssPath)) {
            File::makeDirectory($cssPath, 0755, true);
        }

        // Minify app.css
        $appCss = resource_path('css/app.css');
        if (File::exists($appCss)) {
            $content = File::get($appCss);
            $minified = $this->minifyCSS($content);
            File::put(public_path('css/app.min.css'), $minified);
            $this->line('✓ app.css minified');
        }
    }

    private function optimizeJS()
    {
        $this->info('Optimizing JS files...');
        
        $jsPath = public_path('js');
        if (!File::exists($jsPath)) {
            File::makeDirectory($jsPath, 0755, true);
        }

        // Minify app.js
        $appJs = resource_path('js/app.js');
        if (File::exists($appJs)) {
            $content = File::get($appJs);
            $minified = $this->minifyJS($content);
            File::put(public_path('js/app.min.js'), $minified);
            $this->line('✓ app.js minified');
        }
    }

    private function generateCriticalCSS()
    {
        $this->info('Generating critical CSS...');
        
        $criticalCSS = '
        /* Critical CSS for above-the-fold content */
        .card { border: 1px solid #e3e6f0; border-radius: 0.35rem; }
        .table th, .table td { padding: 15px 10px; vertical-align: middle; }
        .btn { display: inline-block; font-weight: 400; text-align: center; }
        .badge { font-size: 0.875rem; padding: 0.5rem 0.75rem; }
        .loading { opacity: 0.6; pointer-events: none; }
        ';
        
        File::put(public_path('css/critical.min.css'), $this->minifyCSS($criticalCSS));
        $this->line('✓ Critical CSS generated');
    }

    private function minifyCSS($css)
    {
        // Remove comments
        $css = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $css);
        
        // Remove unnecessary whitespace
        $css = str_replace(["\r\n", "\r", "\n", "\t"], '', $css);
        $css = preg_replace('/\s+/', ' ', $css);
        
        // Remove spaces around specific characters
        $css = preg_replace('/\s*{\s*/', '{', $css);
        $css = preg_replace('/;\s*/', ';', $css);
        $css = preg_replace('/\s*}\s*/', '}', $css);
        $css = preg_replace('/\s*,\s*/', ',', $css);
        $css = preg_replace('/\s*:\s*/', ':', $css);
        
        return trim($css);
    }

    private function minifyJS($js)
    {
        // Remove single-line comments
        $js = preg_replace('~//[^\r\n]*~', '', $js);
        
        // Remove multi-line comments
        $js = preg_replace('~/\*.*?\*/~s', '', $js);
        
        // Remove unnecessary whitespace
        $js = preg_replace('/\s+/', ' ', $js);
        $js = preg_replace('/\s*([{}();,=])\s*/', '$1', $js);
        
        return trim($js);
    }
}


