<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;

class OptimizeApplication extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:optimize {--force : Force optimization without confirmation}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Optimize Laravel application for production';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        if (!$this->option('force') && !$this->confirm('This will optimize the application for production. Continue?')) {
            $this->info('Optimization cancelled.');
            return;
        }

        $this->info('Starting application optimization...');

        // Clear all caches
        $this->clearCaches();

        // Optimize for production
        $this->optimizeForProduction();

        // Generate optimized files
        $this->generateOptimizedFiles();

        $this->info('Application optimization completed successfully!');
    }

    protected function clearCaches()
    {
        $this->info('Clearing caches...');
        
        Artisan::call('cache:clear');
        Artisan::call('config:clear');
        Artisan::call('route:clear');
        Artisan::call('view:clear');
        
        $this->info('✓ Caches cleared');
    }

    protected function optimizeForProduction()
    {
        $this->info('Optimizing for production...');
        
        // Cache configuration
        Artisan::call('config:cache');
        $this->info('✓ Configuration cached');
        
        // Cache routes
        Artisan::call('route:cache');
        $this->info('✓ Routes cached');
        
        // Cache views
        Artisan::call('view:cache');
        $this->info('✓ Views cached');
        
        // Optimize autoloader
        Artisan::call('optimize');
        $this->info('✓ Autoloader optimized');
    }

    protected function generateOptimizedFiles()
    {
        $this->info('Generating optimized files...');
        
        // Create .htaccess for Apache
        $this->createHtaccess();
        
        // Create robots.txt
        $this->createRobotsTxt();
        
        $this->info('✓ Optimized files generated');
    }

    protected function createHtaccess()
    {
        $htaccess = <<<'EOT'
# Laravel Optimization
<IfModule mod_rewrite.c>
    RewriteEngine On
    
    # Handle Authorization Header
    RewriteCond %{HTTP:Authorization} .
    RewriteRule .* - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]
    
    # Redirect Trailing Slashes If Not A Folder...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_URI} (.+)/$
    RewriteRule ^ %1 [L,R=301]
    
    # Send Requests To Front Controller...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteRule ^ index.php [L]
</IfModule>

# Security Headers
<IfModule mod_headers.c>
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Referrer-Policy "strict-origin-when-cross-origin"
</IfModule>

# Compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/plain
    AddOutputFilterByType DEFLATE text/html
    AddOutputFilterByType DEFLATE text/xml
    AddOutputFilterByType DEFLATE text/css
    AddOutputFilterByType DEFLATE application/xml
    AddOutputFilterByType DEFLATE application/xhtml+xml
    AddOutputFilterByType DEFLATE application/rss+xml
    AddOutputFilterByType DEFLATE application/javascript
    AddOutputFilterByType DEFLATE application/x-javascript
</IfModule>

# Browser Caching
<IfModule mod_expires.c>
    ExpiresActive on
    ExpiresByType text/css "access plus 1 year"
    ExpiresByType application/javascript "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/ico "access plus 1 year"
    ExpiresByType image/icon "access plus 1 year"
    ExpiresByType text/plain "access plus 1 month"
    ExpiresByType application/pdf "access plus 1 month"
</IfModule>
EOT;

        File::put(public_path('.htaccess'), $htaccess);
    }

    protected function createRobotsTxt()
    {
        $robots = <<<'EOT'
User-agent: *
Disallow: /admin/
Disallow: /api/
Disallow: /storage/
Disallow: /vendor/
Disallow: /bootstrap/
Disallow: /config/
Disallow: /database/
Disallow: /resources/
Disallow: /routes/
Disallow: /tests/
Disallow: /app/
Disallow: /artisan
Disallow: /composer.json
Disallow: /composer.lock
Disallow: /package.json
Disallow: /package-lock.json
Disallow: /phpunit.xml
Disallow: /webpack.mix.js
Disallow: /yarn.lock

Sitemap: https://yourdomain.com/sitemap.xml
EOT;

        File::put(public_path('robots.txt'), $robots);
    }
}