# 🚀 Panduan Optimasi Laravel untuk Production

## 📋 Daftar Optimasi yang Sudah Diterapkan

### ✅ **1. Database Optimizations**
- **Eager Loading**: Menambahkan `with()` untuk mengurangi N+1 queries
- **Pagination**: Mengurangi dari 20 ke 15 item per halaman
- **Query Logging**: Middleware untuk monitoring query lambat

### ✅ **2. Caching System**
- **View Caching**: Cache view yang sudah dikompilasi
- **Config Caching**: Cache konfigurasi aplikasi
- **Route Caching**: Cache routing untuk performa lebih baik
- **Navigation Data Caching**: Cache data dropdown untuk navigasi

### ✅ **3. Middleware Optimizations**
- **OptimizeQueries**: Monitoring dan logging query lambat
- **Security Headers**: Menambahkan header keamanan

### ✅ **4. Service Provider**
- **OptimizationServiceProvider**: Mengelola cache dan optimasi database

### ✅ **5. Artisan Commands**
- **app:optimize**: Command untuk optimasi production

## 🛠️ Cara Menggunakan Optimasi

### **1. Jalankan Optimasi untuk Production**
```bash
# Jalankan command optimasi
php artisan app:optimize --force

# Atau step by step
php artisan config:cache
php artisan route:cache
php artisan view:cache
php artisan optimize
```

### **2. Setup Environment Production**
```bash
# Copy file environment
cp .env.example .env

# Generate app key
php artisan key:generate

# Set environment ke production
# Edit .env file:
APP_ENV=production
APP_DEBUG=false
```

### **3. Database Optimizations**
```bash
# Migrate database
php artisan migrate --force

# Seed data
php artisan db:seed --force

# Clear cache
php artisan cache:clear
```

## 📊 Monitoring Performa

### **1. Query Monitoring**
- Query lambat akan di-log di `storage/logs/laravel.log`
- Threshold: 2000ms (dapat diubah di config)

### **2. Cache Monitoring**
- Gunakan `php artisan cache:clear` untuk clear cache
- Monitor cache hit rate di dashboard

### **3. Memory Usage**
- Monitor memory usage dengan `memory_get_usage()`
- Optimasi query untuk mengurangi memory usage

## 🔧 Konfigurasi Server

### **1. Apache (.htaccess)**
File `.htaccess` sudah dibuat dengan optimasi:
- Gzip compression
- Browser caching
- Security headers

### **2. Nginx**
```nginx
# Gzip compression
gzip on;
gzip_vary on;
gzip_min_length 1024;
gzip_types text/plain text/css text/xml text/javascript application/javascript application/xml+rss application/json;

# Browser caching
location ~* \.(css|js|png|jpg|jpeg|gif|ico|svg)$ {
    expires 1y;
    add_header Cache-Control "public, immutable";
}

# Security headers
add_header X-Frame-Options "SAMEORIGIN" always;
add_header X-XSS-Protection "1; mode=block" always;
add_header X-Content-Type-Options "nosniff" always;
```

### **3. PHP Configuration**
```ini
; php.ini optimizations
memory_limit = 256M
max_execution_time = 30
max_input_vars = 3000
post_max_size = 64M
upload_max_filesize = 64M

; OPcache
opcache.enable=1
opcache.memory_consumption=128
opcache.interned_strings_buffer=8
opcache.max_accelerated_files=4000
opcache.revalidate_freq=2
opcache.fast_shutdown=1
```

## 📈 Tips Optimasi Tambahan

### **1. Database**
- Gunakan index pada kolom yang sering di-query
- Hindari SELECT * (gunakan kolom spesifik)
- Gunakan pagination untuk data besar

### **2. Views**
- Gunakan `@include` untuk komponen yang sering digunakan
- Cache partial view yang kompleks
- Minimize CSS/JS

### **3. Assets**
- Minify CSS dan JavaScript
- Combine multiple files menjadi satu
- Use CDN untuk static assets

### **4. Caching**
- Cache expensive operations
- Use Redis untuk session dan cache
- Implement query result caching

## 🚨 Troubleshooting

### **1. Cache Issues**
```bash
# Clear all caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

### **2. Performance Issues**
```bash
# Check slow queries
tail -f storage/logs/laravel.log | grep "Slow queries"

# Monitor memory usage
php artisan tinker
memory_get_usage(true)
```

### **3. Database Issues**
```bash
# Check database connection
php artisan migrate:status

# Optimize database
php artisan db:show
```

## 📝 Checklist Deploy ke Production

- [ ] Set `APP_ENV=production`
- [ ] Set `APP_DEBUG=false`
- [ ] Generate `APP_KEY`
- [ ] Run `php artisan app:optimize --force`
- [ ] Setup database dengan migration dan seeder
- [ ] Configure web server (Apache/Nginx)
- [ ] Setup SSL certificate
- [ ] Configure backup strategy
- [ ] Monitor logs dan performance

## 🎯 Expected Performance

Setelah optimasi, aplikasi seharusnya:
- ⚡ Loading time < 2 detik
- 💾 Memory usage < 128MB
- 🗄️ Database queries < 50 per request
- 📱 Responsive di semua device

## 📞 Support

Jika ada masalah dengan optimasi, check:
1. Log files di `storage/logs/`
2. Error logs di web server
3. Database connection
4. File permissions
