# Implementasi NIA Per DPC

## Overview
Sistem NIA (Nomor Identitas Anggota) telah dimodifikasi untuk menggunakan counter per DPC, bukan counter global. Setiap DPC memiliki counter NIA yang terpisah.

## Format NIA Baru
```
PKS-{KODE_DPC}-{COUNTER}-{TAHUN}
```

Contoh:
- DPC Jakarta Timur (kode: JKT): `PKS-JKT-001-2025`
- DPC Bandung (kode: BDG): `PKS-BDG-001-2025`

## Perubahan yang Dilakukan

### 1. Model Anggota (`app/Models/Anggota.php`)

#### Method `generateNIA()` - Dimodifikasi
```php
public static function generateNIA($dpcId = null)
{
    $tahun = date('Y');
    
    if ($dpcId) {
        // Generate NIA per DPC
        $counter = static::where('dpc_id', $dpcId)
                        ->whereYear('created_at', $tahun)
                        ->count() + 1;
        $paddedCounter = str_pad($counter, 3, '0', STR_PAD_LEFT);
        
        // Get DPC code for NIA format
        $dpc = \App\Models\Dpc::find($dpcId);
        $dpcCode = $dpc ? $dpc->kode_dpc : 'DPC';
        
        return "PKS-{$dpcCode}-{$paddedCounter}-{$tahun}";
    } else {
        // Fallback: Generate NIA global (untuk backward compatibility)
        $counter = static::whereYear('created_at', $tahun)->count() + 1;
        $paddedCounter = str_pad($counter, 3, '0', STR_PAD_LEFT);
        
        return "PKS-{$paddedCounter}-{$tahun}";
    }
}
```

#### Method Baru
- `generateNIAForDpc($dpcId)` - Generate NIA untuk DPC tertentu
- `getNextNIAForDpc($dpcId)` - Get next NIA number untuk preview

#### Boot Method - Dimodifikasi
```php
static::creating(function ($anggota) {
    if (empty($anggota->nomor_identitas_anggota)) {
        // Generate NIA berdasarkan DPC jika ada
        $dpcId = $anggota->dpc_id ?? null;
        $anggota->nomor_identitas_anggota = static::generateNIA($dpcId);
    }
});
```

### 2. Controller Anggota (`app/Http/Controllers/AnggotaController.php`)

#### Method Baru: `getNextNIA()`
```php
public function getNextNIA(Request $request)
{
    $dpcId = $request->get('dpc_id');
    
    if (!$dpcId) {
        return response()->json(['error' => 'DPC ID required']);
    }
    
    try {
        $nextNIA = \App\Models\Anggota::getNextNIAForDpc($dpcId);
        
        return response()->json([
            'success' => true,
            'next_nia' => $nextNIA
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'error' => 'Error generating NIA: ' . $e->getMessage()
        ]);
    }
}
```

### 3. Routes (`routes/web.php`)

#### Route Baru
```php
Route::get('/api/anggota/next-nia', [AnggotaController::class, 'getNextNIA'])->name('api.anggota.next-nia');
```

### 4. Form Create Anggota (`resources/views/anggota/create.blade.php`)

#### UI Changes
- Input NIA dengan tombol "Generate"
- Preview NIA yang otomatis update saat DPC dipilih
- JavaScript untuk handle preview dan generate NIA

#### JavaScript Functions
- `updateNIAPreview(dpcId)` - Update preview NIA
- `generateNIA()` - Generate dan isi NIA ke form
- `loadUpaByDpc()` - Dimodifikasi untuk include NIA preview

## Cara Kerja

### 1. Saat Membuat Anggota Baru
1. User pilih DPD → DPC dropdown terisi
2. User pilih DPC → Preview NIA muncul otomatis
3. User klik "Generate" → NIA terisi otomatis
4. User submit form → NIA tersimpan dengan format per DPC

### 2. Counter Logic
- Counter dimulai dari 1 untuk setiap DPC
- Counter reset setiap tahun
- Format: `PKS-{KODE_DPC}-{COUNTER}-{TAHUN}`

### 3. Backward Compatibility
- Jika DPC tidak dipilih, sistem fallback ke format lama
- Data lama tetap valid dan tidak terpengaruh

## Testing

### Test Cases
1. **Create anggota dengan DPC** - NIA harus menggunakan format per DPC
2. **Create anggota tanpa DPC** - NIA menggunakan format global (fallback)
3. **Multiple anggota per DPC** - Counter harus increment per DPC
4. **Different DPCs** - Counter terpisah per DPC
5. **Year change** - Counter reset per tahun

### Manual Testing Steps
1. Buka form create anggota
2. Pilih DPD → DPC
3. Lihat preview NIA muncul
4. Klik "Generate" → NIA terisi
5. Submit form
6. Cek di database bahwa NIA tersimpan dengan format yang benar

## Database Requirements

### Tabel `tbl_dpc` harus memiliki:
- `kode_dpc` (string, unique) - Kode DPC untuk NIA
- `nama_dpc` (string) - Nama DPC
- `is_active` (boolean) - Status aktif

### Tabel `tbl_anggota` harus memiliki:
- `dpc_id` (foreign key) - Relasi ke DPC
- `nomor_identitas_anggota` (string) - NIA dengan format baru

## API Endpoints

### GET `/api/anggota/next-nia?dpc_id={dpcId}`
**Response:**
```json
{
    "success": true,
    "next_nia": "PKS-JKT-001-2025"
}
```

**Error Response:**
```json
{
    "success": false,
    "error": "DPC ID required"
}
```

## Benefits

1. **Organized NIA** - NIA terorganisir per DPC
2. **Easy Identification** - Mudah identifikasi asal DPC dari NIA
3. **Scalable** - Setiap DPC punya counter terpisah
4. **Backward Compatible** - Tidak merusak data lama
5. **User Friendly** - Preview dan auto-generate NIA

## Migration Notes

- Tidak perlu migration database
- Data lama tetap valid
- Format baru hanya berlaku untuk anggota baru
- Sistem otomatis detect dan generate NIA sesuai DPC
