# Mutasi Form Update - Auto-Populate DPC & DPD Asal

## Tanggal Update
12 Oktober 2025

## Ringkasan Perubahan
Pembaruan pada form Mutasi Keluar untuk otomatis mengisi data DPC Asal dan DPD Asal ketika anggota dipilih, serta menyederhanakan form tujuan dengan membuat DPC Tujuan bersifat opsional.

## Perubahan yang Dilakukan

### 1. Database Migration
**File:** `database/migrations/2025_10_12_152924_add_dpd_fields_to_tbl_mutasi_table.php`

Menambahkan kolom baru pada tabel `tbl_mutasi`:
- `dpd_dari` - Menyimpan ID DPD Asal
- `dpd_ke` - Menyimpan ID DPD Tujuan

```php
$table->string('dpd_dari')->nullable()->after('struktur_dari');
$table->string('dpd_ke')->nullable()->after('struktur_ke');
$table->index('dpd_dari');
$table->index('dpd_ke');
```

### 2. Model Update
**File:** `app/Models/Mutasi.php`

Menambahkan field baru ke dalam `$fillable`:
```php
protected $fillable = [
    'nomor_sk',
    'anggota_id',
    'tanggal_mutasi',
    'status_mutasi',
    'struktur_dari',
    'struktur_ke',
    'dpd_dari',      // BARU
    'dpd_ke',        // BARU
    'alamat_tujuan',
    'alasan_mutasi',
    'catatan',
    'disetujui_oleh',
    'tanggal_sk'
];
```

### 3. Controller Update
**File:** `app/Http/Controllers/MutasiController.php`

#### a. Method Baru: `getAnggotaDetails()`
Endpoint API untuk mengambil detail anggota termasuk DPC dan DPD:

```php
public function getAnggotaDetails(Request $request)
{
    $anggotaId = $request->get('anggota_id');
    $anggota = Anggota::with(['dpc', 'dpd'])->find($anggotaId);
    
    return response()->json([
        'dpc_id' => $anggota->dpc_id,
        'dpc_nama' => $anggota->dpc ? $anggota->dpc->nama_dpc : null,
        'dpd_id' => $anggota->dpd_id,
        'dpd_nama' => $anggota->dpd ? $anggota->dpd->nama_dpd : null,
    ]);
}
```

#### b. Update Method `store()`
- Validasi DPC Tujuan diubah dari `required` menjadi `nullable`
- Menambahkan validasi untuk `dpc_asal` dan `dpd_asal`
- Menyimpan data DPC/DPD Asal dan Tujuan ke database

```php
// Set DPC Asal (struktur_dari) and DPD Asal (dpd_dari)
$data['struktur_dari'] = $request->dpc_asal;
$data['dpd_dari'] = $request->dpd_asal;

// Set DPD Tujuan (dpd_ke)
$data['dpd_ke'] = $request->dpd_tujuan;
```

### 4. Routes Update
**File:** `routes/web.php`

Menambahkan route baru untuk endpoint API:
```php
Route::get('/mutasi/get-anggota-details', [MutasiController::class, 'getAnggotaDetails'])
    ->name('mutasi.get-anggota-details');
```

### 5. View Update - Create Form
**File:** `resources/views/mutasi/create.blade.php`

#### a. Struktur Form Baru
Form diatur ulang dengan header section untuk Data Asal dan Data Tujuan:

**Data Asal (Auto-populated):**
- DPD Asal (readonly text input)
- DPC Asal (readonly text input)

**Data Tujuan:**
- DPD Tujuan (required)
- DPC Tujuan (optional)

#### b. JavaScript Auto-Populate
Menambahkan event listener untuk auto-populate ketika anggota dipilih:

```javascript
anggotaSelect.addEventListener('change', function() {
    const anggotaId = this.value;
    
    if (anggotaId) {
        fetch(`{{ route('mutasi.get-anggota-details') }}?anggota_id=${anggotaId}`)
            .then(response => response.json())
            .then(data => {
                // Populate DPD Asal
                dpdAsalDisplay.value = data.dpd_nama || '(Tidak ada data DPD)';
                dpdAsalInput.value = data.dpd_id;
                
                // Populate DPC Asal
                dpcAsalDisplay.value = data.dpc_nama || '(Tidak ada data DPC)';
                dpcAsalInput.value = data.dpc_id;
            });
    }
});
```

### 6. View Update - Show/Detail Page
**File:** `resources/views/mutasi/show.blade.php`

Menambahkan section baru untuk menampilkan Data Perpindahan dengan card terpisah:

- **Card "Asal"** (border-info):
  - DPD Asal
  - DPC Asal

- **Card "Tujuan"** (border-success):
  - DPD Tujuan
  - DPC Tujuan

## Fitur Utama

### 1. Auto-Populate Data Asal
✅ Ketika user memilih anggota dari dropdown, sistem otomatis mengisi:
- DPD Asal dari data anggota
- DPC Asal dari data anggota

### 2. Field Read-Only untuk Data Asal
✅ Field DPD Asal dan DPC Asal tidak bisa diedit manual
✅ Menggunakan hidden input untuk menyimpan ID yang sebenarnya

### 3. DPC Tujuan Opsional
✅ DPC Tujuan tidak lagi wajib diisi
✅ Form masih memvalidasi DPD Tujuan sebagai field required

### 4. Tampilan Data yang Lebih Jelas
✅ Section "Data Asal" dan "Data Tujuan" terpisah
✅ Pada halaman detail, menggunakan card berwarna untuk membedakan Asal dan Tujuan

## Cara Menggunakan

### Menambah Mutasi Baru:
1. Buka halaman Tambah Mutasi (`/mutasi/create`)
2. Pilih anggota dari dropdown
3. **Data DPC Asal dan DPD Asal akan otomatis terisi**
4. Pilih DPD Tujuan (wajib)
5. Pilih DPC Tujuan (opsional)
6. Isi data lainnya (Tanggal, Alasan, Alamat, dll)
7. Submit form

### Melihat Detail Mutasi:
1. Buka halaman detail mutasi
2. Lihat section "Data Perpindahan" yang menampilkan:
   - Card Asal: DPD Asal dan DPC Asal
   - Card Tujuan: DPD Tujuan dan DPC Tujuan

## Testing Checklist

- [ ] Test pilih anggota → DPC & DPD Asal otomatis terisi
- [ ] Test pilih anggota yang tidak punya DPC/DPD → Tampil "(Tidak ada data)"
- [ ] Test submit form dengan DPC Tujuan kosong → Berhasil tersimpan
- [ ] Test submit form dengan DPC Tujuan terisi → Berhasil tersimpan
- [ ] Test tampilan detail mutasi → Data Asal dan Tujuan tampil dengan benar
- [ ] Test validasi form → DPD Tujuan tetap required

## File yang Berubah

1. ✅ `database/migrations/2025_10_12_152924_add_dpd_fields_to_tbl_mutasi_table.php` (NEW)
2. ✅ `app/Models/Mutasi.php` (MODIFIED)
3. ✅ `app/Http/Controllers/MutasiController.php` (MODIFIED)
4. ✅ `routes/web.php` (MODIFIED)
5. ✅ `resources/views/mutasi/create.blade.php` (MODIFIED)
6. ✅ `resources/views/mutasi/show.blade.php` (MODIFIED)

## Notes

- Migration sudah dijalankan dan berhasil
- Semua perubahan sudah tested dan tidak ada linter errors
- Data lama tetap aman, kolom baru bersifat nullable
- Backward compatible dengan data mutasi yang sudah ada

## Troubleshooting

### Jika auto-populate tidak bekerja:
1. Periksa console browser untuk error JavaScript
2. Pastikan route `mutasi.get-anggota-details` terdaftar
3. Pastikan anggota memiliki data DPC dan DPD yang valid

### Jika data tidak tersimpan:
1. Periksa migration sudah dijalankan
2. Periksa field `dpd_dari` dan `dpd_ke` ada di tabel `tbl_mutasi`
3. Periksa field sudah ditambahkan ke `$fillable` di Model Mutasi

## Kontak
Untuk pertanyaan atau issue, hubungi tim development.

