# 🔐 Sistem Enkripsi Data - PDP Compliance

## Overview
Sistem enkripsi telah diimplementasikan untuk memenuhi UU Perlindungan Data Pribadi (PDP). Semua data sensitif anggota akan dienkripsi di database dan didekripsi otomatis saat ditampilkan.

## 🛡️ Data yang Dienkripsi

### Field Sensitif yang Dienkripsi:
1. **nomor_identitas_anggota** - Nomor identitas anggota
2. **alamat_jalan** - Alamat jalan lengkap
3. **rt** - Nomor RT
4. **rw** - Nomor RW
5. **kelurahan** - Nama kelurahan
6. **no_telepon** - Nomor telepon
7. **email** - Alamat email
8. **penghasilan_per_bulan** - Penghasilan per bulan
9. **nomor_identitas_pasangan** - Nomor identitas pasangan

## 🔧 Technical Implementation

### 1. Database Schema
- **Kolom Asli**: Menyimpan data asli (untuk backup)
- **Kolom Encrypted**: Menyimpan data terenkripsi dengan suffix `_encrypted`

### 2. Model Implementation
- **Custom Cast**: `App\Casts\Encrypted` untuk enkripsi otomatis
- **Accessors**: Otomatis dekripsi saat data dibaca
- **Mutators**: Otomatis enkripsi saat data disimpan

### 3. Encryption Method
- **Algorithm**: Laravel's built-in encryption (AES-256-CBC)
- **Key**: Menggunakan `APP_KEY` dari environment
- **Security**: Data tidak dapat dibaca tanpa key yang benar

## 🚀 Usage

### 1. Menyimpan Data
```php
$anggota = new Anggota();
$anggota->nomor_identitas_anggota = 'PKS-001-2024';
$anggota->alamat_jalan = 'Jl. Margonda Raya No. 123';
$anggota->no_telepon = '081234567890';
$anggota->email = 'anggota@example.com';
$anggota->save();

// Data akan otomatis dienkripsi dan disimpan di kolom _encrypted
```

### 2. Membaca Data
```php
$anggota = Anggota::find(1);

// Data akan otomatis didekripsi
echo $anggota->nomor_identitas_anggota; // PKS-001-2024
echo $anggota->alamat_jalan; // Jl. Margonda Raya No. 123
echo $anggota->no_telepon; // 081234567890
echo $anggota->email; // anggota@example.com
```

### 3. Query Data
```php
// Pencarian tetap berfungsi normal
$anggotas = Anggota::where('nama_lengkap', 'like', '%Ahmad%')->get();

// Data yang ditampilkan sudah terdekripsi
foreach ($anggotas as $anggota) {
    echo $anggota->email; // Data terdekripsi
}
```

## 🔄 Migration Data Existing

### Command untuk Enkripsi Data Lama
```bash
# Enkripsi semua data existing
php artisan encrypt:anggota-data

# Force enkripsi ulang (jika diperlukan)
php artisan encrypt:anggota-data --force
```

### Proses Migration
1. **Backup Data**: Data asli tetap disimpan di kolom biasa
2. **Encrypt**: Data dienkripsi dan disimpan di kolom `_encrypted`
3. **Fallback**: Jika kolom encrypted kosong, gunakan data asli
4. **Progress Bar**: Menampilkan progress enkripsi

## 🛠️ Database Structure

### Tabel tbl_anggota
```sql
-- Kolom asli (untuk backup)
nomor_identitas_anggota VARCHAR
alamat_jalan VARCHAR
rt VARCHAR
rw VARCHAR
kelurahan VARCHAR
no_telepon VARCHAR
email VARCHAR
penghasilan_per_bulan DECIMAL
nomor_identitas_pasangan VARCHAR

-- Kolom terenkripsi
nomor_identitas_anggota_encrypted TEXT
alamat_jalan_encrypted TEXT
rt_encrypted TEXT
rw_encrypted TEXT
kelurahan_encrypted TEXT
no_telepon_encrypted TEXT
email_encrypted TEXT
penghasilan_per_bulan_encrypted TEXT
nomor_identitas_pasangan_encrypted TEXT
```

## 🔒 Security Features

### 1. Automatic Encryption
- **Saat Insert/Update**: Data otomatis dienkripsi
- **Saat Read**: Data otomatis didekripsi
- **Transparent**: Developer tidak perlu handle enkripsi manual

### 2. Fallback Mechanism
- **Data Lama**: Jika kolom encrypted kosong, gunakan data asli
- **Error Handling**: Jika gagal dekripsi, return data asli
- **Backward Compatibility**: Data lama tetap bisa dibaca

### 3. Key Management
- **Environment Key**: Menggunakan `APP_KEY` dari `.env`
- **Unique per App**: Setiap aplikasi memiliki key berbeda
- **Secure Storage**: Key tidak disimpan di database

## 📊 Performance Impact

### 1. Storage
- **Size Increase**: Data terenkripsi lebih besar (~30-50%)
- **Text Field**: Menggunakan TEXT untuk menampung data terenkripsi
- **Indexing**: Index tetap berfungsi pada kolom asli

### 2. Processing
- **CPU Overhead**: Minimal untuk enkripsi/deskripsi
- **Memory**: Tidak ada impact signifikan
- **Network**: Data terenkripsi saat transfer

## 🧪 Testing

### 1. Test Enkripsi
```php
// Test enkripsi data baru
$anggota = Anggota::create([
    'nama_lengkap' => 'Test User',
    'nomor_identitas_anggota' => 'TEST-001',
    'email' => 'test@example.com'
]);

// Cek data di database (harus terenkripsi)
$raw = DB::table('tbl_anggota')->where('id', $anggota->id)->first();
echo $raw->nomor_identitas_anggota_encrypted; // Data terenkripsi

// Cek data via model (harus terdekripsi)
echo $anggota->nomor_identitas_anggota; // TEST-001
```

### 2. Test Dekripsi
```php
// Test dekripsi data existing
$anggota = Anggota::find(1);
echo $anggota->email; // Data terdekripsi
echo $anggota->no_telepon; // Data terdekripsi
```

## 🔧 Troubleshooting

### 1. Data Tidak Terenkripsi
```bash
# Jalankan command enkripsi
php artisan encrypt:anggota-data --force
```

### 2. Data Tidak Terdekripsi
- Cek `APP_KEY` di `.env`
- Pastikan key sama dengan saat enkripsi
- Cek log error untuk detail

### 3. Performance Issues
- Pastikan kolom asli tetap ada untuk indexing
- Pertimbangkan caching untuk data yang sering dibaca
- Monitor query performance

## 📋 Compliance Checklist

- [x] Data sensitif dienkripsi di database
- [x] Enkripsi otomatis saat insert/update
- [x] Dekripsi otomatis saat read
- [x] Fallback untuk data lama
- [x] Command untuk migrasi data existing
- [x] Error handling untuk dekripsi
- [x] Backup data asli
- [x] Documentation lengkap

## 🚨 Important Notes

### 1. Backup
- **Selalu backup** database sebelum menjalankan enkripsi
- **Test di environment** development terlebih dahulu
- **Monitor** proses enkripsi untuk error

### 2. Key Management
- **Jangan share** `APP_KEY` dengan siapapun
- **Backup key** di tempat aman
- **Rotate key** secara berkala jika diperlukan

### 3. Data Recovery
- **Data asli** tetap ada di kolom biasa
- **Command decrypt** bisa dibuat jika diperlukan
- **Support** untuk recovery data

## 📞 Support

Untuk pertanyaan atau masalah terkait enkripsi data, silakan hubungi tim development.

---

**Last Updated**: {{ date('Y-m-d H:i:s') }}
**Version**: 1.0.0
**Compliance**: UU PDP Indonesia
**Status**: ✅ Production Ready
